#!/bin/bash

# ============================================================================
# Risk Assessment System - Final Verification Script
# Checks all components are in place and working
# ============================================================================

echo "=============================================="
echo "  Risk Assessment System - Final Verification"
echo "=============================================="
echo ""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

ERRORS=0
WARNINGS=0
SUCCESS=0

echo "🔍 Checking System Components..."
echo ""

# ============================================================================
# CHECK 1: CORE FILES
# ============================================================================
echo "1. Core Files"
echo "-------------"

FILES=(
    "includes/config.php"
    "includes/functions.php"
    "includes/auth.php"
    "schema.sql"
)

for file in "${FILES[@]}"; do
    if [ -f "$file" ]; then
        echo -e "${GREEN}✓${NC} $file"
        SUCCESS=$((SUCCESS + 1))
    else
        echo -e "${RED}✗${NC} $file - MISSING"
        ERRORS=$((ERRORS + 1))
    fi
done

echo ""

# ============================================================================
# CHECK 2: TEMPLATES
# ============================================================================
echo "2. Template Files"
echo "-----------------"

TEMPLATES=(
    "templates/header.php"
    "templates/footer.php"
    "templates/header-responsive.php"
    "templates/footer-responsive.php"
)

for file in "${TEMPLATES[@]}"; do
    if [ -f "$file" ]; then
        echo -e "${GREEN}✓${NC} $file"
        SUCCESS=$((SUCCESS + 1))
        
        # Check for function_exists
        if [ "$file" = "templates/header.php" ] || [ "$file" = "templates/header-responsive.php" ]; then
            if grep -q "function_exists" "$file"; then
                echo -e "  ${GREEN}✓${NC} Has function_exists() checks"
            else
                echo -e "  ${YELLOW}⚠${NC} Missing function_exists() checks"
                WARNINGS=$((WARNINGS + 1))
            fi
        fi
    else
        echo -e "${RED}✗${NC} $file - MISSING"
        ERRORS=$((ERRORS + 1))
    fi
done

echo ""

# ============================================================================
# CHECK 3: CSS FILES
# ============================================================================
echo "3. CSS Files"
echo "------------"

CSS_FILES=(
    "public/css/responsive-framework.css"
    "public/css/style.css"
)

for file in "${CSS_FILES[@]}"; do
    if [ -f "$file" ]; then
        size=$(du -h "$file" | cut -f1)
        echo -e "${GREEN}✓${NC} $file ($size)"
        SUCCESS=$((SUCCESS + 1))
    else
        echo -e "${RED}✗${NC} $file - MISSING"
        ERRORS=$((ERRORS + 1))
    fi
done

# Check header includes CSS
if grep -q "responsive-framework.css" templates/header.php 2>/dev/null; then
    echo -e "${GREEN}✓${NC} header.php includes responsive-framework.css"
    SUCCESS=$((SUCCESS + 1))
else
    echo -e "${RED}✗${NC} header.php doesn't include responsive CSS"
    ERRORS=$((ERRORS + 1))
fi

echo ""

# ============================================================================
# CHECK 4: JAVASCRIPT FILES
# ============================================================================
echo "4. JavaScript Files"
echo "-------------------"

JS_FILES=(
    "public/js/responsive.js"
)

for file in "${JS_FILES[@]}"; do
    if [ -f "$file" ]; then
        size=$(du -h "$file" | cut -f1)
        echo -e "${GREEN}✓${NC} $file ($size)"
        SUCCESS=$((SUCCESS + 1))
    else
        echo -e "${RED}✗${NC} $file - MISSING"
        ERRORS=$((ERRORS + 1))
    fi
done

echo ""

# ============================================================================
# CHECK 5: DIRECTORIES
# ============================================================================
echo "5. Required Directories"
echo "-----------------------"

DIRS=(
    "public"
    "includes"
    "templates"
    "logs"
    "uploads"
    "public/css"
    "public/js"
    "docs"
)

for dir in "${DIRS[@]}"; do
    if [ -d "$dir" ]; then
        if [ -w "$dir" ]; then
            echo -e "${GREEN}✓${NC} $dir (writable)"
            SUCCESS=$((SUCCESS + 1))
        else
            echo -e "${YELLOW}⚠${NC} $dir (not writable)"
            WARNINGS=$((WARNINGS + 1))
        fi
    else
        echo -e "${RED}✗${NC} $dir - MISSING"
        ERRORS=$((ERRORS + 1))
    fi
done

echo ""

# ============================================================================
# CHECK 6: KEY PHP FILES
# ============================================================================
echo "6. Key Application Files"
echo "------------------------"

PHP_FILES=(
    "public/login.php"
    "public/dashboard.php"
    "public/admin-panel.php"
    "public/assessments.php"
    "public/new-assessment.php"
    "public/locations.php"
    "public/users.php"
    "public/custom-fields.php"
    "public/system-customization.php"
    "public/version-control.php"
)

for file in "${PHP_FILES[@]}"; do
    if [ -f "$file" ]; then
        # Check PHP syntax
        php -l "$file" >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            echo -e "${GREEN}✓${NC} $file"
            SUCCESS=$((SUCCESS + 1))
        else
            echo -e "${RED}✗${NC} $file - SYNTAX ERROR"
            ERRORS=$((ERRORS + 1))
        fi
    else
        echo -e "${RED}✗${NC} $file - MISSING"
        ERRORS=$((ERRORS + 1))
    fi
done

echo ""

# ============================================================================
# CHECK 7: DOCUMENTATION
# ============================================================================
echo "7. Documentation Files"
echo "----------------------"

DOCS=(
    "README.md"
    "QUICK_START.md"
    "DATABASE_SCHEMA.md"
    "SCHEMA_CLEANUP.md"
    "HEADER_CSS_AUDIT.md"
)

for file in "${DOCS[@]}"; do
    if [ -f "$file" ]; then
        echo -e "${GREEN}✓${NC} $file"
        SUCCESS=$((SUCCESS + 1))
    else
        echo -e "${YELLOW}⚠${NC} $file - missing (optional)"
        WARNINGS=$((WARNINGS + 1))
    fi
done

echo ""

# ============================================================================
# CHECK 8: SCHEMA FILE
# ============================================================================
echo "8. Database Schema"
echo "------------------"

if [ -f "schema.sql" ]; then
    lines=$(wc -l < schema.sql)
    echo -e "${GREEN}✓${NC} schema.sql exists ($lines lines)"
    SUCCESS=$((SUCCESS + 1))
    
    # Check for key tables
    TABLES=(
        "CREATE TABLE.*users"
        "CREATE TABLE.*locations"
        "CREATE TABLE.*assessments"
        "CREATE TABLE.*custom_fields"
        "CREATE TABLE.*system_files"
        "CREATE TABLE.*permissions"
    )
    
    for table in "${TABLES[@]}"; do
        if grep -q "$table" schema.sql; then
            table_name=$(echo "$table" | sed 's/CREATE TABLE.*\(.*\)/\1/')
            echo -e "  ${GREEN}✓${NC} Contains $table_name table"
        else
            echo -e "  ${RED}✗${NC} Missing $table definition"
            ERRORS=$((ERRORS + 1))
        fi
    done
else
    echo -e "${RED}✗${NC} schema.sql - MISSING"
    ERRORS=$((ERRORS + 1))
fi

echo ""

# ============================================================================
# CHECK 9: NO OLD MIGRATION FILES
# ============================================================================
echo "9. Old Migration Files (should be removed)"
echo "-------------------------------------------"

OLD_MIGRATIONS=$(find includes -name "migration_*.sql" 2>/dev/null | wc -l)
OLD_FIXES=$(find includes -name "*_fix.sql" 2>/dev/null | wc -l)

if [ "$OLD_MIGRATIONS" -eq 0 ] && [ "$OLD_FIXES" -eq 0 ]; then
    echo -e "${GREEN}✓${NC} No old migration files (clean)"
    SUCCESS=$((SUCCESS + 1))
else
    echo -e "${YELLOW}⚠${NC} Found $OLD_MIGRATIONS migration files and $OLD_FIXES fix files"
    echo "  These should be removed (use schema.sql only)"
    WARNINGS=$((WARNINGS + 1))
fi

echo ""

# ============================================================================
# SUMMARY
# ============================================================================
echo "=============================================="
echo "  Verification Summary"
echo "=============================================="
echo ""

TOTAL=$((SUCCESS + WARNINGS + ERRORS))

echo -e "Total Checks:    ${BLUE}$TOTAL${NC}"
echo -e "Passed:          ${GREEN}$SUCCESS${NC}"
echo -e "Warnings:        ${YELLOW}$WARNINGS${NC}"
echo -e "Errors:          ${RED}$ERRORS${NC}"
echo ""

if [ $ERRORS -eq 0 ]; then
    if [ $WARNINGS -eq 0 ]; then
        echo -e "${GREEN}✅ PERFECT! All checks passed!${NC}"
        echo ""
        echo "Your system is ready to deploy! 🚀"
    else
        echo -e "${YELLOW}⚠ GOOD! System functional with minor warnings${NC}"
        echo ""
        echo "System is usable but has some optional improvements available."
    fi
else
    echo -e "${RED}✗ ERRORS FOUND! Please fix the issues above.${NC}"
    echo ""
    echo "System may not work correctly until errors are resolved."
fi

echo ""
echo "=============================================="
echo ""

exit $ERRORS
