<?php
require_once '../includes/config.php';
requireLogin();

if (!isset($_GET['id'])) {
    header('Location: locations.php');
    exit;
}

$stmt = $pdo->prepare("SELECT * FROM locations WHERE id = ?");
$stmt->execute([$_GET['id']]);
$location = $stmt->fetch();

if (!$location) {
    header('Location: locations.php');
    exit;
}

$opening_hours = json_decode($location['opening_hours'], true) ?: [];
$delivery_windows = json_decode($location['delivery_windows'], true) ?: [];
$custom_fields = json_decode($location['custom_fields'], true) ?: [];

// Get assessment count for this location
$stmt = $pdo->prepare("SELECT COUNT(*) FROM assessments WHERE location_id = ?");
$stmt->execute([$_GET['id']]);
$assessment_count = $stmt->fetchColumn();

// Fetch custom field definitions to get labels
$stmt = $pdo->query("SELECT * FROM custom_field_definitions WHERE is_active = 1 ORDER BY display_order");
$custom_field_defs = [];
foreach ($stmt->fetchAll() as $field) {
    $custom_field_defs[$field['field_name']] = $field;
}

// Get latest assessment
$stmt = $pdo->prepare("
    SELECT a.*, u.full_name as assessor_name 
    FROM assessments a 
    JOIN users u ON a.assessor_id = u.id 
    WHERE a.location_id = ? 
    ORDER BY a.assessment_date DESC 
    LIMIT 1
");
$stmt->execute([$_GET['id']]);
$latest_assessment = $stmt->fetch();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo sanitizeOutput($location['location_name']); ?> - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
    <?php if (file_exists(__DIR__ . '/css/custom-theme.css')): ?>
    <link rel="stylesheet" href="css/custom-theme.css">
    <?php endif; ?>
    
    <!-- Vue 3 -->
    <script src="https://unpkg.com/vue@3.4.15/dist/vue.global.prod.js"></script>
    <!-- Axios -->
    <script src="https://unpkg.com/axios@1.6.5/dist/axios.min.js"></script>
    <!-- Location Documents Component -->
    <script src="js/location-documents.js"></script>
    
    <style>
        #map {
            width: 100%;
            height: 450px;
            border-radius: 12px;
            border: 2px solid var(--border-color);
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
        }
        
        .info-box {
            background: var(--bg-tertiary);
            padding: 1.5rem;
            border-radius: 8px;
            border: 1px solid var(--border-color);
        }
        
        .info-label {
            font-size: 0.85rem;
            color: var(--text-muted);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-bottom: 0.5rem;
        }
        
        .info-value {
            font-size: 1.1rem;
            color: var(--text-primary);
            font-weight: 600;
        }
        
        .hours-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .hours-table td {
            padding: 0.75rem;
            border-bottom: 1px solid var(--border-color);
        }
        
        .hours-table td:first-child {
            font-weight: 600;
            color: var(--text-primary);
            width: 120px;
        }
        
        .hours-table td:last-child {
            color: var(--text-secondary);
        }
        
        .hours-table tr:last-child td {
            border-bottom: none;
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">📍 <?php echo sanitizeOutput($location['location_name']); ?></h1>
                <p class="page-subtitle">Delivery Location Details</p>
            </div>
            <div class="btn-group">
                <a href="edit-location.php?id=<?php echo $location['id']; ?>" class="btn btn-primary">✏️ Edit Location</a>
                <a href="new-assessment.php?location=<?php echo $location['id']; ?>" class="btn btn-success">➕ New Assessment</a>
                <a href="locations.php" class="btn btn-secondary">← Back to Locations</a>
            </div>
        </div>

        <!-- Quick Stats -->
        <div class="stats-grid" style="margin-bottom: 2rem;">
            <div class="stat-card primary">
                <div class="stat-value"><?php echo $assessment_count; ?></div>
                <div class="stat-label">Total Assessments</div>
            </div>
            <div class="stat-card <?php echo $location['latitude'] && $location['longitude'] ? 'success' : 'danger'; ?>">
                <div class="stat-value"><?php echo $location['latitude'] && $location['longitude'] ? '✓' : '✗'; ?></div>
                <div class="stat-label">GPS Coordinates</div>
            </div>
            <div class="stat-card <?php echo $location['what3words'] ? 'success' : 'danger'; ?>">
                <div class="stat-value"><?php echo $location['what3words'] ? '✓' : '✗'; ?></div>
                <div class="stat-label">What3Words</div>
            </div>
            <div class="stat-card <?php echo !empty($opening_hours) ? 'success' : 'danger'; ?>">
                <div class="stat-value"><?php echo !empty($opening_hours) ? '✓' : '✗'; ?></div>
                <div class="stat-label">Opening Hours Set</div>
            </div>
        </div>

        <!-- Address Information -->
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">📮 Address Information</h2>
            </div>
            
            <div class="info-grid">
                <div class="info-box">
                    <div class="info-label">Full Address</div>
                    <div class="info-value">
                        <?php echo sanitizeOutput($location['address_line1']); ?><br>
                        <?php if ($location['address_line2']): ?>
                            <?php echo sanitizeOutput($location['address_line2']); ?><br>
                        <?php endif; ?>
                        <?php echo sanitizeOutput($location['city']); ?>, <?php echo sanitizeOutput($location['postcode']); ?>
                        <?php if ($location['region']): ?><br><?php echo sanitizeOutput($location['region']); ?><?php endif; ?>
                    </div>
                </div>

                <?php if ($location['contact_name'] || $location['contact_phone'] || $location['contact_email']): ?>
                <div class="info-box">
                    <div class="info-label">Contact Details</div>
                    <div class="info-value">
                        <?php if ($location['contact_name']): ?>
                            👤 <?php echo sanitizeOutput($location['contact_name']); ?><br>
                        <?php endif; ?>
                        <?php if ($location['contact_phone']): ?>
                            📞 <a href="tel:<?php echo sanitizeOutput($location['contact_phone']); ?>" style="color: var(--primary);">
                                <?php echo sanitizeOutput($location['contact_phone']); ?>
                            </a><br>
                        <?php endif; ?>
                        <?php if ($location['contact_email']): ?>
                            📧 <a href="mailto:<?php echo sanitizeOutput($location['contact_email']); ?>" style="color: var(--primary);">
                                <?php echo sanitizeOutput($location['contact_email']); ?>
                            </a>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($location['report_emails'] || $location['email_enabled']): ?>
                <div class="info-box">
                    <div class="info-label">📧 Automated Report Emails</div>
                    <div class="info-value">
                        <?php if ($location['email_enabled']): ?>
                            <span style="display: inline-block; padding: 0.25rem 0.75rem; background: rgba(5, 150, 105, 0.1); 
                                         color: var(--success); border-radius: 12px; font-size: 0.85rem; margin-bottom: 0.5rem;">
                                ✓ Enabled
                            </span>
                        <?php else: ?>
                            <span style="display: inline-block; padding: 0.25rem 0.75rem; background: rgba(107, 114, 128, 0.1); 
                                         color: var(--text-muted); border-radius: 12px; font-size: 0.85rem; margin-bottom: 0.5rem;">
                                ○ Disabled
                            </span>
                        <?php endif; ?>
                        <br>
                        
                        <?php if ($location['report_emails']): ?>
                            <strong>Recipients:</strong><br>
                            <?php 
                            $emails = array_map('trim', explode(',', $location['report_emails']));
                            foreach ($emails as $email): 
                                if (!empty($email)):
                            ?>
                                <a href="mailto:<?php echo sanitizeOutput($email); ?>" style="color: var(--primary); display: block;">
                                    • <?php echo sanitizeOutput($email); ?>
                                </a>
                            <?php 
                                endif;
                            endforeach; 
                            ?>
                            <br>
                        <?php endif; ?>
                        
                        <strong>Frequency:</strong> 
                        <?php 
                        $frequencies = [
                            'none' => 'None - Manual Only',
                            'immediate' => 'Immediate (after each assessment)',
                            'daily' => 'Daily Digest',
                            'weekly' => 'Weekly Summary',
                            'monthly' => 'Monthly Report'
                        ];
                        echo $frequencies[$location['email_frequency'] ?? 'none'] ?? 'Not Set';
                        ?>
                        
                        <?php if ($location['last_report_sent']): ?>
                            <br><small style="color: var(--text-muted);">
                                Last sent: <?php echo date('d M Y H:i', strtotime($location['last_report_sent'])); ?>
                            </small>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($location['latitude'] && $location['longitude']): ?>
                <div class="info-box">
                    <div class="info-label">GPS Coordinates</div>
                    <div class="info-value" style="font-family: var(--font-mono); font-size: 1rem;">
                        📍 <?php echo number_format($location['latitude'], 6); ?>, <?php echo number_format($location['longitude'], 6); ?>
                        <br>
                        <a href="https://www.google.com/maps?q=<?php echo $location['latitude']; ?>,<?php echo $location['longitude']; ?>" 
                           target="_blank" 
                           class="btn btn-secondary" 
                           style="margin-top: 0.5rem; padding: 0.5rem 1rem; font-size: 0.85rem;">
                            🗺️ Open in Google Maps
                        </a>
                    </div>
                </div>
                <?php endif; ?>

                <?php if ($location['what3words']): ?>
                <div class="info-box">
                    <div class="info-label">What3Words</div>
                    <div class="info-value" style="font-family: var(--font-mono);">
                        <span style="color: var(--danger);">//</span><?php echo sanitizeOutput($location['what3words']); ?>
                        <br>
                        <a href="https://what3words.com/<?php echo urlencode($location['what3words']); ?>" 
                           target="_blank" 
                           class="btn btn-secondary" 
                           style="margin-top: 0.5rem; padding: 0.5rem 1rem; font-size: 0.85rem;">
                            🌐 View on What3Words
                        </a>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Google Maps -->
        <?php if ($location['latitude'] && $location['longitude']): ?>
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">🗺️ Location Map</h2>
            </div>
            
            <div id="map"></div>
            
            <div style="margin-top: 1rem; padding: 1rem; background: var(--bg-tertiary); border-radius: 8px;">
                <div style="display: flex; gap: 1rem; flex-wrap: wrap;">
                    <a href="https://www.google.com/maps/dir/?api=1&destination=<?php echo $location['latitude']; ?>,<?php echo $location['longitude']; ?>" 
                       target="_blank" class="btn btn-primary">
                        🧭 Get Directions
                    </a>
                    <?php if ($location['google_street_view_url']): ?>
                        <a href="<?php echo sanitizeOutput($location['google_street_view_url']); ?>" 
                           target="_blank" class="btn btn-secondary">
                            🎥 Street View
                        </a>
                    <?php else: ?>
                        <a href="https://www.google.com/maps/@?api=1&map_action=pano&viewpoint=<?php echo $location['latitude']; ?>,<?php echo $location['longitude']; ?>" 
                           target="_blank" class="btn btn-secondary">
                            🎥 Street View
                        </a>
                    <?php endif; ?>
                    <button onclick="navigator.clipboard.writeText('<?php echo $location['latitude']; ?>,<?php echo $location['longitude']; ?>')" 
                            class="btn btn-secondary">
                        📋 Copy Coordinates
                    </button>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Custom Fields -->
        <?php if (!empty($custom_fields)): ?>
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">📝 Additional Information</h2>
            </div>
            
            <div class="info-grid">
                <?php foreach ($custom_fields as $field_name => $value): 
                    if ($value === '' || $value === null) continue;
                    $field_def = $custom_field_defs[$field_name] ?? null;
                    $label = $field_def ? $field_def['field_label'] : ucwords(str_replace('_', ' ', $field_name));
                ?>
                    <div class="info-box">
                        <div class="info-label"><?php echo sanitizeOutput($label); ?></div>
                        <div class="info-value">
                            <?php 
                            if ($field_def && $field_def['field_type'] === 'checkbox') {
                                echo $value ? '✓ Yes' : '✗ No';
                            } elseif ($field_def && $field_def['field_type'] === 'url') {
                                echo '<a href="' . sanitizeOutput($value) . '" target="_blank" style="color: var(--primary);">' . sanitizeOutput($value) . '</a>';
                            } elseif ($field_def && $field_def['field_type'] === 'email') {
                                echo '<a href="mailto:' . sanitizeOutput($value) . '" style="color: var(--primary);">' . sanitizeOutput($value) . '</a>';
                            } elseif ($field_def && $field_def['field_type'] === 'tel') {
                                echo '<a href="tel:' . sanitizeOutput($value) . '" style="color: var(--primary);">' . sanitizeOutput($value) . '</a>';
                            } else {
                                echo nl2br(sanitizeOutput($value));
                            }
                            ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Opening Hours -->
        <?php if (!empty($opening_hours)): ?>
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">🕐 Opening Hours</h2>
            </div>
            
            <table class="hours-table">
                <?php 
                $days_full = [
                    'monday' => 'Monday',
                    'tuesday' => 'Tuesday',
                    'wednesday' => 'Wednesday',
                    'thursday' => 'Thursday',
                    'friday' => 'Friday',
                    'saturday' => 'Saturday',
                    'sunday' => 'Sunday'
                ];
                foreach ($days_full as $key => $day): 
                    $hours = $opening_hours[$key] ?? null;
                ?>
                    <tr>
                        <td><?php echo $day; ?></td>
                        <td>
                            <?php if ($hours && ($hours['open'] || $hours['close'])): ?>
                                <?php echo $hours['open'] ?: 'Not set'; ?> - <?php echo $hours['close'] ?: 'Not set'; ?>
                            <?php else: ?>
                                <span style="color: var(--text-muted);">Closed</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </table>
        </div>
        <?php endif; ?>

        <!-- Delivery Windows -->
        <?php if (!empty($delivery_windows)): ?>
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">🚚 Delivery Windows</h2>
            </div>
            
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1rem;">
                <?php foreach ($delivery_windows as $window): ?>
                    <div class="info-box">
                        <div class="info-label">
                            <?php echo !empty($window['days']) ? sanitizeOutput($window['days']) : 'All Days'; ?>
                        </div>
                        <div class="info-value">
                            ⏰ <?php echo sanitizeOutput($window['start'] . ' - ' . $window['end']); ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Additional Notes -->
        <?php if ($location['notes']): ?>
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">📝 Additional Notes</h2>
            </div>
            
            <p style="color: var(--text-secondary); line-height: 1.8;">
                <?php echo nl2br(sanitizeOutput($location['notes'])); ?>
            </p>
        </div>
        <?php endif; ?>

        <!-- Latest Assessment -->
        <?php if ($latest_assessment): ?>
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">📋 Latest Assessment</h2>
                <a href="assessments.php?location=<?php echo $location['id']; ?>" class="btn btn-secondary">View All</a>
            </div>
            
            <div class="info-grid">
                <div class="info-box">
                    <div class="info-label">Assessment Number</div>
                    <div class="info-value"><?php echo sanitizeOutput($latest_assessment['assessment_number']); ?></div>
                </div>
                <div class="info-box">
                    <div class="info-label">Assessment Date</div>
                    <div class="info-value"><?php echo date('d/m/Y', strtotime($latest_assessment['assessment_date'])); ?></div>
                </div>
                <div class="info-box">
                    <div class="info-label">Risk Level</div>
                    <div class="info-value">
                        <span class="badge badge-<?php echo $latest_assessment['overall_risk_level']; ?>">
                            <?php echo strtoupper($latest_assessment['overall_risk_level']); ?>
                        </span>
                    </div>
                </div>
                <div class="info-box">
                    <div class="info-label">Assessor</div>
                    <div class="info-value"><?php echo sanitizeOutput($latest_assessment['assessor_name']); ?></div>
                </div>
            </div>
            
            <div style="margin-top: 1rem;">
                <a href="view-assessment.php?id=<?php echo $latest_assessment['id']; ?>" class="btn btn-primary">
                    📋 View Full Assessment
                </a>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <?php if ($location['latitude'] && $location['longitude']): ?>
    <script>
        function initMap() {
            const location = { 
                lat: <?php echo $location['latitude']; ?>, 
                lng: <?php echo $location['longitude']; ?> 
            };
            
            const map = new google.maps.Map(document.getElementById('map'), {
                zoom: 17,
                center: location,
                mapTypeId: 'satellite', // Show satellite view for better delivery point visibility
                mapTypeControl: true,
                mapTypeControlOptions: {
                    style: google.maps.MapTypeControlStyle.HORIZONTAL_BAR,
                    position: google.maps.ControlPosition.TOP_RIGHT,
                    mapTypeIds: ['roadmap', 'satellite', 'hybrid']
                },
                streetViewControl: true,
                fullscreenControl: true
            });
            
            // Add marker
            const marker = new google.maps.Marker({
                position: location,
                map: map,
                title: '<?php echo addslashes($location['location_name']); ?>',
                animation: google.maps.Animation.DROP,
                icon: {
                    path: google.maps.SymbolPath.CIRCLE,
                    scale: 10,
                    fillColor: '#D97706',
                    fillOpacity: 1,
                    strokeColor: '#ffffff',
                    strokeWeight: 3
                }
            });
            
            // Add info window
            const infowindow = new google.maps.InfoWindow({
                content: `
                    <div style="color: #1E293B; padding: 0.5rem;">
                        <h3 style="margin: 0 0 0.5rem 0; color: #D97706;">
                            <?php echo addslashes($location['location_name']); ?>
                        </h3>
                        <p style="margin: 0; line-height: 1.6;">
                            <?php echo addslashes($location['address_line1']); ?><br>
                            <?php echo addslashes($location['city'] . ', ' . $location['postcode']); ?>
                        </p>
                        <?php if ($location['what3words']): ?>
                        <p style="margin: 0.5rem 0 0 0; font-family: monospace; color: #DC2626;">
                            //<?php echo addslashes($location['what3words']); ?>
                        </p>
                        <?php endif; ?>
                    </div>
                `
            });
            
            marker.addListener('click', () => {
                infowindow.open(map, marker);
            });
            
            // Open info window by default
            infowindow.open(map, marker);
        }
        
        window.addEventListener('load', initMap);
    </script>
    <script src="https://maps.googleapis.com/maps/api/js?key=YOUR_GOOGLE_MAPS_API_KEY" async defer></script>
    <?php endif; ?>

    <script src="js/app.js"></script>
    <script>
        // Copy to clipboard notification
        document.querySelectorAll('button[onclick*="clipboard"]').forEach(btn => {
            btn.addEventListener('click', () => {
                window.app.showAlert('Coordinates copied to clipboard!', 'success');
            });
        });
    </script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
