<?php
require_once '../includes/config.php';
requireLogin();

if (!isset($_GET['id'])) {
    header('Location: assessments.php');
    exit;
}

$stmt = $pdo->prepare("
    SELECT a.*, l.*, u.full_name as assessor_name, u.email as assessor_email
    FROM assessments a
    JOIN locations l ON a.location_id = l.id
    JOIN users u ON a.assessor_id = u.id
    WHERE a.id = ?
");
$stmt->execute([$_GET['id']]);
$assessment = $stmt->fetch();

if (!$assessment) {
    header('Location: assessments.php');
    exit;
}

// Fetch assessment details
$stmt = $pdo->prepare("
    SELECT ad.*, ri.item_name, ri.description, rc.category_name
    FROM assessment_details ad
    JOIN risk_items ri ON ad.risk_item_id = ri.id
    JOIN risk_categories rc ON ri.category_id = rc.id
    WHERE ad.assessment_id = ?
    ORDER BY rc.display_order, ri.display_order
");
$stmt->execute([$_GET['id']]);
$details = $stmt->fetchAll();

// Group by category
$grouped_details = [];
foreach ($details as $detail) {
    $grouped_details[$detail['category_name']][] = $detail;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Assessment <?php echo sanitizeOutput($assessment['assessment_number']); ?> - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">Assessment: <?php echo sanitizeOutput($assessment['assessment_number']); ?></h1>
                <p class="page-subtitle">
                    <?php echo sanitizeOutput($assessment['location_name']); ?>
                    <?php if (isset($assessment['version_count']) && $assessment['version_count'] > 1): ?>
                        <span class="badge" style="background: var(--primary); color: white; margin-left: 0.5rem;">
                            v<?php echo $assessment['version_count']; ?>
                        </span>
                    <?php endif; ?>
                </p>
            </div>
            <div class="btn-group">
                <button onclick="showPDFExport()" class="btn btn-primary">
                    📄 Generate PDF
                </button>
                <?php if (isset($assessment['version_count']) && $assessment['version_count'] > 1): ?>
                    <a href="version-history.php?id=<?php echo $assessment['id']; ?>" class="btn btn-secondary">
                        📚 Version History (<?php echo $assessment['version_count']; ?>)
                    </a>
                <?php endif; ?>
                <a href="edit-assessment.php?id=<?php echo $assessment['id']; ?>" class="btn btn-secondary">
                    ✏️ Edit
                </a>
                <a href="assessments.php" class="btn btn-secondary">← Back</a>
            </div>
        </div>

        <!-- Assessment Summary -->
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">Assessment Summary</h2>
            </div>

            <div class="info-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1.5rem;">
                <div class="info-box" style="background: var(--bg-tertiary); padding: 1.5rem; border-radius: 8px;">
                    <div class="info-label" style="font-size: 0.85rem; color: var(--text-muted); text-transform: uppercase; margin-bottom: 0.5rem;">Assessment Date</div>
                    <div class="info-value" style="font-size: 1.1rem; font-weight: 600;">
                        <?php echo date('d/m/Y', strtotime($assessment['assessment_date'])); ?>
                    </div>
                </div>
                
                <div class="info-box" style="background: var(--bg-tertiary); padding: 1.5rem; border-radius: 8px;">
                    <div class="info-label" style="font-size: 0.85rem; color: var(--text-muted); text-transform: uppercase; margin-bottom: 0.5rem;">Assessor</div>
                    <div class="info-value" style="font-size: 1.1rem; font-weight: 600;">
                        <?php echo sanitizeOutput($assessment['assessor_name']); ?>
                    </div>
                </div>
                
                <div class="info-box" style="background: var(--bg-tertiary); padding: 1.5rem; border-radius: 8px;">
                    <div class="info-label" style="font-size: 0.85rem; color: var(--text-muted); text-transform: uppercase; margin-bottom: 0.5rem;">Overall Risk Level</div>
                    <div class="info-value" style="font-size: 1.1rem; font-weight: 600;">
                        <span class="badge badge-<?php echo $assessment['overall_risk_level']; ?>" style="font-size: 1rem; padding: 0.5rem 1rem;">
                            <?php echo strtoupper($assessment['overall_risk_level']); ?>
                        </span>
                    </div>
                </div>
                
                <div class="info-box" style="background: var(--bg-tertiary); padding: 1.5rem; border-radius: 8px;">
                    <div class="info-label" style="font-size: 0.85rem; color: var(--text-muted); text-transform: uppercase; margin-bottom: 0.5rem;">Status</div>
                    <div class="info-value" style="font-size: 1.1rem; font-weight: 600;">
                        <span class="badge badge-<?php echo $assessment['status']; ?>" style="font-size: 1rem; padding: 0.5rem 1rem;">
                            <?php echo strtoupper($assessment['status']); ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Risk Assessment Details -->
        <?php if (!empty($grouped_details)): ?>
            <?php foreach ($grouped_details as $category => $items): ?>
                <div class="card mb-3">
                    <div class="card-header">
                        <h2 class="card-title">📋 <?php echo sanitizeOutput($category); ?></h2>
                    </div>

                    <?php foreach ($items as $item): ?>
                        <div style="padding: 1.5rem; border-bottom: 1px solid var(--border-color);">
                            <h3 style="color: var(--text-primary); margin-bottom: 1rem;">
                                <?php echo sanitizeOutput($item['item_name']); ?>
                            </h3>

                            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 1rem;">
                                <div>
                                    <strong>Likelihood:</strong> <?php echo $item['likelihood']; ?>/5
                                </div>
                                <div>
                                    <strong>Severity:</strong> <?php echo $item['severity']; ?>/5
                                </div>
                                <div>
                                    <strong>Risk Score:</strong> 
                                    <span class="badge badge-<?php echo $item['risk_score'] <= 4 ? 'low' : ($item['risk_score'] <= 9 ? 'medium' : ($item['risk_score'] <= 16 ? 'high' : 'critical')); ?>">
                                        <?php echo $item['risk_score']; ?>
                                    </span>
                                </div>
                            </div>

                            <?php if ($item['current_controls']): ?>
                                <div style="margin-bottom: 1rem;">
                                    <strong style="color: var(--text-primary);">Current Controls:</strong><br>
                                    <span style="color: var(--text-secondary);"><?php echo nl2br(sanitizeOutput($item['current_controls'])); ?></span>
                                </div>
                            <?php endif; ?>

                            <?php if ($item['additional_actions']): ?>
                                <div style="margin-bottom: 1rem;">
                                    <strong style="color: var(--text-primary);">Additional Actions Required:</strong><br>
                                    <span style="color: var(--text-secondary);"><?php echo nl2br(sanitizeOutput($item['additional_actions'])); ?></span>
                                </div>
                            <?php endif; ?>

                            <?php if ($item['action_owner'] || $item['target_date']): ?>
                                <div style="display: flex; gap: 2rem; color: var(--text-secondary); font-size: 0.9rem;">
                                    <?php if ($item['action_owner']): ?>
                                        <div><strong>Owner:</strong> <?php echo sanitizeOutput($item['action_owner']); ?></div>
                                    <?php endif; ?>
                                    <?php if ($item['target_date']): ?>
                                        <div><strong>Target Date:</strong> <?php echo date('d/m/Y', strtotime($item['target_date'])); ?></div>
                                    <?php endif; ?>
                                </div>
                            <?php endif; ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endforeach; ?>
        <?php else: ?>
            <div class="card mb-3">
                <div style="text-align: center; padding: 3rem; color: var(--text-muted);">
                    <p>No detailed risk items recorded for this assessment.</p>
                </div>
            </div>
        <?php endif; ?>

        <!-- Assessment Photos -->
        <?php
        $stmt = $pdo->prepare("
            SELECT p.*, u.full_name as uploaded_by_name
            FROM photos p
            LEFT JOIN users u ON p.uploaded_by = u.id
            WHERE p.assessment_id = ?
            ORDER BY p.display_order, p.uploaded_at
        ");
        $stmt->execute([$_GET['id']]);
        $photos = $stmt->fetchAll();
        ?>

        <?php if (!empty($photos)): ?>
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">📷 Assessment Photos</h2>
            </div>

            <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 1.5rem; padding: 1.5rem;">
                <?php foreach ($photos as $photo): ?>
                    <div style="background: var(--bg-tertiary); border-radius: 8px; overflow: hidden; border: 1px solid var(--border-color);">
                        <div style="position: relative; padding-top: 75%; background: var(--bg-secondary);">
                            <img src="uploads/photos/serve-photo.php?file=<?php echo urlencode($photo['filename']); ?>" 
                                 alt="<?php echo sanitizeOutput($photo['caption'] ?: 'Assessment photo'); ?>"
                                 style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; object-fit: cover; cursor: pointer;"
                                 onclick="openPhotoModal(this.src, '<?php echo addslashes($photo['caption'] ?: 'Assessment photo'); ?>')">
                        </div>
                        <?php if ($photo['caption']): ?>
                            <div style="padding: 1rem;">
                                <p style="color: var(--text-secondary); line-height: 1.6; margin: 0;">
                                    <?php echo nl2br(sanitizeOutput($photo['caption'])); ?>
                                </p>
                                <?php if ($photo['uploaded_by_name']): ?>
                                    <p style="color: var(--text-muted); font-size: 0.85rem; margin-top: 0.5rem;">
                                        📸 <?php echo sanitizeOutput($photo['uploaded_by_name']); ?> • 
                                        <?php echo date('d/m/Y H:i', strtotime($photo['uploaded_at'])); ?>
                                    </p>
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>

        <!-- Notes & Recommendations -->
        <?php if ($assessment['notes'] || $assessment['recommendations']): ?>
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">📝 Notes & Recommendations</h2>
                </div>

                <?php if ($assessment['notes']): ?>
                    <div style="padding: 1.5rem; border-bottom: 1px solid var(--border-color);">
                        <h3 style="color: var(--text-primary); margin-bottom: 0.5rem;">General Notes</h3>
                        <p style="color: var(--text-secondary); line-height: 1.8;">
                            <?php echo nl2br(sanitizeOutput($assessment['notes'])); ?>
                        </p>
                    </div>
                <?php endif; ?>

                <?php if ($assessment['recommendations']): ?>
                    <div style="padding: 1.5rem;">
                        <h3 style="color: var(--text-primary); margin-bottom: 0.5rem;">Recommendations</h3>
                        <p style="color: var(--text-secondary); line-height: 1.8;">
                            <?php echo nl2br(sanitizeOutput($assessment['recommendations'])); ?>
                        </p>
                    </div>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <!-- PDF Export Modal -->
    <div id="pdf-export-modal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h2 class="modal-title">Export Report</h2>
                <button type="button" class="modal-close" onclick="closePDFExport()">&times;</button>
            </div>

            <div style="padding: 1.5rem;">
                <p style="color: var(--text-muted); margin-bottom: 1.5rem;">
                    Generate a printable report of this risk assessment. You can save it as PDF using your browser's print function.
                </p>

                <div class="btn-group" style="margin-top: 2rem;">
                    <button onclick="generatePDF()" class="btn btn-primary" id="generate-btn">
                        📄 Generate Report
                    </button>
                    <button onclick="closePDFExport()" class="btn btn-secondary">
                        Cancel
                    </button>
                </div>

                <div id="pdf-status" style="margin-top: 1rem; display: none;"></div>
            </div>
        </div>
    </div>

    <!-- Photo Modal -->
    <div id="photo-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.9); z-index: 10000; cursor: pointer;" onclick="closePhotoModal()">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); max-width: 90%; max-height: 90%;">
            <img id="modal-photo" src="" alt="" style="max-width: 100%; max-height: 90vh; display: block; margin: 0 auto;">
            <p id="modal-caption" style="color: white; text-align: center; margin-top: 1rem; padding: 0 1rem; font-size: 1.1rem;"></p>
        </div>
        <button onclick="closePhotoModal()" style="position: absolute; top: 20px; right: 20px; background: rgba(255,255,255,0.2); border: none; color: white; font-size: 2rem; width: 50px; height: 50px; border-radius: 50%; cursor: pointer; line-height: 1;">×</button>
    </div>

    <script src="js/app.js"></script>
    <script>
        function showPDFExport() {
            document.getElementById('pdf-export-modal').style.display = 'flex';
            document.getElementById('pdf-export-modal').style.alignItems = 'center';
            document.getElementById('pdf-export-modal').style.justifyContent = 'center';
        }

        function closePDFExport() {
            document.getElementById('pdf-export-modal').style.display = 'none';
            document.getElementById('pdf-status').style.display = 'none';
        }

        async function generatePDF() {
            const assessmentId = <?php echo $assessment['id']; ?>;
            const btn = document.getElementById('generate-btn');
            const status = document.getElementById('pdf-status');
            
            // Show loading
            btn.disabled = true;
            btn.textContent = '⏳ Generating Report...';
            status.style.display = 'block';
            status.style.padding = '1rem';
            status.style.background = '#e3f2fd';
            status.style.borderRadius = '6px';
            status.style.color = '#1565c0';
            status.innerHTML = 'Generating your report. Please wait...';
            
            try {
                const url = `api/export-report.php?assessment_id=${assessmentId}`;
                const response = await fetch(url);
                const data = await response.json();
                
                if (data.success) {
                    status.style.background = '#d4edda';
                    status.style.color = '#155724';
                    status.innerHTML = `
                        ✅ Report generated successfully!<br>
                        <div style="margin-top: 1rem; display: flex; gap: 0.5rem; flex-wrap: wrap;">
                            <a href="${data.print_url}" target="_blank" class="btn btn-primary">
                                🖨️ Print / Save as PDF
                            </a>
                            <a href="${data.download_url}" class="btn btn-secondary">
                                📥 Download HTML
                            </a>
                        </div>
                    `;
                } else {
                    throw new Error(data.error || 'Report generation failed');
                }
            } catch (error) {
                status.style.background = '#f8d7da';
                status.style.color = '#721c24';
                status.innerHTML = '❌ Error: ' + error.message;
            } finally {
                btn.disabled = false;
                btn.textContent = '📄 Generate PDF';
            }
        }

        function openPhotoModal(src, caption) {
            document.getElementById('modal-photo').src = src;
            document.getElementById('modal-caption').textContent = caption;
            document.getElementById('photo-modal').style.display = 'block';
            document.body.style.overflow = 'hidden';
        }

        function closePhotoModal() {
            document.getElementById('photo-modal').style.display = 'none';
            document.body.style.overflow = 'auto';
        }

        // Close modal on Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closePhotoModal();
            }
        });
    </script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
