<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>View Assessment - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
    <?php if (file_exists(__DIR__ . '/css/custom-theme.css')): ?>
    <link rel="stylesheet" href="css/custom-theme.css">
    <?php endif; ?>
    
    <!-- Vue 3 -->
    <script src="https://unpkg.com/vue@3.4.15/dist/vue.global.prod.js"></script>
    <!-- Axios -->
    <script src="https://unpkg.com/axios@1.6.5/dist/axios.min.js"></script>
    <!-- Risk Item Photos Component -->
    <script src="js/risk-item-photos.js"></script>
</head>
<body>
    <?php
    require_once '../includes/config.php';
    requireLogin();

    if (!isset($_GET['id'])) {
        header('Location: assessments.php');
        exit;
    }

    // Fetch assessment with details
    $stmt = $pdo->prepare("
        SELECT a.*, l.*, u.full_name as assessor_name, u.email as assessor_email
        FROM assessments a
        JOIN locations l ON a.location_id = l.id
        JOIN users u ON a.assessor_id = u.id
        WHERE a.id = ?
    ");
    $stmt->execute([$_GET['id']]);
    $assessment = $stmt->fetch();

    if (!$assessment) {
        header('Location: assessments.php');
        exit;
    }

    // Fetch assessment details with risk items
    $stmt = $pdo->prepare("
        SELECT ad.*, ri.item_name, rc.category_name, rc.icon, rc.color
        FROM assessment_details ad
        JOIN risk_items ri ON ad.risk_item_id = ri.id
        JOIN risk_categories rc ON ri.category_id = rc.id
        WHERE ad.assessment_id = ?
        ORDER BY rc.display_order, ri.display_order
    ");
    $stmt->execute([$_GET['id']]);
    $details = $stmt->fetchAll();

    // Group by category
    $grouped = [];
    foreach ($details as $detail) {
        $grouped[$detail['category_name']][] = $detail;
    }

    // Fetch general assessment photos
    $stmt = $pdo->prepare("
        SELECT * FROM photos WHERE assessment_id = ? ORDER BY display_order
    ");
    $stmt->execute([$_GET['id']]);
    $generalPhotos = $stmt->fetchAll();

    include '../templates/header.php';
    ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">📋 Assessment Details</h1>
                <p class="page-subtitle"><?php echo sanitizeOutput($assessment['assessment_number']); ?></p>
            </div>
            <div class="btn-group">
                <a href="assessments.php" class="btn btn-secondary">← Back</a>
                <button onclick="generatePDF()" class="btn btn-primary">📄 Generate PDF</button>
            </div>
        </div>

        <!-- Assessment Info -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Assessment Information</h2>
            </div>
            <div class="card-body">
                <div class="info-grid">
                    <div>
                        <strong>Location:</strong><br>
                        <?php echo sanitizeOutput($assessment['location_name']); ?><br>
                        <small style="color: var(--text-muted);">
                            <?php echo sanitizeOutput($assessment['city'] . ', ' . $assessment['postcode']); ?>
                        </small>
                    </div>
                    <div>
                        <strong>Date & Time:</strong><br>
                        <?php echo date('d M Y', strtotime($assessment['assessment_date'])); ?><br>
                        <?php echo date('H:i', strtotime($assessment['assessment_time'])); ?>
                    </div>
                    <div>
                        <strong>Assessor:</strong><br>
                        <?php echo sanitizeOutput($assessment['assessor_name']); ?>
                    </div>
                    <div>
                        <strong>Overall Risk:</strong><br>
                        <span class="badge badge-<?php echo getRiskClass($assessment['overall_risk_level']); ?>">
                            <?php echo strtoupper($assessment['overall_risk_level']); ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Risk Items with Photos -->
        <?php foreach ($grouped as $category => $items): ?>
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">
                    <?php echo sanitizeOutput($items[0]['icon'] . ' ' . $category); ?>
                </h2>
            </div>
            <div class="card-body" style="padding: 0;">
                <?php foreach ($items as $index => $item): ?>
                <div class="risk-item-detail" style="border-bottom: <?php echo $index < count($items) - 1 ? '1px solid var(--border-color)' : 'none'; ?>;">
                    <div style="padding: 1.5rem;">
                        <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1rem;">
                            <div style="flex: 1;">
                                <h3 style="margin: 0 0 0.5rem 0; color: var(--text-primary);">
                                    <?php echo sanitizeOutput($item['item_name']); ?>
                                </h3>
                                <div style="display: flex; gap: 1rem; flex-wrap: wrap;">
                                    <div>
                                        <strong>Likelihood:</strong> 
                                        <span class="badge badge-info"><?php echo $item['likelihood']; ?>/5</span>
                                    </div>
                                    <div>
                                        <strong>Severity:</strong> 
                                        <span class="badge badge-info"><?php echo $item['severity']; ?>/5</span>
                                    </div>
                                    <div>
                                        <strong>Risk Score:</strong> 
                                        <span class="badge badge-<?php echo getRiskClassByScore($item['risk_score']); ?>">
                                            <?php echo $item['risk_score']; ?>
                                        </span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <?php if ($item['current_controls']): ?>
                        <div style="margin-bottom: 1rem;">
                            <strong>Current Controls:</strong>
                            <p style="margin: 0.5rem 0 0 0; color: var(--text-secondary);">
                                <?php echo nl2br(sanitizeOutput($item['current_controls'])); ?>
                            </p>
                        </div>
                        <?php endif; ?>

                        <?php if ($item['additional_actions']): ?>
                        <div style="margin-bottom: 1rem;">
                            <strong>Additional Actions:</strong>
                            <p style="margin: 0.5rem 0 0 0; color: var(--text-secondary);">
                                <?php echo nl2br(sanitizeOutput($item['additional_actions'])); ?>
                            </p>
                        </div>
                        <?php endif; ?>

                        <!-- Risk Item Photos (Vue Component) -->
                        <div id="risk-photos-<?php echo $item['id']; ?>">
                            <h4 style="margin: 1rem 0 0.5rem 0; color: var(--text-primary);">
                                📷 Photos for this Risk Item
                            </h4>
                            <risk-item-photos
                                :assessment-id="<?php echo $assessment['id']; ?>"
                                :risk-item-id="<?php echo $item['risk_item_id']; ?>"
                                :assessment-detail-id="<?php echo $item['id']; ?>"
                                :editable="<?php echo $_SESSION['role'] === 'admin' || $_SESSION['user_id'] == $assessment['assessor_id'] ? 'true' : 'false'; ?>"
                            ></risk-item-photos>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endforeach; ?>

        <!-- General Assessment Photos -->
        <?php if ($generalPhotos): ?>
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📸 General Assessment Photos</h2>
            </div>
            <div class="card-body">
                <div class="photos-grid">
                    <?php foreach ($generalPhotos as $photo): ?>
                    <div class="photo-item">
                        <img src="<?php echo sanitizeOutput($photo['file_path']); ?>" 
                             alt="<?php echo sanitizeOutput($photo['caption']); ?>"
                             onclick="window.open(this.src, '_blank')">
                        <?php if ($photo['caption']): ?>
                        <p style="margin-top: 0.5rem; font-size: 0.9rem;">
                            <?php echo sanitizeOutput($photo['caption']); ?>
                        </p>
                        <?php endif; ?>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Notes & Recommendations -->
        <?php if ($assessment['notes'] || $assessment['recommendations']): ?>
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📝 Notes & Recommendations</h2>
            </div>
            <div class="card-body">
                <?php if ($assessment['notes']): ?>
                <div style="margin-bottom: 1.5rem;">
                    <strong>Notes:</strong>
                    <p style="margin: 0.5rem 0 0 0; color: var(--text-secondary);">
                        <?php echo nl2br(sanitizeOutput($assessment['notes'])); ?>
                    </p>
                </div>
                <?php endif; ?>

                <?php if ($assessment['recommendations']): ?>
                <div>
                    <strong>Recommendations:</strong>
                    <p style="margin: 0.5rem 0 0 0; color: var(--text-secondary);">
                        <?php echo nl2br(sanitizeOutput($assessment['recommendations'])); ?>
                    </p>
                </div>
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <style>
        .risk-item-detail {
            background: var(--bg-primary);
        }

        .risk-item-detail:hover {
            background: var(--bg-secondary);
        }

        .photos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 1rem;
        }

        .photo-item {
            cursor: pointer;
        }

        .photo-item img {
            width: 100%;
            height: 200px;
            object-fit: cover;
            border-radius: 8px;
            transition: var(--transition);
        }

        .photo-item img:hover {
            transform: scale(1.05);
            box-shadow: var(--shadow);
        }
    </style>

    <script>
        const { createApp } = Vue;

        // Create Vue app for each risk item
        <?php foreach ($details as $item): ?>
        createApp({
            components: {
                'risk-item-photos': RiskItemPhotos
            }
        }).mount('#risk-photos-<?php echo $item['id']; ?>');
        <?php endforeach; ?>

        function generatePDF() {
            window.location.href = 'api/export-report.php?id=<?php echo $assessment['id']; ?>';
        }
    </script>

    <?php include '../templates/footer.php'; ?>

    <?php
    function getRiskClass($level) {
        $classes = [
            'low' => 'success',
            'medium' => 'warning',
            'high' => 'danger',
            'critical' => 'danger'
        ];
        return $classes[$level] ?? 'info';
    }

    function getRiskClassByScore($score) {
        if ($score <= 5) return 'success';
        if ($score <= 10) return 'warning';
        if ($score <= 15) return 'danger';
        return 'danger';
    }
    ?>
</body>
</html>
