<?php
require_once '../includes/config.php';
requireLogin();

$assessment_id = $_GET['id'] ?? null;
if (!$assessment_id) {
    header('Location: assessments.php');
    exit;
}

// Get assessment info
$stmt = $pdo->prepare("
    SELECT a.*, l.location_name, u.full_name as assessor_name
    FROM assessments a
    JOIN locations l ON a.location_id = l.id
    JOIN users u ON a.assessor_id = u.id
    WHERE a.id = ?
");
$stmt->execute([$assessment_id]);
$assessment = $stmt->fetch();

if (!$assessment) {
    header('Location: assessments.php');
    exit;
}

// Get all versions
$stmt = $pdo->prepare("
    SELECT av.*, u.full_name as created_by_name,
           (SELECT COUNT(*) FROM assessment_version_details WHERE version_id = av.id) as detail_count
    FROM assessment_versions av
    JOIN users u ON av.created_by = u.id
    WHERE av.assessment_id = ?
    ORDER BY av.version_number DESC
");
$stmt->execute([$assessment_id]);
$versions = $stmt->fetchAll();

// Get change log
$stmt = $pdo->prepare("
    SELECT acl.*, u.full_name as changed_by_name,
           fv.version_number as from_version_number,
           tv.version_number as to_version_number
    FROM assessment_change_log acl
    JOIN users u ON acl.changed_by = u.id
    LEFT JOIN assessment_versions fv ON acl.from_version = fv.id
    JOIN assessment_versions tv ON acl.to_version = tv.id
    WHERE acl.assessment_id = ?
    ORDER BY acl.changed_at DESC
");
$stmt->execute([$assessment_id]);
$change_log = $stmt->fetchAll();

// Get archive info
$stmt = $pdo->prepare("
    SELECT aa.*, u.full_name as archived_by_name, av.version_number
    FROM assessment_archives aa
    JOIN users u ON aa.archived_by = u.id
    JOIN assessment_versions av ON aa.version_id = av.id
    WHERE aa.assessment_id = ?
    ORDER BY aa.archived_at DESC
");
$stmt->execute([$assessment_id]);
$archives = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Version History - <?php echo sanitizeOutput($assessment['assessment_number']); ?></title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .version-timeline {
            position: relative;
            padding-left: 3rem;
        }
        
        .version-timeline::before {
            content: '';
            position: absolute;
            left: 1rem;
            top: 0;
            bottom: 0;
            width: 2px;
            background: var(--border-color);
        }
        
        .version-item {
            position: relative;
            margin-bottom: 2rem;
            padding: 1.5rem;
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 8px;
        }
        
        .version-item.current {
            border-color: var(--success);
            border-width: 2px;
        }
        
        .version-item::before {
            content: '';
            position: absolute;
            left: -2.5rem;
            top: 1.5rem;
            width: 1rem;
            height: 1rem;
            background: white;
            border: 3px solid var(--primary);
            border-radius: 50%;
        }
        
        .version-item.current::before {
            background: var(--success);
            border-color: var(--success);
        }
        
        .version-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 1rem;
        }
        
        .version-badge {
            display: inline-block;
            padding: 0.25rem 0.75rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .version-badge.current {
            background: var(--success);
            color: white;
        }
        
        .version-badge.archived {
            background: rgba(148, 163, 184, 0.2);
            color: var(--text-muted);
        }
        
        .version-badge.superseded {
            background: rgba(245, 158, 11, 0.2);
            color: #d97706;
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">📚 Version History</h1>
                <p class="page-subtitle">
                    <?php echo sanitizeOutput($assessment['assessment_number']); ?> - 
                    <?php echo sanitizeOutput($assessment['location_name']); ?>
                </p>
            </div>
            <div class="btn-group">
                <a href="create-revision.php?assessment_id=<?php echo $assessment_id; ?>" 
                   class="btn btn-primary">
                    ➕ Create Revision
                </a>
                <a href="view-assessment.php?id=<?php echo $assessment_id; ?>" class="btn btn-secondary">
                    ← Back to Assessment
                </a>
            </div>
        </div>

        <!-- Summary Stats -->
        <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 2rem;">
            <div style="background: var(--bg-card); padding: 1.5rem; border-radius: 8px; border: 1px solid var(--border-color);">
                <div style="font-size: 2rem; font-weight: 700; color: var(--primary);">
                    <?php echo count($versions); ?>
                </div>
                <div style="color: var(--text-muted); font-size: 0.9rem;">Total Versions</div>
            </div>
            
            <div style="background: var(--bg-card); padding: 1.5rem; border-radius: 8px; border: 1px solid var(--border-color);">
                <div style="font-size: 2rem; font-weight: 700; color: var(--success);">
                    v<?php echo $assessment['version_count'] ?? 1; ?>
                </div>
                <div style="color: var(--text-muted); font-size: 0.9rem;">Current Version</div>
            </div>
            
            <div style="background: var(--bg-card); padding: 1.5rem; border-radius: 8px; border: 1px solid var(--border-color);">
                <div style="font-size: 2rem; font-weight: 700; color: #f59e0b;">
                    <?php echo count($archives); ?>
                </div>
                <div style="color: var(--text-muted); font-size: 0.9rem;">Archived</div>
            </div>
            
            <div style="background: var(--bg-card); padding: 1.5rem; border-radius: 8px; border: 1px solid var(--border-color);">
                <div style="font-size: 2rem; font-weight: 700; color: #8b5cf6;">
                    <?php echo count($change_log); ?>
                </div>
                <div style="color: var(--text-muted); font-size: 0.9rem;">Changes Logged</div>
            </div>
        </div>

        <!-- Version Timeline -->
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">Version Timeline</h2>
            </div>

            <div style="padding: 2rem;">
                <div class="version-timeline">
                    <?php foreach ($versions as $version): ?>
                        <div class="version-item <?php echo $version['is_current'] ? 'current' : ''; ?>">
                            <div class="version-header">
                                <div>
                                    <h3 style="color: var(--text-primary); margin-bottom: 0.5rem;">
                                        Version <?php echo $version['version_number']; ?>
                                        <?php if ($version['is_current']): ?>
                                            <span class="version-badge current">Current</span>
                                        <?php elseif ($version['status'] === 'archived'): ?>
                                            <span class="version-badge archived">Archived</span>
                                        <?php elseif ($version['status'] === 'superseded'): ?>
                                            <span class="version-badge superseded">Superseded</span>
                                        <?php endif; ?>
                                    </h3>
                                    <div style="color: var(--text-muted); font-size: 0.9rem;">
                                        <?php echo ucfirst($version['version_type']); ?> • 
                                        Created by <?php echo sanitizeOutput($version['created_by_name']); ?> on 
                                        <?php echo date('d/m/Y H:i', strtotime($version['created_at'])); ?>
                                    </div>
                                </div>
                                
                                <div class="btn-group" style="gap: 0.5rem;">
                                    <a href="view-version.php?version_id=<?php echo $version['id']; ?>" 
                                       class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                        👁️ View
                                    </a>
                                    <button onclick="compareVersions(<?php echo $version['id']; ?>)" 
                                            class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                        🔄 Compare
                                    </button>
                                    <?php if (!$version['is_current'] && $version['status'] !== 'archived'): ?>
                                        <button onclick="restoreVersion(<?php echo $version['id']; ?>)" 
                                                class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                            ↩️ Restore
                                        </button>
                                    <?php endif; ?>
                                </div>
                            </div>

                            <?php if ($version['revision_reason']): ?>
                                <div style="margin-top: 1rem; padding: 1rem; background: var(--bg-tertiary); border-radius: 6px;">
                                    <strong style="color: var(--text-primary);">Revision Reason:</strong>
                                    <p style="color: var(--text-secondary); margin: 0.5rem 0 0 0;">
                                        <?php echo sanitizeOutput($version['revision_reason']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>

                            <?php if ($version['changes_summary']): ?>
                                <div style="margin-top: 1rem; padding: 1rem; background: #e3f2fd; border-radius: 6px;">
                                    <strong style="color: #1565c0;">Changes Summary:</strong>
                                    <p style="color: #1565c0; margin: 0.5rem 0 0 0;">
                                        <?php echo sanitizeOutput($version['changes_summary']); ?>
                                    </p>
                                </div>
                            <?php endif; ?>

                            <div style="margin-top: 1rem; display: flex; gap: 1.5rem; color: var(--text-muted); font-size: 0.85rem;">
                                <span>📊 <?php echo $version['detail_count']; ?> risk items</span>
                                <span>⚠️ <?php echo strtoupper($version['overall_risk_level']); ?> risk</span>
                                <span>📝 <?php echo ucfirst($version['status']); ?></span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <!-- Change Log -->
        <?php if (!empty($change_log)): ?>
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">Change Log</h2>
                </div>

                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Date/Time</th>
                                <th>Action</th>
                                <th>Version</th>
                                <th>Changed By</th>
                                <th>Description</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($change_log as $log): ?>
                                <tr>
                                    <td><?php echo date('d/m/Y H:i', strtotime($log['changed_at'])); ?></td>
                                    <td>
                                        <span class="badge" style="background: var(--primary); color: white;">
                                            <?php echo strtoupper($log['action_type']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($log['from_version_number']): ?>
                                            v<?php echo $log['from_version_number']; ?> → 
                                        <?php endif; ?>
                                        v<?php echo $log['to_version_number']; ?>
                                    </td>
                                    <td><?php echo sanitizeOutput($log['changed_by_name']); ?></td>
                                    <td><?php echo sanitizeOutput($log['change_description'] ?? '-'); ?></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>

        <!-- Archive History -->
        <?php if (!empty($archives)): ?>
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">Archive History</h2>
                </div>

                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Archived Date</th>
                                <th>Version</th>
                                <th>Reason</th>
                                <th>Archived By</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($archives as $archive): ?>
                                <tr>
                                    <td><?php echo date('d/m/Y H:i', strtotime($archive['archived_at'])); ?></td>
                                    <td>v<?php echo $archive['version_number']; ?></td>
                                    <td><?php echo sanitizeOutput($archive['archive_reason']); ?></td>
                                    <td><?php echo sanitizeOutput($archive['archived_by_name']); ?></td>
                                    <td>
                                        <?php if ($archive['can_restore']): ?>
                                            <span class="badge badge-success">Can Restore</span>
                                        <?php else: ?>
                                            <span class="badge" style="background: rgba(239, 68, 68, 0.2); color: #dc2626;">
                                                Permanent
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script src="js/app.js"></script>
    <script>
        function compareVersions(versionId) {
            const currentVersion = <?php echo $assessment['current_version_id'] ?? 'null'; ?>;
            if (!currentVersion) {
                alert('No current version to compare against');
                return;
            }
            window.location.href = `compare-versions.php?v1=${currentVersion}&v2=${versionId}`;
        }

        function restoreVersion(versionId) {
            if (confirm('Restore this version as the current version? This will create a new revision.')) {
                window.location.href = `restore-version.php?version_id=${versionId}`;
            }
        }
    </script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
