<?php
require_once '../includes/config.php';
require_once '../includes/permissions.php';
require_once '../includes/version_control.php';
requireLogin();

// Only admins can manage version control
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$success = '';
$error = '';
$view = isset($_GET['view']) ? $_GET['view'] : 'overview';
$file_path = isset($_GET['file']) ? $_GET['file'] : null;

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    try {
        switch($_POST['action']) {
            case 'rollback':
                $result = rollbackFileVersion(
                    $pdo,
                    $_POST['file_path'],
                    (int)$_POST['target_version'],
                    $_SESSION['user_id'],
                    $_POST['reason'] ?? ''
                );
                
                if ($result['success']) {
                    $success = $result['message'];
                } else {
                    throw new Exception($result['error']);
                }
                break;
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Fetch data based on view
$tracked_files = getTrackedFiles($pdo);
$recent_changes = getRecentChanges($pdo, 50);

// Get file history if viewing specific file
$file_history = [];
if ($file_path) {
    $file_history = getFileHistory($pdo, $file_path, 20);
}

// Get system releases
$stmt = $pdo->query("
    SELECT sr.*, u.full_name as released_by_name,
           COUNT(rf.id) as file_count
    FROM system_releases sr
    JOIN users u ON sr.released_by = u.id
    LEFT JOIN release_files rf ON sr.id = rf.release_id
    GROUP BY sr.id
    ORDER BY sr.release_date DESC
    LIMIT 10
");
$releases = $stmt->fetchAll();

// Calculate statistics
$total_files = count($tracked_files);
$total_versions = $pdo->query("SELECT COUNT(*) FROM file_versions")->fetchColumn();
$files_by_type = [];
foreach ($tracked_files as $file) {
    $type = $file['file_type'];
    if (!isset($files_by_type[$type])) {
        $files_by_type[$type] = 0;
    }
    $files_by_type[$type]++;
}

include '../templates/header.php';
?>

<div class="container">
    <div class="page-header">
        <div>
            <h1 class="page-title">🔄 Version Control</h1>
            <p class="page-subtitle">Track, manage, and rollback system file changes</p>
        </div>
        <div class="btn-group">
            <a href="admin-panel.php" class="btn btn-secondary">← Back to Admin</a>
        </div>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success">✅ <?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger">❌ <?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <!-- Navigation Tabs -->
    <div class="version-tabs">
        <a href="?view=overview" class="version-tab <?php echo $view === 'overview' ? 'active' : ''; ?>">
            📊 Overview
        </a>
        <a href="?view=files" class="version-tab <?php echo $view === 'files' ? 'active' : ''; ?>">
            📁 Files
        </a>
        <a href="?view=changes" class="version-tab <?php echo $view === 'changes' ? 'active' : ''; ?>">
            📝 Recent Changes
        </a>
        <a href="?view=releases" class="version-tab <?php echo $view === 'releases' ? 'active' : ''; ?>">
            🚀 Releases
        </a>
    </div>

    <?php if ($view === 'overview'): ?>
        <!-- Overview -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon" style="color: var(--primary);">📁</div>
                <div>
                    <div class="stat-value"><?php echo $total_files; ?></div>
                    <div class="stat-label">Tracked Files</div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-icon" style="color: var(--info);">🔢</div>
                <div>
                    <div class="stat-value"><?php echo $total_versions; ?></div>
                    <div class="stat-label">Total Versions</div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-icon" style="color: var(--success);">✅</div>
                <div>
                    <div class="stat-value"><?php echo number_format($total_versions / max($total_files, 1), 1); ?></div>
                    <div class="stat-label">Avg Versions/File</div>
                </div>
            </div>

            <div class="stat-card">
                <div class="stat-icon" style="color: var(--warning);">🚀</div>
                <div>
                    <div class="stat-value"><?php echo count($releases); ?></div>
                    <div class="stat-label">Releases</div>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Files by Type</h2>
            </div>
            <div class="card-body">
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem;">
                    <?php foreach ($files_by_type as $type => $count): ?>
                    <div style="padding: 1rem; background: var(--bg-secondary); border-radius: 8px; text-align: center;">
                        <div style="font-size: 2rem; margin-bottom: 0.5rem;">
                            <?php echo getFileTypeIcon($type); ?>
                        </div>
                        <div style="font-size: 1.5rem; font-weight: 600; color: var(--text-primary);">
                            <?php echo $count; ?>
                        </div>
                        <div style="color: var(--text-muted); text-transform: uppercase; font-size: 0.85rem;">
                            <?php echo $type; ?> Files
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Recent Activity</h2>
            </div>
            <div class="card-body" style="padding: 0;">
                <?php foreach (array_slice($recent_changes, 0, 10) as $change): ?>
                <div class="activity-item">
                    <div class="activity-icon"><?php echo getChangeTypeIcon($change['change_type']); ?></div>
                    <div class="activity-content">
                        <strong><?php echo htmlspecialchars($change['file_name']); ?></strong>
                        <span style="color: var(--text-muted);">v<?php echo $change['version_number']; ?></span>
                        <p style="margin: 0.25rem 0 0 0; color: var(--text-muted);">
                            <?php echo htmlspecialchars($change['change_summary']); ?>
                        </p>
                    </div>
                    <div class="activity-meta">
                        <div><?php echo htmlspecialchars($change['changed_by']); ?></div>
                        <div style="color: var(--text-muted); font-size: 0.85rem;">
                            <?php echo date('d M Y H:i', strtotime($change['change_date'])); ?>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>

    <?php elseif ($view === 'files'): ?>
        <!-- Files List -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Tracked Files (<?php echo count($tracked_files); ?>)</h2>
            </div>
            <div class="card-body" style="padding: 0;">
                <div class="table-container">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>File</th>
                                <th>Type</th>
                                <th>Category</th>
                                <th>Version</th>
                                <th>Last Modified</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($tracked_files as $file): ?>
                            <tr>
                                <td>
                                    <strong><?php echo htmlspecialchars($file['file_name']); ?></strong><br>
                                    <small style="color: var(--text-muted);"><?php echo htmlspecialchars($file['file_path']); ?></small>
                                </td>
                                <td>
                                    <span class="badge badge-info">
                                        <?php echo strtoupper($file['file_type']); ?>
                                    </span>
                                </td>
                                <td><?php echo ucfirst($file['file_category']); ?></td>
                                <td>v<?php echo $file['current_version'] ?? '1'; ?></td>
                                <td><?php echo $file['last_modified'] ? date('d M Y H:i', strtotime($file['last_modified'])) : 'Never'; ?></td>
                                <td>
                                    <a href="?view=history&file=<?php echo urlencode($file['file_path']); ?>" class="btn btn-secondary">
                                        View History
                                    </a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

    <?php elseif ($view === 'changes'): ?>
        <!-- Recent Changes -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Recent Changes (Last 50)</h2>
            </div>
            <div class="card-body" style="padding: 0;">
                <?php foreach ($recent_changes as $change): ?>
                <div class="change-item">
                    <div class="change-icon"><?php echo getChangeTypeIcon($change['change_type']); ?></div>
                    <div class="change-content">
                        <div class="change-header">
                            <strong><?php echo htmlspecialchars($change['file_name']); ?></strong>
                            <span class="badge badge-<?php echo getChangeTypeBadge($change['change_type']); ?>">
                                <?php echo ucfirst(str_replace('_', ' ', $change['change_type'])); ?>
                            </span>
                        </div>
                        <p style="margin: 0.5rem 0; color: var(--text-secondary);">
                            <?php echo htmlspecialchars($change['change_summary']); ?>
                        </p>
                        <div class="change-meta">
                            <span>Version <?php echo $change['version_number']; ?></span>
                            <span>by <?php echo htmlspecialchars($change['changed_by']); ?></span>
                            <span><?php echo date('d M Y H:i', strtotime($change['change_date'])); ?></span>
                            <a href="?view=history&file=<?php echo urlencode($change['file_path']); ?>">View History</a>
                        </div>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>

    <?php elseif ($view === 'releases'): ?>
        <!-- Releases -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">System Releases</h2>
            </div>
            <div class="card-body">
                <?php if (empty($releases)): ?>
                    <div style="text-align: center; padding: 2rem; color: var(--text-muted);">
                        No releases yet
                    </div>
                <?php else: ?>
                    <?php foreach ($releases as $release): ?>
                    <div class="release-item">
                        <div class="release-header">
                            <div>
                                <h3 style="margin: 0; color: var(--text-primary);">
                                    <?php echo htmlspecialchars($release['release_name']); ?>
                                </h3>
                                <div style="margin-top: 0.25rem;">
                                    <span class="badge badge-<?php echo getReleaseTypeBadge($release['release_type']); ?>">
                                        <?php echo strtoupper($release['release_type']); ?>
                                    </span>
                                    <span style="color: var(--text-muted); margin-left: 0.5rem;">
                                        v<?php echo htmlspecialchars($release['release_version']); ?>
                                    </span>
                                </div>
                            </div>
                            <div style="text-align: right;">
                                <div><?php echo htmlspecialchars($release['released_by_name']); ?></div>
                                <div style="color: var(--text-muted); font-size: 0.85rem;">
                                    <?php echo date('d M Y', strtotime($release['release_date'])); ?>
                                </div>
                            </div>
                        </div>
                        <p style="margin: 1rem 0; color: var(--text-secondary);">
                            <?php echo nl2br(htmlspecialchars($release['release_notes'])); ?>
                        </p>
                        <div style="color: var(--text-muted); font-size: 0.9rem;">
                            <?php echo $release['file_count']; ?> files in this release
                        </div>
                    </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

    <?php elseif ($view === 'history' && $file_path): ?>
        <!-- File History -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title"><?php echo htmlspecialchars(basename($file_path)); ?> - Version History</h2>
            </div>
            <div class="card-body">
                <p style="color: var(--text-muted); margin-bottom: 1.5rem;">
                    Path: <?php echo htmlspecialchars($file_path); ?>
                </p>

                <?php if (empty($file_history)): ?>
                    <div style="text-align: center; padding: 2rem; color: var(--text-muted);">
                        No version history available
                    </div>
                <?php else: ?>
                    <div class="timeline">
                        <?php foreach ($file_history as $version): ?>
                        <div class="timeline-item">
                            <div class="timeline-marker">v<?php echo $version['version_number']; ?></div>
                            <div class="timeline-content">
                                <div class="timeline-header">
                                    <strong><?php echo htmlspecialchars($version['change_summary']); ?></strong>
                                    <span class="badge badge-<?php echo getChangeTypeBadge($version['change_type']); ?>">
                                        <?php echo ucfirst(str_replace('_', ' ', $version['change_type'])); ?>
                                    </span>
                                </div>
                                <?php if ($version['change_description']): ?>
                                <p style="margin: 0.5rem 0; color: var(--text-secondary);">
                                    <?php echo nl2br(htmlspecialchars($version['change_description'])); ?>
                                </p>
                                <?php endif; ?>
                                <div style="margin-top: 0.5rem; color: var(--text-muted); font-size: 0.85rem;">
                                    <?php echo htmlspecialchars($version['changed_by_name']); ?> • 
                                    <?php echo date('d M Y H:i', strtotime($version['change_date'])); ?> • 
                                    <?php echo number_format($version['file_size']); ?> bytes
                                </div>
                                <?php if ($version['version_number'] < $file_history[0]['version_number']): ?>
                                <form method="POST" style="margin-top: 1rem;" onsubmit="return confirm('Rollback to this version?');">
                                    <input type="hidden" name="action" value="rollback">
                                    <input type="hidden" name="file_path" value="<?php echo htmlspecialchars($file_path); ?>">
                                    <input type="hidden" name="target_version" value="<?php echo $version['version_number']; ?>">
                                    <input type="text" name="reason" placeholder="Reason for rollback..." class="form-control" style="margin-bottom: 0.5rem;" required>
                                    <button type="submit" class="btn btn-warning">🔄 Rollback to v<?php echo $version['version_number']; ?></button>
                                </form>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<style>
.version-tabs {
    display: flex;
    gap: 0.5rem;
    margin-bottom: 1.5rem;
    border-bottom: 2px solid var(--border-color);
}

.version-tab {
    padding: 0.75rem 1.5rem;
    text-decoration: none;
    color: var(--text-secondary);
    border-bottom: 3px solid transparent;
    transition: var(--transition);
}

.version-tab:hover {
    color: var(--primary);
}

.version-tab.active {
    color: var(--primary);
    border-bottom-color: var(--primary);
    font-weight: 600;
}

.activity-item, .change-item {
    display: flex;
    gap: 1rem;
    padding: 1rem;
    border-bottom: 1px solid var(--border-color);
    align-items: start;
}

.activity-item:last-child, .change-item:last-child {
    border-bottom: none;
}

.activity-icon, .change-icon {
    font-size: 1.5rem;
}

.activity-content, .change-content {
    flex: 1;
}

.change-header {
    display: flex;
    gap: 1rem;
    align-items: center;
    margin-bottom: 0.5rem;
}

.change-meta {
    display: flex;
    gap: 1rem;
    font-size: 0.85rem;
    color: var(--text-muted);
    margin-top: 0.5rem;
}

.release-item {
    padding: 1.5rem;
    background: var(--bg-secondary);
    border-radius: 8px;
    margin-bottom: 1rem;
}

.release-header {
    display: flex;
    justify-content: space-between;
    margin-bottom: 1rem;
}

.timeline {
    position: relative;
    padding-left: 2rem;
}

.timeline::before {
    content: '';
    position: absolute;
    left: 0;
    top: 0;
    bottom: 0;
    width: 2px;
    background: var(--border-color);
}

.timeline-item {
    position: relative;
    padding-bottom: 2rem;
}

.timeline-marker {
    position: absolute;
    left: -2.5rem;
    background: var(--primary);
    color: white;
    padding: 0.25rem 0.5rem;
    border-radius: 12px;
    font-size: 0.85rem;
    font-weight: 600;
}

.timeline-content {
    background: var(--bg-secondary);
    padding: 1rem;
    border-radius: 8px;
}

.timeline-header {
    display: flex;
    gap: 1rem;
    align-items: center;
    margin-bottom: 0.5rem;
}

.alert {
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1.5rem;
}

.alert-success {
    background: rgba(5, 150, 105, 0.1);
    border: 1px solid var(--success);
    color: var(--success);
}

.alert-danger {
    background: rgba(220, 38, 38, 0.1);
    border: 1px solid var(--danger);
    color: var(--danger);
}
</style>

<?php include '../templates/footer.php'; ?>

<?php
function getFileTypeIcon($type) {
    $icons = [
        'php' => '🐘',
        'js' => '📜',
        'css' => '🎨',
        'sql' => '🗄️',
        'html' => '📄',
        'config' => '⚙️',
        'template' => '📋'
    ];
    return $icons[$type] ?? '📁';
}

function getChangeTypeIcon($type) {
    $icons = [
        'create' => '➕',
        'update' => '✏️',
        'bugfix' => '🐛',
        'feature' => '✨',
        'security' => '🔒',
        'refactor' => '♻️'
    ];
    return $icons[$type] ?? '📝';
}

function getChangeTypeBadge($type) {
    $badges = [
        'create' => 'success',
        'update' => 'info',
        'bugfix' => 'warning',
        'feature' => 'primary',
        'security' => 'danger',
        'refactor' => 'secondary'
    ];
    return $badges[$type] ?? 'info';
}

function getReleaseTypeBadge($type) {
    $badges = [
        'major' => 'danger',
        'minor' => 'info',
        'patch' => 'success',
        'hotfix' => 'warning'
    ];
    return $badges[$type] ?? 'info';
}
?>
