<?php
require_once '../includes/config.php';
requireLogin();

// Check admin access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$success = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['add_type'])) {
            // Add new vehicle type
            $stmt = $pdo->prepare("
                INSERT INTO vehicle_types 
                (type_name, description, icon, requires_license, license_class, max_capacity, fuel_type, display_order, created_by)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $_POST['type_name'],
                $_POST['description'] ?? null,
                $_POST['icon'] ?? '🚗',
                isset($_POST['requires_license']) ? 1 : 0,
                $_POST['license_class'] ?? null,
                $_POST['max_capacity'] ?? null,
                $_POST['fuel_type'] ?? null,
                $_POST['display_order'] ?? 0,
                $_SESSION['user_id']
            ]);
            $success = "Vehicle type added successfully!";
            
        } elseif (isset($_POST['update_type'])) {
            // Update existing type
            $stmt = $pdo->prepare("
                UPDATE vehicle_types 
                SET type_name = ?, description = ?, icon = ?, requires_license = ?, 
                    license_class = ?, max_capacity = ?, fuel_type = ?, display_order = ?, is_active = ?
                WHERE id = ?
            ");
            $stmt->execute([
                $_POST['type_name'],
                $_POST['description'] ?? null,
                $_POST['icon'] ?? '🚗',
                isset($_POST['requires_license']) ? 1 : 0,
                $_POST['license_class'] ?? null,
                $_POST['max_capacity'] ?? null,
                $_POST['fuel_type'] ?? null,
                $_POST['display_order'] ?? 0,
                isset($_POST['is_active']) ? 1 : 0,
                $_POST['type_id']
            ]);
            $success = "Vehicle type updated successfully!";
            
        } elseif (isset($_POST['delete_type'])) {
            // Check if type is in use
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM vehicle_assignments WHERE vehicle_type_id = ?");
            $stmt->execute([$_POST['type_id']]);
            $inUse = $stmt->fetchColumn();
            
            if ($inUse > 0) {
                $error = "Cannot delete vehicle type: it is currently assigned to $inUse vehicle(s)";
            } else {
                $stmt = $pdo->prepare("DELETE FROM vehicle_types WHERE id = ?");
                $stmt->execute([$_POST['type_id']]);
                $success = "Vehicle type deleted successfully!";
            }
        }
    } catch (PDOException $e) {
        $error = "Database error: " . $e->getMessage();
        error_log("Vehicle types error: " . $e->getMessage());
    }
}

// Get all vehicle types
try {
    $stmt = $pdo->query("
        SELECT vt.*, 
               COUNT(va.id) as assignment_count,
               u.full_name as created_by_name
        FROM vehicle_types vt
        LEFT JOIN vehicle_assignments va ON vt.id = va.vehicle_type_id AND va.is_active = 1
        LEFT JOIN users u ON vt.created_by = u.id
        GROUP BY vt.id
        ORDER BY vt.display_order, vt.type_name
    ");
    $types = $stmt->fetchAll();
} catch (PDOException $e) {
    $error = "Could not load vehicle types: " . $e->getMessage();
    $types = [];
}

$pageTitle = "Vehicle Types Management";
include '../templates/header.php';
?>

<style>
.vehicle-type-card {
    border-left: 4px solid var(--primary);
    transition: all 0.2s ease;
}
.vehicle-type-card:hover {
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    transform: translateY(-2px);
}
.vehicle-icon {
    font-size: 3rem;
    text-align: center;
    padding: 1rem;
}
.modal {
    display: none;
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    z-index: 1000;
    align-items: center;
    justify-content: center;
}
.modal.active {
    display: flex;
}
.modal-content {
    background: white;
    border-radius: 8px;
    max-width: 600px;
    width: 90%;
    max-height: 90vh;
    overflow-y: auto;
}
.modal-header {
    padding: 1.5rem;
    border-bottom: 1px solid #e5e7eb;
    display: flex;
    justify-content: space-between;
    align-items: center;
}
.modal-body {
    padding: 1.5rem;
}
.modal-footer {
    padding: 1.5rem;
    border-top: 1px solid #e5e7eb;
    display: flex;
    gap: 0.5rem;
    justify-content: flex-end;
}
.modal-close {
    background: none;
    border: none;
    font-size: 1.5rem;
    cursor: pointer;
    color: #6b7280;
}
.badge-count {
    display: inline-flex;
    align-items: center;
    padding: 0.25rem 0.75rem;
    font-size: 0.875rem;
    font-weight: 600;
    border-radius: 9999px;
    background: var(--gray-100);
    color: var(--text-secondary);
}
</style>

<div class="container">
    <div class="page-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 2rem;">
        <div>
            <h1>🚗 Vehicle Types Management</h1>
            <p style="color: var(--text-muted); margin-top: 0.5rem;">Manage vehicle types and classifications</p>
        </div>
        <div style="display: flex; gap: 1rem;">
            <button class="btn btn-primary" onclick="openAddModal()">+ Add Vehicle Type</button>
            <a href="admin-panel.php" class="btn btn-secondary">← Back to Admin</a>
        </div>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success" style="margin-bottom: 1.5rem;">
            ✓ <?php echo htmlspecialchars($success); ?>
        </div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="alert alert-danger" style="margin-bottom: 1.5rem;">
            ✗ <?php echo htmlspecialchars($error); ?>
        </div>
    <?php endif; ?>

    <?php if (empty($types)): ?>
        <div class="card">
            <div class="card-body" style="text-align: center; padding: 3rem;">
                <div style="font-size: 4rem; margin-bottom: 1rem;">🚗</div>
                <h3>No Vehicle Types Defined</h3>
                <p style="color: var(--text-muted); margin-bottom: 1.5rem;">Get started by adding your first vehicle type</p>
                <button class="btn btn-primary" onclick="openAddModal()">Add Vehicle Type</button>
            </div>
        </div>
    <?php else: ?>
        <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 1.5rem;">
            <?php foreach ($types as $type): ?>
                <div class="card vehicle-type-card" style="<?php echo !$type['is_active'] ? 'opacity: 0.6;' : ''; ?>">
                    <div class="card-body">
                        <div style="display: flex; justify-content: space-between; align-items: start;">
                            <div class="vehicle-icon"><?php echo htmlspecialchars($type['icon']); ?></div>
                            <div>
                                <?php if ($type['is_active']): ?>
                                    <span class="badge badge-success">Active</span>
                                <?php else: ?>
                                    <span class="badge badge-secondary">Inactive</span>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <h3 style="font-size: 1.25rem; font-weight: 600; margin-bottom: 0.5rem;">
                            <?php echo htmlspecialchars($type['type_name']); ?>
                        </h3>
                        
                        <?php if ($type['description']): ?>
                            <p style="color: var(--text-muted); font-size: 0.875rem; margin-bottom: 1rem;">
                                <?php echo htmlspecialchars($type['description']); ?>
                            </p>
                        <?php endif; ?>
                        
                        <div style="display: grid; gap: 0.5rem; font-size: 0.875rem; margin-bottom: 1rem;">
                            <?php if ($type['requires_license']): ?>
                                <div style="display: flex; align-items: center; gap: 0.5rem;">
                                    <span>📋</span>
                                    <span>Requires: <?php echo htmlspecialchars($type['license_class'] ?? 'License'); ?></span>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($type['max_capacity']): ?>
                                <div style="display: flex; align-items: center; gap: 0.5rem;">
                                    <span>👥</span>
                                    <span>Capacity: <?php echo htmlspecialchars($type['max_capacity']); ?> people</span>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($type['fuel_type']): ?>
                                <div style="display: flex; align-items: center; gap: 0.5rem;">
                                    <span>⛽</span>
                                    <span><?php echo htmlspecialchars($type['fuel_type']); ?></span>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($type['assignment_count'] > 0): ?>
                                <div style="display: flex; align-items: center; gap: 0.5rem;">
                                    <span>🔗</span>
                                    <span><?php echo $type['assignment_count']; ?> assignment<?php echo $type['assignment_count'] != 1 ? 's' : ''; ?></span>
                                </div>
                            <?php endif; ?>
                        </div>
                        
                        <div style="display: flex; gap: 0.5rem; border-top: 1px solid var(--border-color); padding-top: 1rem;">
                            <button class="btn btn-sm btn-primary" onclick='editType(<?php echo json_encode($type); ?>)'>
                                Edit
                            </button>
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this vehicle type?');">
                                <input type="hidden" name="type_id" value="<?php echo $type['id']; ?>">
                                <button type="submit" name="delete_type" class="btn btn-sm btn-danger" 
                                        <?php echo $type['assignment_count'] > 0 ? 'title="Cannot delete - in use"' : ''; ?>>
                                    Delete
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
</div>

<!-- Add/Edit Modal -->
<div id="typeModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3 id="modalTitle" class="modal-title">Add Vehicle Type</h3>
            <button class="modal-close" onclick="closeModal()">&times;</button>
        </div>
        <form method="POST" id="typeForm">
            <div class="modal-body">
                <input type="hidden" name="type_id" id="type_id">
                
                <div class="form-group">
                    <label class="form-label">Vehicle Type Name *</label>
                    <input type="text" name="type_name" id="type_name" class="form-control" required
                           placeholder="e.g., Sedan, Truck, Van">
                </div>

                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea name="description" id="description" class="form-control" rows="3"
                              placeholder="Brief description of this vehicle type"></textarea>
                </div>

                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                    <div class="form-group">
                        <label class="form-label">Icon Emoji</label>
                        <input type="text" name="icon" id="icon" class="form-control" 
                               value="🚗" placeholder="🚗">
                        <small class="form-text">Use an emoji like 🚗 🚙 🚚 🏍️</small>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Display Order</label>
                        <input type="number" name="display_order" id="display_order" class="form-control" 
                               value="0" min="0">
                    </div>
                </div>

                <div class="form-group">
                    <label style="display: flex; align-items: center; gap: 0.5rem;">
                        <input type="checkbox" name="requires_license" id="requires_license" value="1">
                        <span>Requires Driver's License</span>
                    </label>
                </div>

                <div class="form-group" id="license_group" style="display: none;">
                    <label class="form-label">License Class Required</label>
                    <input type="text" name="license_class" id="license_class" class="form-control"
                           placeholder="e.g., Class C, CDL Class A, Forklift Cert">
                </div>

                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                    <div class="form-group">
                        <label class="form-label">Max Capacity</label>
                        <input type="number" name="max_capacity" id="max_capacity" class="form-control"
                               placeholder="Number of passengers" min="1">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Fuel Type</label>
                        <select name="fuel_type" id="fuel_type" class="form-control">
                            <option value="">Select...</option>
                            <option value="Gasoline">Gasoline</option>
                            <option value="Diesel">Diesel</option>
                            <option value="Electric">Electric</option>
                            <option value="Hybrid">Hybrid</option>
                            <option value="Propane">Propane</option>
                            <option value="Other">Other</option>
                        </select>
                    </div>
                </div>

                <div class="form-group" id="active_group" style="display: none;">
                    <label style="display: flex; align-items: center; gap: 0.5rem;">
                        <input type="checkbox" name="is_active" id="is_active" value="1" checked>
                        <span>Active</span>
                    </label>
                </div>
            </div>
            <div class="modal-footer">
                <button type="submit" name="add_type" id="submitBtn" class="btn btn-primary">Add Vehicle Type</button>
                <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            </div>
        </form>
    </div>
</div>

<script>
// Toggle license class field
document.getElementById('requires_license').addEventListener('change', function() {
    document.getElementById('license_group').style.display = this.checked ? 'block' : 'none';
});

function openAddModal() {
    document.getElementById('modalTitle').textContent = 'Add Vehicle Type';
    document.getElementById('typeForm').reset();
    document.getElementById('type_id').value = '';
    document.getElementById('submitBtn').name = 'add_type';
    document.getElementById('submitBtn').textContent = 'Add Vehicle Type';
    document.getElementById('active_group').style.display = 'none';
    document.getElementById('license_group').style.display = 'none';
    document.getElementById('typeModal').classList.add('active');
}

function editType(type) {
    document.getElementById('modalTitle').textContent = 'Edit Vehicle Type';
    document.getElementById('type_id').value = type.id;
    document.getElementById('type_name').value = type.type_name;
    document.getElementById('description').value = type.description || '';
    document.getElementById('icon').value = type.icon || '🚗';
    document.getElementById('display_order').value = type.display_order || 0;
    document.getElementById('requires_license').checked = type.requires_license == 1;
    document.getElementById('license_class').value = type.license_class || '';
    document.getElementById('max_capacity').value = type.max_capacity || '';
    document.getElementById('fuel_type').value = type.fuel_type || '';
    document.getElementById('is_active').checked = type.is_active == 1;
    
    document.getElementById('license_group').style.display = type.requires_license ? 'block' : 'none';
    document.getElementById('active_group').style.display = 'block';
    document.getElementById('submitBtn').name = 'update_type';
    document.getElementById('submitBtn').textContent = 'Update Vehicle Type';
    
    document.getElementById('typeModal').classList.add('active');
}

function closeModal() {
    document.getElementById('typeModal').classList.remove('active');
}

// Close modal on backdrop click
document.getElementById('typeModal').addEventListener('click', function(e) {
    if (e.target === this) {
        closeModal();
    }
});

// Close modal on Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeModal();
    }
});
</script>

<?php include '../templates/footer.php'; ?>
