<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$error = '';
$success = '';

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            switch ($_POST['action']) {
                case 'add':
                    $password_hash = password_hash($_POST['password'], PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare("
                        INSERT INTO users (username, email, password, full_name, role, is_active)
                        VALUES (?, ?, ?, ?, ?, 1)
                    ");
                    $stmt->execute([
                        $_POST['username'],
                        $_POST['email'],
                        $password_hash,
                        $_POST['full_name'],
                        $_POST['role']
                    ]);
                    $success = 'User created successfully!';
                    break;
                
                case 'update':
                    if (!empty($_POST['password'])) {
                        $password_hash = password_hash($_POST['password'], PASSWORD_DEFAULT);
                        $stmt = $pdo->prepare("
                            UPDATE users 
                            SET username = ?, email = ?, password = ?, full_name = ?, role = ?, is_active = ?
                            WHERE id = ?
                        ");
                        $stmt->execute([
                            $_POST['username'],
                            $_POST['email'],
                            $password_hash,
                            $_POST['full_name'],
                            $_POST['role'],
                            isset($_POST['is_active']) ? 1 : 0,
                            $_POST['user_id']
                        ]);
                    } else {
                        $stmt = $pdo->prepare("
                            UPDATE users 
                            SET username = ?, email = ?, full_name = ?, role = ?, is_active = ?
                            WHERE id = ?
                        ");
                        $stmt->execute([
                            $_POST['username'],
                            $_POST['email'],
                            $_POST['full_name'],
                            $_POST['role'],
                            isset($_POST['is_active']) ? 1 : 0,
                            $_POST['user_id']
                        ]);
                    }
                    $success = 'User updated successfully!';
                    break;
                
                case 'delete':
                    // Don't allow deleting yourself
                    if ($_POST['user_id'] == $_SESSION['user_id']) {
                        $error = 'You cannot delete your own account!';
                    } else {
                        $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
                        $stmt->execute([$_POST['user_id']]);
                        $success = 'User deleted successfully!';
                    }
                    break;
            }
        } catch (PDOException $e) {
            $error = 'Database error: ' . $e->getMessage();
        }
    }
}

// Fetch all users
$stmt = $pdo->query("SELECT * FROM users ORDER BY created_at DESC");
$users = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Management - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">👥 User Management</h1>
                <p class="page-subtitle">Manage system users and permissions</p>
            </div>
            <button onclick="window.app.openModal('add-user-modal')" class="btn btn-primary">
                ➕ Add User
            </button>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <span>✅</span>
                <span><?php echo sanitizeOutput($success); ?></span>
            </div>
        <?php endif; ?>

        <!-- Users Table -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">System Users (<?php echo count($users); ?>)</h2>
            </div>

            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Username</th>
                            <th>Full Name</th>
                            <th>Email</th>
                            <th>Role</th>
                            <th>Last Login</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                            <tr>
                                <td><strong><?php echo sanitizeOutput($user['username']); ?></strong></td>
                                <td><?php echo sanitizeOutput($user['full_name']); ?></td>
                                <td><?php echo sanitizeOutput($user['email']); ?></td>
                                <td>
                                    <span class="badge" style="background: rgba(217, 119, 6, 0.2); color: var(--primary);">
                                        <?php echo strtoupper($user['role']); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php echo $user['last_login'] ? date('d/m/Y H:i', strtotime($user['last_login'])) : 'Never'; ?>
                                </td>
                                <td>
                                    <?php if ($user['is_active']): ?>
                                        <span class="badge badge-success">Active</span>
                                    <?php else: ?>
                                        <span class="badge" style="background: rgba(148, 163, 184, 0.2); color: var(--text-muted);">Inactive</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="btn-group" style="gap: 0.5rem;">
                                        <button onclick="editUser(<?php echo htmlspecialchars(json_encode($user)); ?>)" 
                                                class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                            ✏️ Edit
                                        </button>
                                        <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                            <form method="POST" style="display: inline;" 
                                                  onsubmit="return confirm('Are you sure you want to delete this user?');">
                                                <input type="hidden" name="action" value="delete">
                                                <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                <button type="submit" class="btn btn-danger" style="padding: 0.5rem 1rem;">
                                                    🗑️ Delete
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Add/Edit User Modal -->
    <div id="add-user-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 class="modal-title" id="modal-title">Add User</h2>
                <button type="button" class="modal-close" onclick="window.app.closeModal('add-user-modal')">&times;</button>
            </div>

            <form method="POST" id="user-form">
                <input type="hidden" name="action" id="form-action" value="add">
                <input type="hidden" name="user_id" id="user-id">

                <div class="form-group">
                    <label class="form-label required" for="username">Username</label>
                    <input type="text" id="username" name="username" class="form-control" required>
                </div>

                <div class="form-group">
                    <label class="form-label required" for="full_name">Full Name</label>
                    <input type="text" id="full_name" name="full_name" class="form-control" required>
                </div>

                <div class="form-group">
                    <label class="form-label required" for="email">Email</label>
                    <input type="email" id="email" name="email" class="form-control" required>
                </div>

                <div class="form-group">
                    <label class="form-label required" for="role">Role</label>
                    <select id="role" name="role" class="form-select" required>
                        <option value="viewer">Viewer (Read-only)</option>
                        <option value="assessor">Assessor (Create & Edit)</option>
                        <option value="admin">Administrator (Full Access)</option>
                    </select>
                </div>

                <div class="form-group">
                    <label class="form-label required" for="password">Password</label>
                    <input type="password" id="password" name="password" class="form-control" required>
                    <small style="color: var(--text-muted); display: block; margin-top: 0.25rem;">
                        Leave blank when editing to keep current password
                    </small>
                </div>

                <div class="form-group" id="active-group" style="display: none;">
                    <div style="display: flex; align-items: center; gap: 0.5rem;">
                        <input type="checkbox" id="is_active" name="is_active" checked style="width: auto;">
                        <label for="is_active" style="margin: 0; font-weight: 600;">Active User</label>
                    </div>
                </div>

                <div class="btn-group" style="margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">💾 Save User</button>
                    <button type="button" class="btn btn-secondary" onclick="window.app.closeModal('add-user-modal')">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <script src="js/app.js"></script>
    <script>
        function editUser(user) {
            document.getElementById('modal-title').textContent = 'Edit User';
            document.getElementById('form-action').value = 'update';
            document.getElementById('user-id').value = user.id;
            
            document.getElementById('username').value = user.username;
            document.getElementById('full_name').value = user.full_name;
            document.getElementById('email').value = user.email;
            document.getElementById('role').value = user.role;
            document.getElementById('password').value = '';
            document.getElementById('password').required = false;
            document.getElementById('is_active').checked = user.is_active == 1;
            document.getElementById('active-group').style.display = 'block';
            
            window.app.openModal('add-user-modal');
        }

        // Reset form when opening for add
        document.querySelector('[onclick*="add-user-modal"]').addEventListener('click', function() {
            document.getElementById('modal-title').textContent = 'Add User';
            document.getElementById('user-form').reset();
            document.getElementById('form-action').value = 'add';
            document.getElementById('user-id').value = '';
            document.getElementById('password').required = true;
            document.getElementById('active-group').style.display = 'none';
        });
    </script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
