<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$success = '';
$error = '';

// Handle theme preference save
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['save_default_theme'])) {
            $defaultTheme = $_POST['default_theme'] ?? 'light';
            
            // Save to system settings
            $stmt = $pdo->prepare("
                INSERT INTO system_settings (setting_key, setting_value, setting_type, description) 
                VALUES ('default_theme', ?, 'text', 'Default theme for new users')
                ON DUPLICATE KEY UPDATE setting_value = ?
            ");
            $stmt->execute([$defaultTheme, $defaultTheme]);
            
            $success = "Default theme saved successfully!";
        }
        
        if (isset($_POST['save_theme_options'])) {
            $allowUserTheme = isset($_POST['allow_user_theme']) ? '1' : '0';
            $autoDetectTheme = isset($_POST['auto_detect_theme']) ? '1' : '0';
            
            // Save settings
            $stmt = $pdo->prepare("
                INSERT INTO system_settings (setting_key, setting_value, setting_type, description) 
                VALUES 
                    ('allow_user_theme', ?, 'boolean', 'Allow users to change theme'),
                    ('auto_detect_theme', ?, 'boolean', 'Auto-detect system theme preference')
                ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value)
            ");
            $stmt->execute([$allowUserTheme, $autoDetectTheme]);
            
            $success = "Theme options saved successfully!";
        }
    } catch (PDOException $e) {
        $error = "Error saving settings: " . $e->getMessage();
    }
}

// Get current settings
try {
    $stmt = $pdo->query("SELECT setting_key, setting_value FROM system_settings WHERE setting_key IN ('default_theme', 'allow_user_theme', 'auto_detect_theme')");
    $settings = [];
    while ($row = $stmt->fetch()) {
        $settings[$row['setting_key']] = $row['setting_value'];
    }
} catch (PDOException $e) {
    $settings = [
        'default_theme' => 'light',
        'allow_user_theme' => '1',
        'auto_detect_theme' => '1'
    ];
}

$pageTitle = "Theme Settings";
include '../templates/header.php';
?>

<div class="container">
    <div class="page-header">
        <div>
            <h1 class="page-title">🎨 Theme Settings</h1>
            <p class="page-subtitle">Configure light and dark theme preferences</p>
        </div>
        <a href="admin-panel.php" class="btn btn-secondary">← Back to Admin</a>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <div class="grid grid-cols-1 grid-cols-2-lg gap-4">
        <!-- Theme Preview -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📱 Current Theme</h2>
            </div>
            <div class="card-body">
                <div id="theme-preview" class="text-center">
                    <div class="mb-4">
                        <div id="theme-icon" style="font-size: 4rem; margin-bottom: 1rem;">🌞</div>
                        <h3 id="theme-name" style="font-size: 1.5rem; font-weight: 600; margin-bottom: 0.5rem;">Light Theme</h3>
                        <p id="theme-description" style="color: var(--text-muted);">Bright and clear interface</p>
                    </div>
                    
                    <button onclick="toggleTheme()" class="btn btn-primary" style="width: 100%; max-width: 300px;">
                        <span id="toggle-text">Switch to Dark Theme</span>
                    </button>
                </div>
                
                <div class="mt-4 p-3" style="background: var(--bg-surface); border-radius: 8px; border: 1px solid var(--border-default);">
                    <h4 style="font-size: 0.875rem; font-weight: 600; margin-bottom: 0.5rem;">Preview</h4>
                    <div class="flex gap-2 mb-2">
                        <div class="badge badge-success">Success</div>
                        <div class="badge badge-warning">Warning</div>
                        <div class="badge badge-danger">Danger</div>
                    </div>
                    <button class="btn btn-sm btn-primary">Primary Button</button>
                    <button class="btn btn-sm btn-secondary">Secondary Button</button>
                </div>
            </div>
        </div>

        <!-- Default Theme Setting -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">⚙️ Default Theme</h2>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="form-group">
                        <label class="form-label">System Default Theme</label>
                        <select name="default_theme" class="form-control">
                            <option value="light" <?php echo ($settings['default_theme'] ?? 'light') === 'light' ? 'selected' : ''; ?>>
                                ☀️ Light Theme
                            </option>
                            <option value="dark" <?php echo ($settings['default_theme'] ?? 'light') === 'dark' ? 'selected' : ''; ?>>
                                🌙 Dark Theme
                            </option>
                            <option value="auto" <?php echo ($settings['default_theme'] ?? 'light') === 'auto' ? 'selected' : ''; ?>>
                                🔄 Auto (System Preference)
                            </option>
                        </select>
                        <small class="form-text">Default theme for new users and logged-out visitors</small>
                    </div>
                    
                    <button type="submit" name="save_default_theme" class="btn btn-primary">
                        Save Default Theme
                    </button>
                </form>
            </div>
        </div>

        <!-- Theme Options -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">🎯 Theme Options</h2>
            </div>
            <div class="card-body">
                <form method="POST">
                    <div class="form-group">
                        <label class="form-label">
                            <input type="checkbox" name="allow_user_theme" value="1" 
                                   <?php echo ($settings['allow_user_theme'] ?? '1') === '1' ? 'checked' : ''; ?>>
                            Allow users to change theme
                        </label>
                        <small class="form-text">Users can switch between light and dark themes</small>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">
                            <input type="checkbox" name="auto_detect_theme" value="1"
                                   <?php echo ($settings['auto_detect_theme'] ?? '1') === '1' ? 'checked' : ''; ?>>
                            Auto-detect system theme
                        </label>
                        <small class="form-text">Automatically use user's system theme preference</small>
                    </div>
                    
                    <button type="submit" name="save_theme_options" class="btn btn-primary">
                        Save Options
                    </button>
                </form>
            </div>
        </div>

        <!-- Theme Information -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">ℹ️ Theme Information</h2>
            </div>
            <div class="card-body">
                <h4 style="font-size: 0.95rem; font-weight: 600; margin-bottom: 0.75rem;">Features</h4>
                <ul style="margin: 0; padding-left: 1.5rem; color: var(--text-secondary);">
                    <li>Automatic theme persistence</li>
                    <li>Smooth transitions between themes</li>
                    <li>Respects system preferences</li>
                    <li>Works across all pages</li>
                    <li>Mobile-friendly toggle button</li>
                </ul>
                
                <h4 style="font-size: 0.95rem; font-weight: 600; margin: 1rem 0 0.75rem;">How It Works</h4>
                <p style="color: var(--text-secondary); font-size: 0.875rem;">
                    The theme switcher uses localStorage to remember user preferences. 
                    If no preference is set, it automatically detects the system theme. 
                    Users can toggle themes using the floating button on any page.
                </p>
                
                <h4 style="font-size: 0.95rem; font-weight: 600; margin: 1rem 0 0.75rem;">Accessibility</h4>
                <p style="color: var(--text-secondary); font-size: 0.875rem;">
                    Dark theme reduces eye strain in low-light environments and 
                    conserves battery on OLED displays. Both themes are fully 
                    accessible and meet WCAG contrast requirements.
                </p>
            </div>
        </div>
    </div>

    <!-- Usage Guide -->
    <div class="card mt-4">
        <div class="card-header">
            <h2 class="card-title">📖 User Guide</h2>
        </div>
        <div class="card-body">
            <div class="grid grid-cols-1 grid-cols-3-md gap-4">
                <div>
                    <h4 style="font-size: 0.95rem; font-weight: 600; margin-bottom: 0.5rem;">For Users</h4>
                    <p style="color: var(--text-secondary); font-size: 0.875rem;">
                        Click the floating button (🌙/☀️) in the bottom-right corner to switch themes. 
                        Your preference is automatically saved.
                    </p>
                </div>
                
                <div>
                    <h4 style="font-size: 0.95rem; font-weight: 600; margin-bottom: 0.5rem;">System Preference</h4>
                    <p style="color: var(--text-secondary); font-size: 0.875rem;">
                        If you haven't set a preference, the system automatically uses your 
                        operating system's theme setting (Windows/Mac/Linux/Mobile).
                    </p>
                </div>
                
                <div>
                    <h4 style="font-size: 0.95rem; font-weight: 600; margin-bottom: 0.5rem;">Keyboard Shortcut</h4>
                    <p style="color: var(--text-secondary); font-size: 0.875rem;">
                        Press <code>Ctrl+Shift+L</code> (or <code>Cmd+Shift+L</code> on Mac) 
                        to quickly toggle between themes.
                    </p>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
#theme-preview {
    padding: 2rem;
}

@media (max-width: 768px) {
    #theme-preview {
        padding: 1rem;
    }
}
</style>

<script>
// Update preview when theme changes
function updatePreview() {
    const isDark = isDarkTheme();
    const icon = document.getElementById('theme-icon');
    const name = document.getElementById('theme-name');
    const description = document.getElementById('theme-description');
    const toggleText = document.getElementById('toggle-text');
    
    if (isDark) {
        icon.textContent = '🌙';
        name.textContent = 'Dark Theme';
        description.textContent = 'Easy on the eyes in low light';
        toggleText.textContent = 'Switch to Light Theme';
    } else {
        icon.textContent = '🌞';
        name.textContent = 'Light Theme';
        description.textContent = 'Bright and clear interface';
        toggleText.textContent = 'Switch to Dark Theme';
    }
}

// Listen for theme changes
window.addEventListener('themeChange', updatePreview);

// Update on load
updatePreview();

// Keyboard shortcut
document.addEventListener('keydown', (e) => {
    if ((e.ctrlKey || e.metaKey) && e.shiftKey && e.key === 'L') {
        e.preventDefault();
        toggleTheme();
    }
});
</script>

<?php include '../templates/footer.php'; ?>
