<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$success = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['action'])) {
            switch ($_POST['action']) {
                case 'save_colors':
                    // Save color scheme to database
                    $colors = [
                        'primary' => $_POST['primary_color'],
                        'primary_dark' => $_POST['primary_dark_color'],
                        'primary_light' => $_POST['primary_light_color'],
                        'secondary' => $_POST['secondary_color'],
                        'success' => $_POST['success_color'],
                        'warning' => $_POST['warning_color'],
                        'danger' => $_POST['danger_color'],
                        'info' => $_POST['info_color']
                    ];
                    
                    // Store in database
                    $stmt = $pdo->prepare("
                        INSERT INTO system_settings (setting_key, setting_value, updated_by)
                        VALUES ('color_scheme', ?, ?)
                        ON DUPLICATE KEY UPDATE setting_value = ?, updated_by = ?, updated_at = CURRENT_TIMESTAMP
                    ");
                    $colorJson = json_encode($colors);
                    $stmt->execute([$colorJson, $_SESSION['user_id'], $colorJson, $_SESSION['user_id']]);
                    
                    // Generate CSS file
                    generateCustomCSS($colors);
                    
                    $success = 'Color scheme saved successfully!';
                    break;
                
                case 'reset_colors':
                    // Delete custom colors
                    $stmt = $pdo->prepare("DELETE FROM system_settings WHERE setting_key = 'color_scheme'");
                    $stmt->execute();
                    
                    // Delete custom CSS file
                    $customCSSPath = __DIR__ . '/css/custom-theme.css';
                    if (file_exists($customCSSPath)) {
                        unlink($customCSSPath);
                    }
                    
                    $success = 'Color scheme reset to default!';
                    break;
                
                case 'upload_logo':
                    if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                        $allowedTypes = ['image/png', 'image/jpeg', 'image/svg+xml', 'image/webp'];
                        $fileType = $_FILES['logo']['type'];
                        
                        if (!in_array($fileType, $allowedTypes)) {
                            throw new Exception('Invalid file type. Please upload PNG, JPG, SVG, or WebP.');
                        }
                        
                        if ($_FILES['logo']['size'] > 2 * 1024 * 1024) { // 2MB
                            throw new Exception('File too large. Maximum size is 2MB.');
                        }
                        
                        // Generate filename
                        $extension = pathinfo($_FILES['logo']['name'], PATHINFO_EXTENSION);
                        $filename = 'logo.' . $extension;
                        $uploadPath = __DIR__ . '/uploads/branding/' . $filename;
                        
                        // Create directory if needed
                        if (!file_exists(__DIR__ . '/uploads/branding')) {
                            mkdir(__DIR__ . '/uploads/branding', 0755, true);
                        }
                        
                        // Delete old logo files
                        $oldLogos = glob(__DIR__ . '/uploads/branding/logo.*');
                        foreach ($oldLogos as $oldLogo) {
                            unlink($oldLogo);
                        }
                        
                        // Upload new logo
                        if (move_uploaded_file($_FILES['logo']['tmp_name'], $uploadPath)) {
                            // Save to database
                            $stmt = $pdo->prepare("
                                INSERT INTO system_settings (setting_key, setting_value, updated_by)
                                VALUES ('logo_filename', ?, ?)
                                ON DUPLICATE KEY UPDATE setting_value = ?, updated_by = ?, updated_at = CURRENT_TIMESTAMP
                            ");
                            $stmt->execute([$filename, $_SESSION['user_id'], $filename, $_SESSION['user_id']]);
                            
                            $success = 'Logo uploaded successfully!';
                        } else {
                            throw new Exception('Failed to upload logo.');
                        }
                    }
                    break;
                
                case 'delete_logo':
                    // Delete logo files
                    $oldLogos = glob(__DIR__ . '/uploads/branding/logo.*');
                    foreach ($oldLogos as $oldLogo) {
                        unlink($oldLogo);
                    }
                    
                    // Delete from database
                    $stmt = $pdo->prepare("DELETE FROM system_settings WHERE setting_key = 'logo_filename'");
                    $stmt->execute();
                    
                    $success = 'Logo deleted successfully!';
                    break;
                
                case 'save_system_name':
                    $systemName = $_POST['system_name'];
                    $stmt = $pdo->prepare("
                        INSERT INTO system_settings (setting_key, setting_value, updated_by)
                        VALUES ('system_name', ?, ?)
                        ON DUPLICATE KEY UPDATE setting_value = ?, updated_by = ?, updated_at = CURRENT_TIMESTAMP
                    ");
                    $stmt->execute([$systemName, $_SESSION['user_id'], $systemName, $_SESSION['user_id']]);
                    $success = 'System name saved successfully!';
                    break;
            }
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Fetch current settings
$currentColors = getSystemColors($pdo);
$currentLogo = getSystemLogo($pdo);
$systemName = getSystemName($pdo);

// Helper functions
function getSystemColors($pdo) {
    $stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = 'color_scheme'");
    $stmt->execute();
    $result = $stmt->fetch();
    
    if ($result) {
        return json_decode($result['setting_value'], true);
    }
    
    // Default colors
    return [
        'primary' => '#D97706',
        'primary_dark' => '#B45309',
        'primary_light' => '#FCD34D',
        'secondary' => '#1E40AF',
        'success' => '#059669',
        'warning' => '#F59E0B',
        'danger' => '#DC2626',
        'info' => '#0891B2'
    ];
}

function getSystemLogo($pdo) {
    $stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = 'logo_filename'");
    $stmt->execute();
    $result = $stmt->fetch();
    
    if ($result && file_exists(__DIR__ . '/uploads/branding/' . $result['setting_value'])) {
        return 'uploads/branding/' . $result['setting_value'];
    }
    
    return null;
}

function getSystemName($pdo) {
    $stmt = $pdo->prepare("SELECT setting_value FROM system_settings WHERE setting_key = 'system_name'");
    $stmt->execute();
    $result = $stmt->fetch();
    return $result ? $result['setting_value'] : 'Risk Assessment';
}

function generateCustomCSS($colors) {
    $css = ":root {\n";
    $css .= "    --primary: {$colors['primary']};\n";
    $css .= "    --primary-dark: {$colors['primary_dark']};\n";
    $css .= "    --primary-light: {$colors['primary_light']};\n";
    $css .= "    --secondary: {$colors['secondary']};\n";
    $css .= "    --success: {$colors['success']};\n";
    $css .= "    --warning: {$colors['warning']};\n";
    $css .= "    --danger: {$colors['danger']};\n";
    $css .= "    --info: {$colors['info']};\n";
    $css .= "}\n";
    
    $cssPath = __DIR__ . '/css/custom-theme.css';
    file_put_contents($cssPath, $css);
}

include '../templates/header.php';
?>

<div class="container">
    <div class="page-header">
        <div>
            <h1 class="page-title">⚙️ System Customization</h1>
            <p class="page-subtitle">Customize colors, branding, and appearance</p>
        </div>
        <div class="btn-group">
            <a href="admin-panel.php" class="btn btn-secondary">← Back to Admin</a>
        </div>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success">
            ✅ <?php echo sanitizeOutput($success); ?>
        </div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger">
            ❌ <?php echo sanitizeOutput($error); ?>
        </div>
    <?php endif; ?>

    <!-- System Name -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">System Name</h2>
        </div>
        <div class="card-body">
            <form method="POST" class="form">
                <input type="hidden" name="action" value="save_system_name">
                
                <div class="form-group">
                    <label class="form-label">System Name</label>
                    <input type="text" name="system_name" class="form-control" 
                           value="<?php echo sanitizeOutput($systemName); ?>" required>
                    <small style="color: var(--text-muted);">
                        This name appears in the header and browser title
                    </small>
                </div>
                
                <button type="submit" class="btn btn-primary">💾 Save System Name</button>
            </form>
        </div>
    </div>

    <!-- Logo Upload -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">System Logo</h2>
        </div>
        <div class="card-body">
            <?php if ($currentLogo): ?>
                <div style="margin-bottom: 2rem; padding: 2rem; background: var(--bg-secondary); border-radius: 8px; text-align: center;">
                    <p style="margin-bottom: 1rem; color: var(--text-muted);">Current Logo:</p>
                    <img src="<?php echo $currentLogo; ?>" alt="Current Logo" style="max-width: 300px; max-height: 100px;">
                </div>
            <?php endif; ?>

            <form method="POST" enctype="multipart/form-data" class="form">
                <input type="hidden" name="action" value="upload_logo">
                
                <div class="form-group">
                    <label class="form-label">Upload New Logo</label>
                    <input type="file" name="logo" class="form-control" accept="image/png,image/jpeg,image/svg+xml,image/webp" required>
                    <small style="color: var(--text-muted);">
                        Accepted formats: PNG, JPG, SVG, WebP | Max size: 2MB | Recommended: 200x50px
                    </small>
                </div>
                
                <div class="btn-group">
                    <button type="submit" class="btn btn-primary">📤 Upload Logo</button>
                    
                    <?php if ($currentLogo): ?>
                        <button type="submit" name="action" value="delete_logo" class="btn btn-danger"
                                onclick="return confirm('Delete the current logo?')">
                            🗑️ Delete Logo
                        </button>
                    <?php endif; ?>
                </div>
            </form>
        </div>
    </div>

    <!-- Color Scheme -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Color Scheme</h2>
        </div>
        <div class="card-body">
            <form method="POST" class="form" id="color-form">
                <input type="hidden" name="action" value="save_colors">
                
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1.5rem;">
                    <!-- Primary Color -->
                    <div class="form-group">
                        <label class="form-label">Primary Color</label>
                        <div style="display: flex; gap: 0.5rem; align-items: center;">
                            <input type="color" name="primary_color" class="form-control" 
                                   value="<?php echo $currentColors['primary']; ?>" 
                                   style="width: 80px; height: 50px; padding: 5px;">
                            <input type="text" class="form-control" 
                                   value="<?php echo $currentColors['primary']; ?>" 
                                   readonly style="flex: 1;">
                        </div>
                        <small style="color: var(--text-muted);">Main brand color, buttons, links</small>
                    </div>

                    <!-- Primary Dark -->
                    <div class="form-group">
                        <label class="form-label">Primary Dark</label>
                        <div style="display: flex; gap: 0.5rem; align-items: center;">
                            <input type="color" name="primary_dark_color" class="form-control" 
                                   value="<?php echo $currentColors['primary_dark']; ?>" 
                                   style="width: 80px; height: 50px; padding: 5px;">
                            <input type="text" class="form-control" 
                                   value="<?php echo $currentColors['primary_dark']; ?>" 
                                   readonly style="flex: 1;">
                        </div>
                        <small style="color: var(--text-muted);">Hover states, active items</small>
                    </div>

                    <!-- Primary Light -->
                    <div class="form-group">
                        <label class="form-label">Primary Light</label>
                        <div style="display: flex; gap: 0.5rem; align-items: center;">
                            <input type="color" name="primary_light_color" class="form-control" 
                                   value="<?php echo $currentColors['primary_light']; ?>" 
                                   style="width: 80px; height: 50px; padding: 5px;">
                            <input type="text" class="form-control" 
                                   value="<?php echo $currentColors['primary_light']; ?>" 
                                   readonly style="flex: 1;">
                        </div>
                        <small style="color: var(--text-muted);">Backgrounds, highlights</small>
                    </div>

                    <!-- Secondary -->
                    <div class="form-group">
                        <label class="form-label">Secondary Color</label>
                        <div style="display: flex; gap: 0.5rem; align-items: center;">
                            <input type="color" name="secondary_color" class="form-control" 
                                   value="<?php echo $currentColors['secondary']; ?>" 
                                   style="width: 80px; height: 50px; padding: 5px;">
                            <input type="text" class="form-control" 
                                   value="<?php echo $currentColors['secondary']; ?>" 
                                   readonly style="flex: 1;">
                        </div>
                        <small style="color: var(--text-muted);">Secondary buttons, accents</small>
                    </div>

                    <!-- Success -->
                    <div class="form-group">
                        <label class="form-label">Success Color</label>
                        <div style="display: flex; gap: 0.5rem; align-items: center;">
                            <input type="color" name="success_color" class="form-control" 
                                   value="<?php echo $currentColors['success']; ?>" 
                                   style="width: 80px; height: 50px; padding: 5px;">
                            <input type="text" class="form-control" 
                                   value="<?php echo $currentColors['success']; ?>" 
                                   readonly style="flex: 1;">
                        </div>
                        <small style="color: var(--text-muted);">Success messages, low risk</small>
                    </div>

                    <!-- Warning -->
                    <div class="form-group">
                        <label class="form-label">Warning Color</label>
                        <div style="display: flex; gap: 0.5rem; align-items: center;">
                            <input type="color" name="warning_color" class="form-control" 
                                   value="<?php echo $currentColors['warning']; ?>" 
                                   style="width: 80px; height: 50px; padding: 5px;">
                            <input type="text" class="form-control" 
                                   value="<?php echo $currentColors['warning']; ?>" 
                                   readonly style="flex: 1;">
                        </div>
                        <small style="color: var(--text-muted);">Warning messages, medium risk</small>
                    </div>

                    <!-- Danger -->
                    <div class="form-group">
                        <label class="form-label">Danger Color</label>
                        <div style="display: flex; gap: 0.5rem; align-items: center;">
                            <input type="color" name="danger_color" class="form-control" 
                                   value="<?php echo $currentColors['danger']; ?>" 
                                   style="width: 80px; height: 50px; padding: 5px;">
                            <input type="text" class="form-control" 
                                   value="<?php echo $currentColors['danger']; ?>" 
                                   readonly style="flex: 1;">
                        </div>
                        <small style="color: var(--text-muted);">Error messages, high risk</small>
                    </div>

                    <!-- Info -->
                    <div class="form-group">
                        <label class="form-label">Info Color</label>
                        <div style="display: flex; gap: 0.5rem; align-items: center;">
                            <input type="color" name="info_color" class="form-control" 
                                   value="<?php echo $currentColors['info']; ?>" 
                                   style="width: 80px; height: 50px; padding: 5px;">
                            <input type="text" class="form-control" 
                                   value="<?php echo $currentColors['info']; ?>" 
                                   readonly style="flex: 1;">
                        </div>
                        <small style="color: var(--text-muted);">Info messages, badges</small>
                    </div>
                </div>

                <!-- Color Preview -->
                <div style="margin-top: 2rem; padding: 2rem; background: var(--bg-secondary); border-radius: 8px;">
                    <h3 style="margin-bottom: 1rem;">Preview:</h3>
                    <div style="display: flex; gap: 0.5rem; flex-wrap: wrap;">
                        <button type="button" class="btn btn-primary" id="preview-primary">Primary Button</button>
                        <button type="button" class="btn btn-secondary" id="preview-secondary">Secondary Button</button>
                        <span class="badge badge-success" id="preview-success">Success</span>
                        <span class="badge badge-warning" id="preview-warning">Warning</span>
                        <span class="badge badge-danger" id="preview-danger">Danger</span>
                        <span class="badge badge-info" id="preview-info">Info</span>
                    </div>
                </div>

                <div class="btn-group" style="margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">💾 Save Color Scheme</button>
                    <button type="submit" name="action" value="reset_colors" class="btn btn-secondary"
                            onclick="return confirm('Reset to default colors?')">
                        🔄 Reset to Default
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Preset Color Schemes -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">🎨 Preset Color Schemes</h2>
        </div>
        <div class="card-body">
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem;">
                <button class="preset-scheme" onclick="applyPreset('default')">
                    <div style="display: flex; gap: 5px; margin-bottom: 0.5rem;">
                        <div style="width: 40px; height: 40px; background: #D97706; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #1E40AF; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #059669; border-radius: 4px;"></div>
                    </div>
                    <strong>Default (Orange)</strong>
                </button>

                <button class="preset-scheme" onclick="applyPreset('blue')">
                    <div style="display: flex; gap: 5px; margin-bottom: 0.5rem;">
                        <div style="width: 40px; height: 40px; background: #3B82F6; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #1E40AF; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #0EA5E9; border-radius: 4px;"></div>
                    </div>
                    <strong>Blue</strong>
                </button>

                <button class="preset-scheme" onclick="applyPreset('green')">
                    <div style="display: flex; gap: 5px; margin-bottom: 0.5rem;">
                        <div style="width: 40px; height: 40px; background: #10B981; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #059669; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #34D399; border-radius: 4px;"></div>
                    </div>
                    <strong>Green</strong>
                </button>

                <button class="preset-scheme" onclick="applyPreset('purple')">
                    <div style="display: flex; gap: 5px; margin-bottom: 0.5rem;">
                        <div style="width: 40px; height: 40px; background: #8B5CF6; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #7C3AED; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #A78BFA; border-radius: 4px;"></div>
                    </div>
                    <strong>Purple</strong>
                </button>

                <button class="preset-scheme" onclick="applyPreset('red')">
                    <div style="display: flex; gap: 5px; margin-bottom: 0.5rem;">
                        <div style="width: 40px; height: 40px; background: #EF4444; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #DC2626; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #F87171; border-radius: 4px;"></div>
                    </div>
                    <strong>Red</strong>
                </button>

                <button class="preset-scheme" onclick="applyPreset('teal')">
                    <div style="display: flex; gap: 5px; margin-bottom: 0.5rem;">
                        <div style="width: 40px; height: 40px; background: #14B8A6; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #0D9488; border-radius: 4px;"></div>
                        <div style="width: 40px; height: 40px; background: #2DD4BF; border-radius: 4px;"></div>
                    </div>
                    <strong>Teal</strong>
                </button>
            </div>
        </div>
    </div>
</div>

<style>
    .preset-scheme {
        padding: 1rem;
        background: var(--bg-secondary);
        border: 2px solid var(--border-color);
        border-radius: 8px;
        cursor: pointer;
        transition: var(--transition);
        text-align: left;
    }

    .preset-scheme:hover {
        border-color: var(--primary);
        transform: translateY(-2px);
        box-shadow: var(--shadow);
    }

    .alert {
        padding: 1rem;
        border-radius: 8px;
        margin-bottom: 1.5rem;
    }

    .alert-success {
        background: #d4edda;
        color: #155724;
        border: 1px solid #c3e6cb;
    }

    .alert-danger {
        background: #f8d7da;
        color: #721c24;
        border: 1px solid #f5c6cb;
    }
</style>

<script>
    // Update color preview in real-time
    document.querySelectorAll('input[type="color"]').forEach(input => {
        input.addEventListener('input', function() {
            const textInput = this.nextElementSibling;
            textInput.value = this.value;
            updatePreview();
        });
    });

    function updatePreview() {
        const primary = document.querySelector('input[name="primary_color"]').value;
        const secondary = document.querySelector('input[name="secondary_color"]').value;
        const success = document.querySelector('input[name="success_color"]').value;
        const warning = document.querySelector('input[name="warning_color"]').value;
        const danger = document.querySelector('input[name="danger_color"]').value;
        const info = document.querySelector('input[name="info_color"]').value;

        document.getElementById('preview-primary').style.background = primary;
        document.getElementById('preview-secondary').style.background = secondary;
        document.getElementById('preview-success').style.background = success;
        document.getElementById('preview-warning').style.background = warning;
        document.getElementById('preview-danger').style.background = danger;
        document.getElementById('preview-info').style.background = info;
    }

    // Preset schemes
    const presets = {
        default: {
            primary: '#D97706',
            primary_dark: '#B45309',
            primary_light: '#FCD34D',
            secondary: '#1E40AF',
            success: '#059669',
            warning: '#F59E0B',
            danger: '#DC2626',
            info: '#0891B2'
        },
        blue: {
            primary: '#3B82F6',
            primary_dark: '#2563EB',
            primary_light: '#93C5FD',
            secondary: '#1E40AF',
            success: '#059669',
            warning: '#F59E0B',
            danger: '#DC2626',
            info: '#0EA5E9'
        },
        green: {
            primary: '#10B981',
            primary_dark: '#059669',
            primary_light: '#6EE7B7',
            secondary: '#14B8A6',
            success: '#10B981',
            warning: '#F59E0B',
            danger: '#DC2626',
            info: '#0891B2'
        },
        purple: {
            primary: '#8B5CF6',
            primary_dark: '#7C3AED',
            primary_light: '#C4B5FD',
            secondary: '#6366F1',
            success: '#059669',
            warning: '#F59E0B',
            danger: '#DC2626',
            info: '#0891B2'
        },
        red: {
            primary: '#EF4444',
            primary_dark: '#DC2626',
            primary_light: '#FCA5A5',
            secondary: '#F97316',
            success: '#059669',
            warning: '#F59E0B',
            danger: '#DC2626',
            info: '#0891B2'
        },
        teal: {
            primary: '#14B8A6',
            primary_dark: '#0D9488',
            primary_light: '#5EEAD4',
            secondary: '#06B6D4',
            success: '#059669',
            warning: '#F59E0B',
            danger: '#DC2626',
            info: '#0891B2'
        }
    };

    function applyPreset(preset) {
        const colors = presets[preset];
        document.querySelector('input[name="primary_color"]').value = colors.primary;
        document.querySelector('input[name="primary_dark_color"]').value = colors.primary_dark;
        document.querySelector('input[name="primary_light_color"]').value = colors.primary_light;
        document.querySelector('input[name="secondary_color"]').value = colors.secondary;
        document.querySelector('input[name="success_color"]').value = colors.success;
        document.querySelector('input[name="warning_color"]').value = colors.warning;
        document.querySelector('input[name="danger_color"]').value = colors.danger;
        document.querySelector('input[name="info_color"]').value = colors.info;

        // Update text fields
        document.querySelectorAll('input[type="color"]').forEach(input => {
            const textInput = input.nextElementSibling;
            textInput.value = input.value;
        });

        updatePreview();
    }

    // Initial preview update
    updatePreview();
</script>

<?php include '../templates/footer.php'; ?>
