<?php
/**
 * System Validation and Diagnostic Tool
 * Checks all files, links, and database integrity
 * Access: http://yoursite.com/system-check.php
 */

require_once '../includes/config.php';

// Only admins can access
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    die('Access denied. Admin only.');
}

$errors = [];
$warnings = [];
$passed = [];

?>
<!DOCTYPE html>
<html>
<head>
    <title>System Validation Check</title>
    <style>
        body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
        h1 { color: #333; border-bottom: 3px solid #667eea; padding-bottom: 10px; }
        h2 { color: #667eea; margin-top: 30px; }
        .check { padding: 10px; margin: 5px 0; border-left: 4px solid #ddd; background: #f9f9f9; }
        .pass { border-left-color: #059669; background: #f0fdf4; }
        .warn { border-left-color: #f59e0b; background: #fffbeb; }
        .fail { border-left-color: #dc2626; background: #fef2f2; }
        .icon { font-weight: bold; margin-right: 10px; }
        .summary { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; margin: 20px 0; }
        .stat { padding: 20px; text-align: center; border-radius: 8px; }
        .stat-pass { background: #f0fdf4; color: #059669; }
        .stat-warn { background: #fffbeb; color: #f59e0b; }
        .stat-fail { background: #fef2f2; color: #dc2626; }
        .stat-number { font-size: 2.5rem; font-weight: bold; }
        .stat-label { font-size: 0.9rem; text-transform: uppercase; margin-top: 5px; }
        table { width: 100%; border-collapse: collapse; margin: 20px 0; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #e5e7eb; }
        th { background: #f9fafb; font-weight: 600; }
        .badge { padding: 4px 8px; border-radius: 4px; font-size: 0.875rem; font-weight: 600; }
        .badge-success { background: #d1fae5; color: #065f46; }
        .badge-danger { background: #fee2e2; color: #991b1b; }
        .badge-warning { background: #fef3c7; color: #92400e; }
        code { background: #f3f4f6; padding: 2px 6px; border-radius: 3px; font-family: monospace; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 System Validation Check</h1>
        <p>Comprehensive system health and integrity check</p>

<?php

// ============================================================================
// CHECK 1: REQUIRED PHP FILES
// ============================================================================

echo "<h2>1. PHP Files Check</h2>";

$requiredFiles = [
    'public/admin-panel.php' => 'Admin Panel',
    'public/custom-fields.php' => 'Custom Fields Management',
    'public/vehicle-types.php' => 'Vehicle Types Management',
    'public/version-control.php' => 'Version Control',
    'public/add-location.php' => 'Add/Edit Location',
    'public/locations.php' => 'Locations List',
    'public/assessments.php' => 'Assessments List',
    'public/new-assessment.php' => 'New Assessment',
    'public/dashboard.php' => 'Dashboard',
    'public/users.php' => 'User Management',
    'public/reports.php' => 'Reports',
    'public/system-customization.php' => 'System Customization',
    'public/smtp-settings.php' => 'SMTP Settings',
    'public/permissions-management.php' => 'Permissions Management',
    'includes/config.php' => 'Configuration',
    'includes/version_control.php' => 'Version Control Library',
    'templates/header.php' => 'Header Template',
    'templates/footer.php' => 'Footer Template',
];

foreach ($requiredFiles as $file => $description) {
    $fullPath = __DIR__ . '/../' . $file;
    if (file_exists($fullPath)) {
        // Check for PHP syntax errors
        $output = [];
        $return = 0;
        exec("php -l " . escapeshellarg($fullPath) . " 2>&1", $output, $return);
        
        if ($return === 0) {
            $passed[] = "$description file exists and has valid syntax";
            echo "<div class='check pass'><span class='icon'>✓</span><strong>$description</strong> - <code>$file</code></div>";
        } else {
            $errors[] = "$description has syntax errors";
            echo "<div class='check fail'><span class='icon'>✗</span><strong>$description</strong> - Syntax Error in <code>$file</code></div>";
        }
    } else {
        $errors[] = "$description is missing";
        echo "<div class='check fail'><span class='icon'>✗</span><strong>$description</strong> - Missing: <code>$file</code></div>";
    }
}

// ============================================================================
// CHECK 2: DATABASE TABLES
// ============================================================================

echo "<h2>2. Database Tables Check</h2>";

$requiredTables = [
    'users' => 'User accounts',
    'locations' => 'Location data',
    'assessments' => 'Assessment records',
    'assessment_details' => 'Assessment detail items',
    'risk_categories' => 'Risk categories',
    'risk_items' => 'Risk items library',
    'system_settings' => 'System settings',
    'system_files' => 'Version control files',
    'file_versions' => 'File version history',
    'custom_fields' => 'Custom field definitions',
    'custom_field_values' => 'Custom field values',
    'vehicle_types' => 'Vehicle type definitions',
    'vehicle_assignments' => 'Vehicle assignments',
    'permissions' => 'Permission rules',
];

foreach ($requiredTables as $table => $description) {
    try {
        $stmt = $pdo->query("SHOW TABLES LIKE '$table'");
        if ($stmt->rowCount() > 0) {
            // Get row count
            $countStmt = $pdo->query("SELECT COUNT(*) as count FROM $table");
            $count = $countStmt->fetch()['count'];
            
            $passed[] = "Table $table exists ($count rows)";
            echo "<div class='check pass'><span class='icon'>✓</span><strong>$description</strong> - Table <code>$table</code> exists ($count rows)</div>";
        } else {
            $errors[] = "Table $table is missing";
            echo "<div class='check fail'><span class='icon'>✗</span><strong>$description</strong> - Table <code>$table</code> missing</div>";
        }
    } catch (PDOException $e) {
        $errors[] = "Error checking table $table: " . $e->getMessage();
        echo "<div class='check fail'><span class='icon'>✗</span>Error checking <code>$table</code>: " . htmlspecialchars($e->getMessage()) . "</div>";
    }
}

// ============================================================================
// CHECK 3: ADMIN PANEL LINKS
// ============================================================================

echo "<h2>3. Admin Panel Links Check</h2>";

$adminLinks = [
    'users.php' => 'User Management',
    'locations.php' => 'Locations',
    'reports.php' => 'Reports',
    'risk-categories.php' => 'Risk Categories',
    'custom-fields.php' => 'Custom Fields',
    'vehicle-types.php' => 'Vehicle Types',
    'pdf-templates.php' => 'PDF Templates',
    'system-customization.php' => 'System Customization',
    'smtp-settings.php' => 'SMTP Settings',
    'permissions-management.php' => 'Permissions',
    'version-control.php' => 'Version Control',
    'error-logs.php' => 'Error Logs',
];

echo "<table>";
echo "<thead><tr><th>Link</th><th>File</th><th>Status</th></tr></thead>";
echo "<tbody>";

foreach ($adminLinks as $file => $name) {
    $fullPath = __DIR__ . '/' . $file;
    $exists = file_exists($fullPath);
    
    if ($exists) {
        $passed[] = "Admin link '$name' is valid";
        echo "<tr><td><strong>$name</strong></td><td><code>$file</code></td><td><span class='badge badge-success'>✓ Exists</span></td></tr>";
    } else {
        $errors[] = "Admin link '$name' points to missing file";
        echo "<tr><td><strong>$name</strong></td><td><code>$file</code></td><td><span class='badge badge-danger'>✗ Missing</span></td></tr>";
    }
}

echo "</tbody></table>";

// ============================================================================
// CHECK 4: FILE PERMISSIONS
// ============================================================================

echo "<h2>4. File Permissions Check</h2>";

$writableDirs = [
    __DIR__ . '/../uploads' => 'Uploads directory',
    __DIR__ . '/../logs' => 'Logs directory',
    __DIR__ . '/../public/uploads/branding' => 'Branding uploads',
];

foreach ($writableDirs as $dir => $description) {
    if (!is_dir($dir)) {
        $warnings[] = "$description doesn't exist";
        echo "<div class='check warn'><span class='icon'>⚠</span><strong>$description</strong> - Directory doesn't exist: <code>$dir</code></div>";
    } elseif (!is_writable($dir)) {
        $errors[] = "$description is not writable";
        echo "<div class='check fail'><span class='icon'>✗</span><strong>$description</strong> - Not writable: <code>$dir</code></div>";
    } else {
        $passed[] = "$description is writable";
        echo "<div class='check pass'><span class='icon'>✓</span><strong>$description</strong> - Writable: <code>$dir</code></div>";
    }
}

// ============================================================================
// CHECK 5: FUNCTION CONFLICTS
// ============================================================================

echo "<h2>5. Function Declaration Check</h2>";

$headerFiles = [
    __DIR__ . '/../templates/header.php',
    __DIR__ . '/../templates/header-responsive.php',
];

foreach ($headerFiles as $file) {
    if (file_exists($file)) {
        $content = file_get_contents($file);
        if (strpos($content, 'function_exists') !== false) {
            $passed[] = basename($file) . " has proper function_exists checks";
            echo "<div class='check pass'><span class='icon'>✓</span>" . basename($file) . " - Has function_exists() checks</div>";
        } else {
            $warnings[] = basename($file) . " may cause function redeclaration";
            echo "<div class='check warn'><span class='icon'>⚠</span>" . basename($file) . " - Missing function_exists() checks</div>";
        }
    }
}

// ============================================================================
// CHECK 6: DATABASE CONFIGURATION
// ============================================================================

echo "<h2>6. Database Configuration Check</h2>";

try {
    $stmt = $pdo->query("SELECT VERSION() as version");
    $version = $stmt->fetch()['version'];
    $passed[] = "Database connection successful (MySQL $version)";
    echo "<div class='check pass'><span class='icon'>✓</span>Database connected - MySQL version: $version</div>";
} catch (PDOException $e) {
    $errors[] = "Database connection failed: " . $e->getMessage();
    echo "<div class='check fail'><span class='icon'>✗</span>Database connection error: " . htmlspecialchars($e->getMessage()) . "</div>";
}

// Check critical constants
$constants = ['DB_HOST', 'DB_USER', 'DB_NAME', 'SITE_URL'];
foreach ($constants as $const) {
    if (defined($const)) {
        $value = constant($const);
        $passed[] = "Constant $const is defined";
        echo "<div class='check pass'><span class='icon'>✓</span><code>$const</code> = " . htmlspecialchars($value) . "</div>";
    } else {
        $errors[] = "Constant $const is not defined";
        echo "<div class='check fail'><span class='icon'>✗</span><code>$const</code> - Not defined</div>";
    }
}

// ============================================================================
// SUMMARY
// ============================================================================

$totalChecks = count($passed) + count($warnings) + count($errors);
$passRate = $totalChecks > 0 ? round((count($passed) / $totalChecks) * 100) : 0;

echo "<h2>Summary</h2>";
echo "<div class='summary'>";
echo "<div class='stat stat-pass'><div class='stat-number'>" . count($passed) . "</div><div class='stat-label'>Passed</div></div>";
echo "<div class='stat stat-warn'><div class='stat-number'>" . count($warnings) . "</div><div class='stat-label'>Warnings</div></div>";
echo "<div class='stat stat-fail'><div class='stat-number'>" . count($errors) . "</div><div class='stat-label'>Errors</div></div>";
echo "</div>";

echo "<p><strong>Overall Health: " . $passRate . "%</strong></p>";

if (count($errors) === 0 && count($warnings) === 0) {
    echo "<div class='check pass' style='font-size: 1.2rem; padding: 20px;'><span class='icon'>✓</span><strong>System is fully operational!</strong> All checks passed.</div>";
} elseif (count($errors) === 0) {
    echo "<div class='check warn' style='font-size: 1.2rem; padding: 20px;'><span class='icon'>⚠</span><strong>System is operational</strong> but has " . count($warnings) . " warning(s) that should be addressed.</div>";
} else {
    echo "<div class='check fail' style='font-size: 1.2rem; padding: 20px;'><span class='icon'>✗</span><strong>System has issues!</strong> Found " . count($errors) . " error(s) that need immediate attention.</div>";
}

// ============================================================================
// RECOMMENDED ACTIONS
// ============================================================================

if (count($errors) > 0 || count($warnings) > 0) {
    echo "<h2>Recommended Actions</h2>";
    
    if (count($errors) > 0) {
        echo "<h3 style='color: #dc2626;'>Critical Issues:</h3>";
        echo "<ol>";
        foreach ($errors as $error) {
            echo "<li>" . htmlspecialchars($error) . "</li>";
        }
        echo "</ol>";
        
        echo "<p><strong>To fix:</strong></p>";
        echo "<ol>";
        echo "<li>Run the database fix: <code>mysql -u root -p database &lt; includes/complete_system_fix.sql</code></li>";
        echo "<li>Check file permissions: <code>chmod 755 uploads logs</code></li>";
        echo "<li>Verify all PHP files are uploaded correctly</li>";
        echo "<li>Clear browser cache and retry</li>";
        echo "</ol>";
    }
    
    if (count($warnings) > 0) {
        echo "<h3 style='color: #f59e0b;'>Warnings:</h3>";
        echo "<ul>";
        foreach ($warnings as $warning) {
            echo "<li>" . htmlspecialchars($warning) . "</li>";
        }
        echo "</ul>";
    }
}

?>

        <div style="margin-top: 40px; padding: 20px; background: #f9fafb; border-radius: 8px; border-left: 4px solid #667eea;">
            <h3>Need Help?</h3>
            <ul>
                <li><strong>Database Issues:</strong> Run <code>includes/complete_system_fix.sql</code></li>
                <li><strong>Missing Files:</strong> Re-upload the complete system package</li>
                <li><strong>Permission Errors:</strong> Set directories to 755, files to 644</li>
                <li><strong>Function Errors:</strong> Check templates/header.php for function_exists()</li>
            </ul>
        </div>

        <div style="margin-top: 20px; text-align: center;">
            <a href="admin-panel.php" class="btn" style="display: inline-block; padding: 10px 20px; background: #667eea; color: white; text-decoration: none; border-radius: 6px;">← Back to Admin Panel</a>
        </div>
    </div>
</body>
</html>

<?php include '../templates/footer.php'; ?>
