<?php
require_once '../includes/config.php';
require_once '../includes/email_helper.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$success = '';
$error = '';
$testResult = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['test_email'])) {
        // Test email configuration
        if (testEmailConfiguration()) {
            $testResult = 'success';
            $success = 'Test email sent successfully! Check your inbox.';
        } else {
            $testResult = 'error';
            $error = 'Failed to send test email. Check error logs for details.';
        }
    } elseif (isset($_POST['save_settings'])) {
        // Save settings to config.php (this is a template - actual implementation needs file writing)
        $success = 'Settings saved! Update your config.php file with these values.';
    }
}

include '../templates/header.php';
?>

<div class="container">
    <div class="page-header">
        <div>
            <h1 class="page-title">📧 SMTP Email Configuration</h1>
            <p class="page-subtitle">Configure email settings for automated reports</p>
        </div>
        <div class="btn-group">
            <a href="admin-panel.php" class="btn btn-secondary">← Back to Admin</a>
        </div>
    </div>

    <?php if ($success): ?>
    <div class="alert alert-success">
        ✅ <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <?php if ($error): ?>
    <div class="alert alert-error">
        ❌ <?php echo htmlspecialchars($error); ?>
    </div>
    <?php endif; ?>

    <!-- Current Configuration -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Current SMTP Configuration</h2>
        </div>
        <div class="card-body">
            <div class="config-grid">
                <div class="config-item">
                    <span class="config-label">SMTP Enabled:</span>
                    <span class="config-value">
                        <?php if (SMTP_ENABLED): ?>
                            <span class="badge badge-success">✓ Enabled</span>
                        <?php else: ?>
                            <span class="badge badge-warning">○ Disabled</span>
                        <?php endif; ?>
                    </span>
                </div>

                <div class="config-item">
                    <span class="config-label">SMTP Host:</span>
                    <span class="config-value"><code><?php echo SMTP_HOST ?: 'Not configured'; ?></code></span>
                </div>

                <div class="config-item">
                    <span class="config-label">SMTP Port:</span>
                    <span class="config-value"><code><?php echo SMTP_PORT; ?></code></span>
                </div>

                <div class="config-item">
                    <span class="config-label">Encryption:</span>
                    <span class="config-value"><code><?php echo SMTP_ENCRYPTION ?: 'None'; ?></code></span>
                </div>

                <div class="config-item">
                    <span class="config-label">Username:</span>
                    <span class="config-value"><code><?php echo SMTP_USERNAME ?: 'Not configured'; ?></code></span>
                </div>

                <div class="config-item">
                    <span class="config-label">Password:</span>
                    <span class="config-value">
                        <?php if (SMTP_PASSWORD): ?>
                            <code>••••••••</code>
                        <?php else: ?>
                            <code>Not configured</code>
                        <?php endif; ?>
                    </span>
                </div>

                <div class="config-item">
                    <span class="config-label">From Email:</span>
                    <span class="config-value"><code><?php echo SMTP_FROM_EMAIL ?: 'Not configured'; ?></code></span>
                </div>

                <div class="config-item">
                    <span class="config-label">From Name:</span>
                    <span class="config-value"><code><?php echo SMTP_FROM_NAME; ?></code></span>
                </div>
            </div>
        </div>
    </div>

    <!-- Test Email -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">🧪 Test Email Configuration</h2>
        </div>
        <div class="card-body">
            <p>Send a test email to verify your SMTP configuration is working correctly.</p>
            
            <?php if (!SMTP_ENABLED): ?>
            <div class="alert alert-warning">
                ⚠️ SMTP is currently disabled. Enable it in config.php to send emails.
            </div>
            <?php elseif (empty(SMTP_HOST) || empty(SMTP_USERNAME) || empty(SMTP_PASSWORD)): ?>
            <div class="alert alert-warning">
                ⚠️ SMTP configuration is incomplete. Configure all settings in config.php.
            </div>
            <?php endif; ?>

            <form method="POST">
                <p><strong>Test email will be sent to:</strong> <code><?php echo SMTP_USERNAME; ?></code></p>
                <button type="submit" name="test_email" class="btn btn-primary" 
                        <?php echo (!SMTP_ENABLED || empty(SMTP_HOST) || empty(SMTP_USERNAME) || empty(SMTP_PASSWORD)) ? 'disabled' : ''; ?>>
                    📧 Send Test Email
                </button>
            </form>
        </div>
    </div>

    <!-- Configuration Guide -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">⚙️ Configuration Guide</h2>
        </div>
        <div class="card-body">
            <h3>Step 1: Edit config.php</h3>
            <p>Edit <code>includes/config.php</code> and update the SMTP settings:</p>
            
            <pre style="background: #f5f5f5; padding: 15px; border-radius: 6px; overflow-x: auto;"><code>// SMTP Email Configuration
define('SMTP_ENABLED', true);
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-email@gmail.com');
define('SMTP_PASSWORD', 'your-app-password');
define('SMTP_FROM_EMAIL', 'your-email@gmail.com');
define('SMTP_FROM_NAME', 'Risk Assessment System');</code></pre>

            <h3 style="margin-top: 2rem;">Step 2: Common SMTP Providers</h3>
            
            <div class="smtp-providers">
                <div class="provider-card">
                    <h4>📧 Gmail</h4>
                    <ul>
                        <li><strong>Host:</strong> smtp.gmail.com</li>
                        <li><strong>Port:</strong> 587 (TLS) or 465 (SSL)</li>
                        <li><strong>Encryption:</strong> tls or ssl</li>
                        <li><strong>Note:</strong> Use App Password (not regular password)</li>
                        <li><a href="https://support.google.com/accounts/answer/185833" target="_blank">Create App Password →</a></li>
                    </ul>
                </div>

                <div class="provider-card">
                    <h4>📧 Microsoft 365 / Outlook</h4>
                    <ul>
                        <li><strong>Host:</strong> smtp.office365.com</li>
                        <li><strong>Port:</strong> 587</li>
                        <li><strong>Encryption:</strong> tls</li>
                        <li><strong>Username:</strong> Full email address</li>
                    </ul>
                </div>

                <div class="provider-card">
                    <h4>📧 SendGrid</h4>
                    <ul>
                        <li><strong>Host:</strong> smtp.sendgrid.net</li>
                        <li><strong>Port:</strong> 587</li>
                        <li><strong>Encryption:</strong> tls</li>
                        <li><strong>Username:</strong> apikey</li>
                        <li><strong>Password:</strong> Your SendGrid API key</li>
                    </ul>
                </div>

                <div class="provider-card">
                    <h4>📧 Amazon SES</h4>
                    <ul>
                        <li><strong>Host:</strong> email-smtp.us-east-1.amazonaws.com</li>
                        <li><strong>Port:</strong> 587</li>
                        <li><strong>Encryption:</strong> tls</li>
                        <li><strong>Username:</strong> SMTP username from SES</li>
                        <li><strong>Password:</strong> SMTP password from SES</li>
                    </ul>
                </div>
            </div>

            <h3 style="margin-top: 2rem;">Step 3: Test Configuration</h3>
            <ol>
                <li>Save your changes to config.php</li>
                <li>Click "Send Test Email" button above</li>
                <li>Check your inbox for the test email</li>
                <li>If it fails, check the error logs</li>
            </ol>

            <h3 style="margin-top: 2rem;">Troubleshooting</h3>
            <ul>
                <li><strong>Authentication failed:</strong> Check username/password, use app password for Gmail</li>
                <li><strong>Connection timeout:</strong> Check firewall, ensure port is open</li>
                <li><strong>SSL/TLS errors:</strong> Try different encryption settings (tls/ssl/none)</li>
                <li><strong>Still not working:</strong> Check logs at <code>logs/email.log</code></li>
            </ul>
        </div>
    </div>

    <!-- Email Logs -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">📋 Recent Email Activity</h2>
        </div>
        <div class="card-body">
            <?php
            $logFile = __DIR__ . '/../logs/email.log';
            if (file_exists($logFile)) {
                $lines = file($logFile);
                $lines = array_reverse(array_slice($lines, -10)); // Last 10 entries
                
                if (!empty($lines)) {
                    echo '<div class="log-container">';
                    foreach ($lines as $line) {
                        $line = htmlspecialchars(trim($line));
                        $class = '';
                        if (strpos($line, 'successfully') !== false) {
                            $class = 'log-success';
                        } elseif (strpos($line, 'failed') !== false || strpos($line, 'error') !== false) {
                            $class = 'log-error';
                        }
                        echo "<div class='log-entry $class'>$line</div>";
                    }
                    echo '</div>';
                } else {
                    echo '<p style="color: var(--text-muted);">No email activity yet.</p>';
                }
            } else {
                echo '<p style="color: var(--text-muted);">Email log file not found.</p>';
            }
            ?>
        </div>
    </div>
</div>

<style>
.config-grid {
    display: grid;
    grid-template-columns: 200px 1fr;
    gap: 1rem;
    max-width: 800px;
}

.config-item {
    display: contents;
}

.config-label {
    font-weight: 600;
    color: var(--text-secondary);
}

.config-value {
    color: var(--text-primary);
}

.config-value code {
    background: #f5f5f5;
    padding: 2px 8px;
    border-radius: 4px;
    font-family: monospace;
}

.badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 0.85rem;
    font-weight: 600;
}

.badge-success {
    background: rgba(5, 150, 105, 0.1);
    color: var(--success);
}

.badge-warning {
    background: rgba(251, 191, 36, 0.1);
    color: var(--warning);
}

.alert {
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1.5rem;
}

.alert-success {
    background: rgba(5, 150, 105, 0.1);
    border: 1px solid var(--success);
    color: var(--success);
}

.alert-error {
    background: rgba(220, 38, 38, 0.1);
    border: 1px solid var(--danger);
    color: var(--danger);
}

.alert-warning {
    background: rgba(251, 191, 36, 0.1);
    border: 1px solid var(--warning);
    color: #856404;
}

.smtp-providers {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 1rem;
    margin: 1rem 0;
}

.provider-card {
    background: #f9fafb;
    padding: 1.5rem;
    border-radius: 8px;
    border: 1px solid var(--border-color);
}

.provider-card h4 {
    margin-bottom: 1rem;
    color: var(--primary);
}

.provider-card ul {
    list-style: none;
    padding: 0;
}

.provider-card li {
    padding: 0.25rem 0;
    font-size: 0.9rem;
}

.log-container {
    background: #f5f5f5;
    padding: 1rem;
    border-radius: 6px;
    max-height: 300px;
    overflow-y: auto;
    font-family: monospace;
    font-size: 0.85rem;
}

.log-entry {
    padding: 0.5rem;
    border-bottom: 1px solid #e5e7eb;
}

.log-entry:last-child {
    border-bottom: none;
}

.log-success {
    color: var(--success);
}

.log-error {
    color: var(--danger);
}
</style>

<?php include '../templates/footer.php'; ?>
