<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$error = '';
$success = '';

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        switch ($_POST['action']) {
            case 'add_category':
                $stmt = $pdo->prepare("
                    INSERT INTO risk_categories (category_name, description, display_order, is_active)
                    VALUES (?, ?, ?, 1)
                ");
                $stmt->execute([
                    $_POST['category_name'],
                    $_POST['description'] ?? '',
                    $_POST['display_order'] ?? 0
                ]);
                $success = 'Risk category added successfully!';
                break;
            
            case 'update_category':
                $stmt = $pdo->prepare("
                    UPDATE risk_categories 
                    SET category_name = ?, description = ?, display_order = ?, is_active = ?
                    WHERE id = ?
                ");
                $stmt->execute([
                    $_POST['category_name'],
                    $_POST['description'] ?? '',
                    $_POST['display_order'] ?? 0,
                    isset($_POST['is_active']) ? 1 : 0,
                    $_POST['category_id']
                ]);
                $success = 'Category updated successfully!';
                break;
            
            case 'delete_category':
                // Check if category has items
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM risk_items WHERE category_id = ?");
                $stmt->execute([$_POST['category_id']]);
                $count = $stmt->fetchColumn();
                
                if ($count > 0) {
                    $error = "Cannot delete: This category has $count risk item(s). Delete items first or deactivate the category.";
                } else {
                    $stmt = $pdo->prepare("DELETE FROM risk_categories WHERE id = ?");
                    $stmt->execute([$_POST['category_id']]);
                    $success = 'Category deleted successfully!';
                }
                break;
        }
    } catch (PDOException $e) {
        $error = 'Database error: ' . $e->getMessage();
    }
}

// Fetch categories with item counts
$stmt = $pdo->query("
    SELECT 
        rc.*,
        COUNT(ri.id) as item_count
    FROM risk_categories rc
    LEFT JOIN risk_items ri ON rc.id = ri.category_id
    GROUP BY rc.id
    ORDER BY rc.display_order, rc.category_name
");
$categories = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Risk Categories - Risk System</title>
    <link rel="stylesheet" href="css/style.css">
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">⚠️ Risk Categories Management</h1>
                <p class="page-subtitle">Manage risk assessment categories and items</p>
            </div>
            <button onclick="openAddCategory()" class="btn btn-primary">➕ Add Category</button>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <span>✅</span>
                <span><?php echo sanitizeOutput($success); ?></span>
            </div>
        <?php endif; ?>

        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Risk Categories (<?php echo count($categories); ?>)</h2>
            </div>

            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Category Name</th>
                            <th>Description</th>
                            <th>Display Order</th>
                            <th>Risk Items</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($categories as $category): ?>
                            <tr>
                                <td><strong><?php echo sanitizeOutput($category['category_name']); ?></strong></td>
                                <td><?php echo sanitizeOutput($category['description']); ?></td>
                                <td><?php echo $category['display_order']; ?></td>
                                <td>
                                    <span style="background: var(--bg-tertiary); padding: 0.25rem 0.5rem; border-radius: 4px;">
                                        <?php echo $category['item_count']; ?> items
                                    </span>
                                </td>
                                <td>
                                    <?php if ($category['is_active']): ?>
                                        <span class="badge badge-success">Active</span>
                                    <?php else: ?>
                                        <span class="badge" style="background: rgba(148, 163, 184, 0.2);">Inactive</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="btn-group" style="gap: 0.5rem;">
                                        <a href="risk-items.php?category=<?php echo $category['id']; ?>" 
                                           class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                            📋 Items
                                        </a>
                                        <button onclick='editCategory(<?php echo htmlspecialchars(json_encode($category)); ?>)' 
                                                class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                            ✏️ Edit
                                        </button>
                                        <?php if ($category['item_count'] == 0): ?>
                                            <form method="POST" style="display: inline;" 
                                                  onsubmit="return confirm('Delete this category?');">
                                                <input type="hidden" name="action" value="delete_category">
                                                <input type="hidden" name="category_id" value="<?php echo $category['id']; ?>">
                                                <button type="submit" class="btn btn-danger" style="padding: 0.5rem 1rem;">
                                                    🗑️ Delete
                                                </button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <div style="margin-top: 2rem; padding: 1.5rem; background: var(--bg-tertiary); border-radius: 8px;">
            <h3 style="color: var(--text-primary); margin-bottom: 0.5rem;">ℹ️ About Risk Categories</h3>
            <p style="color: var(--text-secondary); line-height: 1.6;">
                Risk categories organize the assessment process into logical groups. Each category contains 
                multiple risk items that assessors evaluate. Deactivating a category hides it from new 
                assessments but preserves historical data.
            </p>
        </div>
    </div>

    <!-- Add/Edit Modal -->
    <div id="category-modal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 class="modal-title" id="modal-title">Add Category</h2>
                <button type="button" class="modal-close" onclick="window.app.closeModal('category-modal')">&times;</button>
            </div>

            <form method="POST" id="category-form">
                <input type="hidden" name="action" id="form-action" value="add_category">
                <input type="hidden" name="category_id" id="category-id">

                <div class="form-group">
                    <label class="form-label required" for="category_name">Category Name</label>
                    <input type="text" id="category_name" name="category_name" class="form-control" required>
                </div>

                <div class="form-group">
                    <label class="form-label" for="description">Description</label>
                    <textarea id="description" name="description" class="form-textarea" rows="3"></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label" for="display_order">Display Order</label>
                    <input type="number" id="display_order" name="display_order" class="form-control" value="0">
                </div>

                <div class="form-group" id="active-group" style="display: none;">
                    <div style="display: flex; align-items: center; gap: 0.5rem;">
                        <input type="checkbox" id="is_active" name="is_active" checked style="width: auto;">
                        <label for="is_active" style="margin: 0;">Active</label>
                    </div>
                </div>

                <div class="btn-group" style="margin-top: 2rem;">
                    <button type="submit" class="btn btn-primary">💾 Save Category</button>
                    <button type="button" class="btn btn-secondary" onclick="window.app.closeModal('category-modal')">Cancel</button>
                </div>
            </form>
        </div>
    </div>

    <script src="js/app.js"></script>
    <script>
        function openAddCategory() {
            document.getElementById('modal-title').textContent = 'Add Risk Category';
            document.getElementById('form-action').value = 'add_category';
            document.getElementById('category-form').reset();
            document.getElementById('category-id').value = '';
            document.getElementById('active-group').style.display = 'none';
            window.app.openModal('category-modal');
        }

        function editCategory(category) {
            document.getElementById('modal-title').textContent = 'Edit Risk Category';
            document.getElementById('form-action').value = 'update_category';
            document.getElementById('category-id').value = category.id;
            document.getElementById('category_name').value = category.category_name;
            document.getElementById('description').value = category.description || '';
            document.getElementById('display_order').value = category.display_order;
            document.getElementById('is_active').checked = category.is_active == 1;
            document.getElementById('active-group').style.display = 'block';
            window.app.openModal('category-modal');
        }
    </script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
