<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

// Get report statistics
$stats = [];

// Risk level distribution
$stmt = $pdo->query("
    SELECT overall_risk_level, COUNT(*) as count 
    FROM assessments 
    WHERE status != 'draft'
    GROUP BY overall_risk_level
");
$risk_distribution = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

// Assessments by month (last 6 months)
$stmt = $pdo->query("
    SELECT DATE_FORMAT(assessment_date, '%Y-%m') as month, COUNT(*) as count
    FROM assessments
    WHERE assessment_date >= DATE_SUB(NOW(), INTERVAL 6 MONTH)
    GROUP BY month
    ORDER BY month DESC
");
$monthly_assessments = $stmt->fetchAll();

// Top locations by assessment count
$stmt = $pdo->query("
    SELECT l.location_name, COUNT(a.id) as assessment_count
    FROM locations l
    LEFT JOIN assessments a ON l.id = a.location_id
    GROUP BY l.id
    ORDER BY assessment_count DESC
    LIMIT 10
");
$top_locations = $stmt->fetchAll();

// Most common vehicle types
$stmt = $pdo->query("
    SELECT vehicle_type, COUNT(*) as count
    FROM assessments
    WHERE vehicle_type IS NOT NULL AND vehicle_type != ''
    GROUP BY vehicle_type
    ORDER BY count DESC
    LIMIT 5
");
$vehicle_usage = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports & Analytics - Risk System</title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .report-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }
        
        .report-card {
            background: var(--bg-card);
            border-radius: 12px;
            padding: 1.5rem;
            border: 1px solid var(--border-color);
        }
        
        .report-card h3 {
            color: var(--text-primary);
            margin-bottom: 1rem;
            font-size: 1.1rem;
        }
        
        .chart-bar {
            display: flex;
            align-items: center;
            margin-bottom: 0.75rem;
        }
        
        .chart-label {
            width: 120px;
            font-size: 0.9rem;
            color: var(--text-secondary);
        }
        
        .chart-value {
            flex: 1;
            height: 30px;
            background: var(--primary);
            border-radius: 4px;
            display: flex;
            align-items: center;
            padding: 0 0.75rem;
            color: white;
            font-weight: 600;
            font-size: 0.85rem;
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">📊 Reports & Analytics</h1>
                <p class="page-subtitle">System-wide statistics and insights</p>
            </div>
            <a href="admin-panel.php" class="btn btn-secondary">← Back to Admin Panel</a>
        </div>

        <!-- Risk Level Distribution -->
        <div class="report-grid">
            <div class="report-card">
                <h3>🎯 Risk Level Distribution</h3>
                <?php
                $total = array_sum($risk_distribution);
                $levels = ['low' => 'Low', 'medium' => 'Medium', 'high' => 'High', 'critical' => 'Critical'];
                foreach ($levels as $key => $label):
                    $count = $risk_distribution[$key] ?? 0;
                    $percentage = $total > 0 ? ($count / $total * 100) : 0;
                ?>
                    <div class="chart-bar">
                        <div class="chart-label"><?php echo $label; ?>:</div>
                        <div class="chart-value" style="width: <?php echo $percentage; ?>%; background: var(--<?php echo $key == 'low' ? 'success' : ($key == 'critical' ? 'danger' : 'warning'); ?>);">
                            <?php echo $count; ?> (<?php echo round($percentage); ?>%)
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <div class="report-card">
                <h3>📅 Assessments by Month</h3>
                <?php if (empty($monthly_assessments)): ?>
                    <p style="color: var(--text-muted); text-align: center; padding: 2rem 0;">No assessment data</p>
                <?php else: ?>
                    <?php $max = max(array_column($monthly_assessments, 'count')); ?>
                    <?php foreach ($monthly_assessments as $month): ?>
                        <div class="chart-bar">
                            <div class="chart-label"><?php echo date('M Y', strtotime($month['month'] . '-01')); ?></div>
                            <div class="chart-value" style="width: <?php echo ($month['count'] / $max * 100); ?>%;">
                                <?php echo $month['count']; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>

        <!-- Top Locations -->
        <div class="report-card" style="margin-bottom: 2rem;">
            <h3>📍 Top 10 Locations by Assessment Count</h3>
            <?php if (empty($top_locations)): ?>
                <p style="color: var(--text-muted); text-align: center; padding: 2rem 0;">No location data</p>
            <?php else: ?>
                <?php $max = max(array_column($top_locations, 'assessment_count')); ?>
                <?php foreach ($top_locations as $location): ?>
                    <div class="chart-bar">
                        <div class="chart-label" style="width: 200px;">
                            <?php echo sanitizeOutput(substr($location['location_name'], 0, 25)); ?>
                        </div>
                        <div class="chart-value" style="width: <?php echo ($location['assessment_count'] / $max * 100); ?>%;">
                            <?php echo $location['assessment_count']; ?> assessments
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Vehicle Usage -->
        <div class="report-card" style="margin-bottom: 2rem;">
            <h3>🚚 Most Common Vehicle Types</h3>
            <?php if (empty($vehicle_usage)): ?>
                <p style="color: var(--text-muted); text-align: center; padding: 2rem 0;">No vehicle data</p>
            <?php else: ?>
                <?php $max = max(array_column($vehicle_usage, 'count')); ?>
                <?php foreach ($vehicle_usage as $vehicle): ?>
                    <div class="chart-bar">
                        <div class="chart-label" style="width: 150px;">
                            <?php echo sanitizeOutput($vehicle['vehicle_type']); ?>
                        </div>
                        <div class="chart-value" style="width: <?php echo ($vehicle['count'] / $max * 100); ?>%;">
                            <?php echo $vehicle['count']; ?> times
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Export Options -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">📥 Export Options</h2>
            </div>

            <div style="padding: 1.5rem;">
                <!-- Action Items Report - Featured -->
                <div style="padding: 1.5rem; background: linear-gradient(135deg, rgba(217, 119, 6, 0.1) 0%, rgba(217, 119, 6, 0.05) 100%); border: 2px solid var(--primary); border-radius: 8px; margin-bottom: 1.5rem;">
                    <div style="display: flex; align-items: start; gap: 1.5rem;">
                        <div style="font-size: 3rem;">📋</div>
                        <div style="flex: 1;">
                            <h3 style="color: var(--text-primary); margin: 0 0 0.5rem 0;">Action Items Report</h3>
                            <p style="color: var(--text-secondary); margin: 0 0 1rem 0;">
                                Track and manage pending actions from risk assessments. View overdue items, actions due soon, and on-track items organized by location.
                            </p>
                            <a href="action-items-report.php" class="btn btn-primary">
                                View Action Items Report →
                            </a>
                        </div>
                    </div>
                </div>

                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 1rem;">
                    <div style="padding: 1.5rem; background: var(--bg-tertiary); border-radius: 8px; text-align: center;">
                        <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">📊</div>
                        <strong style="color: var(--text-primary); display: block; margin-bottom: 0.5rem;">CSV Export</strong>
                        <p style="color: var(--text-muted); font-size: 0.85rem; margin-bottom: 1rem;">
                            Export all assessments to CSV
                        </p>
                        <button class="btn btn-secondary" style="padding: 0.5rem 1rem;">Download CSV</button>
                    </div>

                    <div style="padding: 1.5rem; background: var(--bg-tertiary); border-radius: 8px; text-align: center;">
                        <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">📄</div>
                        <strong style="color: var(--text-primary); display: block; margin-bottom: 0.5rem;">PDF Report</strong>
                        <p style="color: var(--text-muted); font-size: 0.85rem; margin-bottom: 1rem;">
                            Generate comprehensive PDF
                        </p>
                        <button class="btn btn-secondary" style="padding: 0.5rem 1rem;">Generate PDF</button>
                    </div>

                    <div style="padding: 1.5rem; background: var(--bg-tertiary); border-radius: 8px; text-align: center;">
                        <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">📊</div>
                        <strong style="color: var(--text-primary); display: block; margin-bottom: 0.5rem;">Excel Export</strong>
                        <p style="color: var(--text-muted); font-size: 0.85rem; margin-bottom: 1rem;">
                            Export to Excel format
                        </p>
                        <button class="btn btn-secondary" style="padding: 0.5rem 1rem;">Download XLSX</button>
                    </div>

                    <div style="padding: 1.5rem; background: var(--bg-tertiary); border-radius: 8px; text-align: center;">
                        <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">📈</div>
                        <strong style="color: var(--text-primary); display: block; margin-bottom: 0.5rem;">Custom Report</strong>
                        <p style="color: var(--text-muted); font-size: 0.85rem; margin-bottom: 1rem;">
                            Build custom report query
                        </p>
                        <button class="btn btn-secondary" style="padding: 0.5rem 1rem;">Build Report</button>
                    </div>
                </div>

                <div style="margin-top: 1.5rem; padding: 1rem; background: #e3f2fd; border-left: 4px solid #2196f3; border-radius: 4px;">
                    <strong style="color: #1565c0;">ℹ️ Note:</strong>
                    <p style="color: #1565c0; margin: 0.5rem 0 0 0;">
                        Export functionality can be implemented based on your specific reporting requirements. 
                        Contact support for custom report development.
                    </p>
                </div>
            </div>
        </div>
    </div>

    <script src="js/app.js"></script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
