<?php
require_once '../includes/config.php';
require_once '../includes/permissions.php';
requireLogin();

// Only admins can manage permissions
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$success = '';
$error = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['action'])) {
            switch($_POST['action']) {
                case 'update_role_permissions':
                    $role = $_POST['role'];
                    $permissions = $_POST['permissions'] ?? [];
                    
                    if (updateRolePermissions($pdo, $role, $permissions)) {
                        $success = ucfirst($role) . ' role permissions updated successfully!';
                    } else {
                        throw new Exception('Failed to update role permissions');
                    }
                    break;
            }
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Fetch all permissions grouped
$permissions_grouped = isPermissionSystemEnabled($pdo) ? getPermissionsGrouped($pdo) : [];

// Fetch role permissions
$admin_perms = isPermissionSystemEnabled($pdo) ? getRolePermissions($pdo, 'admin') : [];
$assessor_perms = isPermissionSystemEnabled($pdo) ? getRolePermissions($pdo, 'assessor') : [];
$viewer_perms = isPermissionSystemEnabled($pdo) ? getRolePermissions($pdo, 'viewer') : [];

// Fetch all users
$stmt = $pdo->query("SELECT id, username, full_name, email, role FROM users WHERE is_active = 1 ORDER BY role, full_name");
$users = $stmt->fetchAll();

include '../templates/header.php';
?>

<div class="container">
    <div class="page-header">
        <div>
            <h1 class="page-title">🔐 Permissions Management</h1>
            <p class="page-subtitle">Control user access to system sections and features</p>
        </div>
        <div class="btn-group">
            <a href="admin-panel.php" class="btn btn-secondary">← Back to Admin</a>
        </div>
    </div>

    <?php if (!isPermissionSystemEnabled($pdo)): ?>
        <div class="alert alert-warning">
            ⚠️ <strong>Permission system not yet installed.</strong> 
            Run the migration: <code>mysql -u scubatricky_risk -p scubatricky_risk < includes/migration_permissions.sql</code>
        </div>
    <?php endif; ?>

    <?php if ($success): ?>
        <div class="alert alert-success">✅ <?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>

    <?php if ($error): ?>
        <div class="alert alert-danger">❌ <?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>

    <?php if (isPermissionSystemEnabled($pdo)): ?>
    <!-- Role-Based Permissions -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Role-Based Permissions</h2>
        </div>
        <div class="card-body">
            <p style="color: var(--text-muted); margin-bottom: 1.5rem;">
                Configure default permissions for each role. Users inherit permissions from their role.
            </p>

            <div class="permissions-tabs">
                <button class="tab-button active" onclick="switchTab(event, 'admin')">
                    👑 Admin (<?php echo count($admin_perms); ?>)
                </button>
                <button class="tab-button" onclick="switchTab(event, 'assessor')">
                    👤 Assessor (<?php echo count($assessor_perms); ?>)
                </button>
                <button class="tab-button" onclick="switchTab(event, 'viewer')">
                    👁️ Viewer (<?php echo count($viewer_perms); ?>)
                </button>
            </div>

            <?php foreach (['admin', 'assessor', 'viewer'] as $role): ?>
            <div id="tab-<?php echo $role; ?>" class="tab-content" style="<?php echo $role !== 'admin' ? 'display:none;' : ''; ?>">
                <form method="POST" style="margin-top: 1.5rem;">
                    <input type="hidden" name="action" value="update_role_permissions">
                    <input type="hidden" name="role" value="<?php echo $role; ?>">
                    
                    <?php
                    $role_permissions = $role === 'admin' ? $admin_perms : ($role === 'assessor' ? $assessor_perms : $viewer_perms);
                    ?>
                    
                    <?php foreach ($permissions_grouped as $group => $perms): ?>
                    <div class="permission-group">
                        <h3 class="permission-group-title">
                            <?php echo htmlspecialchars($group); ?>
                            <small>(<?php echo count($perms); ?> permissions)</small>
                        </h3>
                        
                        <div class="permission-items">
                            <?php foreach ($perms as $perm): ?>
                            <label class="permission-checkbox">
                                <input 
                                    type="checkbox" 
                                    name="permissions[]" 
                                    value="<?php echo htmlspecialchars($perm['permission_key']); ?>"
                                    <?php echo in_array($perm['permission_key'], $role_permissions) ? 'checked' : ''; ?>
                                    <?php echo $role === 'admin' ? 'checked disabled' : ''; ?>
                                >
                                <div>
                                    <strong><?php echo htmlspecialchars($perm['permission_name']); ?></strong>
                                    <p><?php echo htmlspecialchars($perm['description']); ?></p>
                                </div>
                            </label>
                            <?php endforeach; ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                    
                    <?php if ($role !== 'admin'): ?>
                    <div style="margin-top: 1.5rem; padding-top: 1.5rem; border-top: 1px solid var(--border-color);">
                        <button type="submit" class="btn btn-primary">
                            💾 Save <?php echo ucfirst($role); ?> Permissions
                        </button>
                    </div>
                    <?php else: ?>
                    <div style="margin-top: 1.5rem; padding: 1rem; background: rgba(5, 150, 105, 0.1); border-radius: 8px;">
                        ℹ️ <strong>Admin role has full access to all features.</strong>
                    </div>
                    <?php endif; ?>
                </form>
            </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>

    <!-- User List -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Users by Role</h2>
        </div>
        <div class="card-body">
            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>User</th>
                            <th>Role</th>
                            <th>Email</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user): ?>
                        <tr>
                            <td>
                                <strong><?php echo htmlspecialchars($user['full_name']); ?></strong><br>
                                <small style="color: var(--text-muted);"><?php echo htmlspecialchars($user['username']); ?></small>
                            </td>
                            <td>
                                <span class="badge badge-<?php echo $user['role'] === 'admin' ? 'danger' : ($user['role'] === 'assessor' ? 'info' : 'secondary'); ?>">
                                    <?php echo ucfirst($user['role']); ?>
                                </span>
                            </td>
                            <td><?php echo htmlspecialchars($user['email']); ?></td>
                            <td><span class="badge badge-success">Active</span></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<style>
.alert {
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1.5rem;
}

.alert-success {
    background: rgba(5, 150, 105, 0.1);
    border: 1px solid var(--success);
    color: var(--success);
}

.alert-danger {
    background: rgba(220, 38, 38, 0.1);
    border: 1px solid var(--danger);
    color: var(--danger);
}

.alert-warning {
    background: rgba(245, 158, 11, 0.1);
    border: 1px solid var(--warning);
    color: var(--warning);
}

.permissions-tabs {
    display: flex;
    gap: 0.5rem;
    border-bottom: 2px solid var(--border-color);
    margin-bottom: 1.5rem;
}

.tab-button {
    padding: 0.75rem 1.5rem;
    background: transparent;
    border: none;
    border-bottom: 3px solid transparent;
    cursor: pointer;
    font-weight: 600;
    color: var(--text-secondary);
    transition: var(--transition);
}

.tab-button:hover {
    color: var(--primary);
}

.tab-button.active {
    color: var(--primary);
    border-bottom-color: var(--primary);
}

.permission-group {
    margin-bottom: 2rem;
}

.permission-group-title {
    color: var(--text-primary);
    margin-bottom: 1rem;
    padding-bottom: 0.5rem;
    border-bottom: 2px solid var(--border-color);
}

.permission-group-title small {
    color: var(--text-muted);
    font-weight: normal;
    font-size: 0.85rem;
}

.permission-items {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(350px, 1fr));
    gap: 1rem;
}

.permission-checkbox {
    display: flex;
    gap: 0.75rem;
    padding: 1rem;
    background: var(--bg-secondary);
    border: 2px solid var(--border-color);
    border-radius: 8px;
    cursor: pointer;
    transition: var(--transition);
}

.permission-checkbox:hover {
    border-color: var(--primary);
    box-shadow: var(--shadow);
}

.permission-checkbox input[type="checkbox"] {
    margin-top: 0.25rem;
    cursor: pointer;
}

.permission-checkbox strong {
    display: block;
    color: var(--text-primary);
    margin-bottom: 0.25rem;
}

.permission-checkbox p {
    margin: 0;
    font-size: 0.85rem;
    color: var(--text-muted);
}
</style>

<script>
function switchTab(event, role) {
    // Hide all tabs
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.style.display = 'none';
    });
    
    // Remove active class from all buttons
    document.querySelectorAll('.tab-button').forEach(btn => {
        btn.classList.remove('active');
    });
    
    // Show selected tab
    document.getElementById('tab-' + role).style.display = 'block';
    
    // Add active class to clicked button
    event.target.classList.add('active');
}
</script>

<?php include '../templates/footer.php'; ?>
