<?php
require_once '../includes/config.php';
requireLogin();

// Only admins can access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$error = '';
$success = '';

// Handle template actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        switch ($_POST['action']) {
            case 'save_template':
                $config = json_decode($_POST['template_config'], true);
                if (!$config) {
                    throw new Exception('Invalid template configuration');
                }
                
                if (!empty($_POST['template_id'])) {
                    // Update existing
                    $stmt = $pdo->prepare("
                        UPDATE pdf_templates 
                        SET template_name = ?, description = ?, template_type = ?,
                            page_size = ?, orientation = ?, template_config = ?,
                            is_default = ?, is_active = ?
                        WHERE id = ?
                    ");
                    $stmt->execute([
                        $_POST['template_name'],
                        $_POST['description'],
                        $_POST['template_type'],
                        $_POST['page_size'],
                        $_POST['orientation'],
                        $_POST['template_config'],
                        isset($_POST['is_default']) ? 1 : 0,
                        isset($_POST['is_active']) ? 1 : 0,
                        $_POST['template_id']
                    ]);
                    $success = 'Template updated successfully!';
                } else {
                    // Create new
                    $stmt = $pdo->prepare("
                        INSERT INTO pdf_templates 
                        (template_name, description, template_type, page_size, orientation, 
                         template_config, is_default, is_active, created_by)
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([
                        $_POST['template_name'],
                        $_POST['description'],
                        $_POST['template_type'],
                        $_POST['page_size'],
                        $_POST['orientation'],
                        $_POST['template_config'],
                        isset($_POST['is_default']) ? 1 : 0,
                        isset($_POST['is_active']) ? 1 : 0,
                        $_SESSION['user_id']
                    ]);
                    $success = 'Template created successfully!';
                }
                break;
            
            case 'delete':
                $stmt = $pdo->prepare("DELETE FROM pdf_templates WHERE id = ?");
                $stmt->execute([$_POST['template_id']]);
                $success = 'Template deleted successfully!';
                break;
        }
    } catch (Exception $e) {
        $error = 'Error: ' . $e->getMessage();
    }
}

// Fetch all templates
$stmt = $pdo->query("
    SELECT pt.*, u.full_name as created_by_name
    FROM pdf_templates pt
    LEFT JOIN users u ON pt.created_by = u.id
    ORDER BY pt.is_default DESC, pt.created_at DESC
");
$templates = $stmt->fetchAll();

// Fetch available fields grouped by table
$stmt = $pdo->query("
    SELECT * FROM pdf_field_mapping 
    WHERE is_active = 1 
    ORDER BY table_name, display_order
");
$all_fields = $stmt->fetchAll();

$fields_by_table = [];
foreach ($all_fields as $field) {
    $fields_by_table[$field['table_name']][] = $field;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PDF Template Designer - Risk System</title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .designer-layout {
            display: grid;
            grid-template-columns: 300px 1fr 350px;
            gap: 1.5rem;
            margin-top: 1.5rem;
        }
        
        .fields-panel, .canvas-panel, .properties-panel {
            background: var(--bg-card);
            border-radius: 12px;
            border: 1px solid var(--border-color);
            padding: 1.5rem;
        }
        
        .fields-panel {
            max-height: calc(100vh - 200px);
            overflow-y: auto;
        }
        
        .field-group {
            margin-bottom: 1.5rem;
        }
        
        .field-group-title {
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.75rem;
            padding: 0.5rem;
            background: var(--bg-tertiary);
            border-radius: 6px;
        }
        
        .field-item {
            padding: 0.75rem;
            margin-bottom: 0.5rem;
            background: var(--bg-secondary);
            border: 1px solid var(--border-color);
            border-radius: 6px;
            cursor: move;
            transition: all 0.2s;
        }
        
        .field-item:hover {
            background: var(--bg-tertiary);
            border-color: var(--primary);
            transform: translateX(4px);
        }
        
        .field-item-name {
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.25rem;
        }
        
        .field-item-type {
            font-size: 0.75rem;
            color: var(--text-muted);
        }
        
        .canvas-area {
            min-height: 600px;
            background: white;
            border: 2px dashed var(--border-color);
            border-radius: 8px;
            padding: 2rem;
            position: relative;
        }
        
        .canvas-area.dragover {
            border-color: var(--primary);
            background: rgba(37, 99, 235, 0.05);
        }
        
        .canvas-element {
            padding: 1rem;
            margin-bottom: 1rem;
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 6px;
            cursor: move;
            position: relative;
        }
        
        .canvas-element:hover {
            border-color: var(--primary);
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .canvas-element.selected {
            border-color: var(--primary);
            border-width: 2px;
            background: rgba(37, 99, 235, 0.05);
        }
        
        .element-controls {
            position: absolute;
            top: 0.5rem;
            right: 0.5rem;
            display: flex;
            gap: 0.25rem;
        }
        
        .element-control-btn {
            background: white;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 0.25rem 0.5rem;
            cursor: pointer;
            font-size: 0.75rem;
        }
        
        .element-control-btn:hover {
            background: var(--bg-tertiary);
        }
        
        .properties-panel {
            max-height: calc(100vh - 200px);
            overflow-y: auto;
        }
        
        .property-group {
            margin-bottom: 1.5rem;
        }
        
        .property-label {
            font-weight: 600;
            color: var(--text-primary);
            margin-bottom: 0.5rem;
            display: block;
        }
        
        .template-list {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 1rem;
        }
        
        .template-card {
            background: var(--bg-card);
            border: 1px solid var(--border-color);
            border-radius: 8px;
            padding: 1.5rem;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .template-card:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .template-card.default {
            border-color: var(--success);
            border-width: 2px;
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <div>
                <h1 class="page-title">📄 PDF Template Designer</h1>
                <p class="page-subtitle">Design and customize PDF export templates</p>
            </div>
            <div class="btn-group">
                <button onclick="showDesigner()" class="btn btn-primary">➕ New Template</button>
                <a href="admin-panel.php" class="btn btn-secondary">← Admin Panel</a>
            </div>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success">
                <span>✅</span>
                <span><?php echo sanitizeOutput($success); ?></span>
            </div>
        <?php endif; ?>

        <!-- Template List View -->
        <div id="template-list-view">
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">Available Templates (<?php echo count($templates); ?>)</h2>
                </div>

                <?php if (empty($templates)): ?>
                    <div style="text-align: center; padding: 4rem 2rem;">
                        <div style="font-size: 4rem; margin-bottom: 1rem;">📄</div>
                        <h3 style="margin-bottom: 1rem;">No Templates Yet</h3>
                        <p style="color: var(--text-muted); margin-bottom: 2rem;">
                            Create your first PDF template to customize report exports
                        </p>
                        <button onclick="showDesigner()" class="btn btn-primary">Create First Template</button>
                    </div>
                <?php else: ?>
                    <div class="template-list" style="padding: 1.5rem;">
                        <?php foreach ($templates as $template): ?>
                            <div class="template-card <?php echo $template['is_default'] ? 'default' : ''; ?>">
                                <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 1rem;">
                                    <div>
                                        <h3 style="color: var(--text-primary); margin-bottom: 0.5rem;">
                                            <?php echo sanitizeOutput($template['template_name']); ?>
                                            <?php if ($template['is_default']): ?>
                                                <span class="badge badge-success" style="font-size: 0.7rem; margin-left: 0.5rem;">DEFAULT</span>
                                            <?php endif; ?>
                                        </h3>
                                        <p style="color: var(--text-muted); font-size: 0.85rem;">
                                            <?php echo sanitizeOutput($template['description']); ?>
                                        </p>
                                    </div>
                                    <?php if (!$template['is_active']): ?>
                                        <span class="badge" style="background: rgba(148, 163, 184, 0.2);">Inactive</span>
                                    <?php endif; ?>
                                </div>
                                
                                <div style="display: flex; gap: 1rem; color: var(--text-muted); font-size: 0.85rem; margin-bottom: 1rem;">
                                    <span>📄 <?php echo strtoupper($template['page_size']); ?></span>
                                    <span>📐 <?php echo ucfirst($template['orientation']); ?></span>
                                    <span>📋 <?php echo ucfirst($template['template_type']); ?></span>
                                </div>
                                
                                <div style="display: flex; gap: 0.5rem;">
                                    <button onclick='editTemplate(<?php echo $template['id']; ?>)' 
                                            class="btn btn-secondary" style="flex: 1; padding: 0.5rem;">
                                        ✏️ Edit
                                    </button>
                                    <button onclick='duplicateTemplate(<?php echo $template['id']; ?>)' 
                                            class="btn btn-secondary" style="padding: 0.5rem;">
                                        📋 Copy
                                    </button>
                                    <?php if (!$template['is_default']): ?>
                                        <form method="POST" style="display: inline;" 
                                              onsubmit="return confirm('Delete this template?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="template_id" value="<?php echo $template['id']; ?>">
                                            <button type="submit" class="btn btn-danger" style="padding: 0.5rem;">
                                                🗑️
                                            </button>
                                        </form>
                                    <?php endif; ?>
                                </div>
                                
                                <div style="margin-top: 1rem; padding-top: 1rem; border-top: 1px solid var(--border-color); font-size: 0.75rem; color: var(--text-muted);">
                                    Created by <?php echo sanitizeOutput($template['created_by_name'] ?? 'Unknown'); ?> on 
                                    <?php echo date('d/m/Y', strtotime($template['created_at'])); ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Template Designer View -->
        <div id="template-designer-view" style="display: none;">
            <div class="card mb-3">
                <div class="card-header" style="display: flex; justify-content: space-between; align-items: center;">
                    <h2 class="card-title">Template Configuration</h2>
                    <button onclick="hideDesigner()" class="btn btn-secondary">← Back to List</button>
                </div>

                <div style="padding: 1.5rem;">
                    <form id="template-form" method="POST">
                        <input type="hidden" name="action" value="save_template">
                        <input type="hidden" name="template_id" id="template_id">
                        <input type="hidden" name="template_config" id="template_config">

                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label required">Template Name</label>
                                <input type="text" name="template_name" id="template_name" class="form-control" required>
                            </div>

                            <div class="form-group">
                                <label class="form-label">Description</label>
                                <input type="text" name="description" id="description" class="form-control">
                            </div>
                        </div>

                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Template Type</label>
                                <select name="template_type" id="template_type" class="form-select">
                                    <option value="assessment">Assessment Report</option>
                                    <option value="location">Location Report</option>
                                    <option value="summary">Summary Report</option>
                                    <option value="custom">Custom Report</option>
                                </select>
                            </div>

                            <div class="form-group">
                                <label class="form-label">Page Size</label>
                                <select name="page_size" id="page_size" class="form-select">
                                    <option value="A4">A4</option>
                                    <option value="Letter">Letter</option>
                                    <option value="Legal">Legal</option>
                                    <option value="A3">A3</option>
                                </select>
                            </div>

                            <div class="form-group">
                                <label class="form-label">Orientation</label>
                                <select name="orientation" id="orientation" class="form-select">
                                    <option value="portrait">Portrait</option>
                                    <option value="landscape">Landscape</option>
                                </select>
                            </div>
                        </div>

                        <div class="form-row">
                            <div class="form-group">
                                <div style="display: flex; align-items: center; gap: 0.5rem;">
                                    <input type="checkbox" name="is_default" id="is_default" style="width: auto;">
                                    <label for="is_default" style="margin: 0;">Set as Default Template</label>
                                </div>
                            </div>

                            <div class="form-group">
                                <div style="display: flex; align-items: center; gap: 0.5rem;">
                                    <input type="checkbox" name="is_active" id="is_active" checked style="width: auto;">
                                    <label for="is_active" style="margin: 0;">Active</label>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Designer Layout -->
            <div class="designer-layout">
                <!-- Available Fields -->
                <div class="fields-panel">
                    <h3 style="color: var(--text-primary); margin-bottom: 1rem;">Available Fields</h3>
                    <p style="color: var(--text-muted); font-size: 0.85rem; margin-bottom: 1.5rem;">
                        Drag fields to the canvas to add them to your template
                    </p>
                    
                    <?php foreach ($fields_by_table as $table => $fields): ?>
                        <div class="field-group">
                            <div class="field-group-title">
                                <?php echo ucwords(str_replace('_', ' ', $table)); ?>
                            </div>
                            <?php foreach ($fields as $field): ?>
                                <div class="field-item" draggable="true" 
                                     data-table="<?php echo $field['table_name']; ?>"
                                     data-field="<?php echo $field['field_name']; ?>"
                                     data-label="<?php echo $field['field_label']; ?>"
                                     data-type="<?php echo $field['field_type']; ?>">
                                    <div class="field-item-name"><?php echo sanitizeOutput($field['field_label']); ?></div>
                                    <div class="field-item-type"><?php echo $field['field_type']; ?></div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Canvas -->
                <div class="canvas-panel">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem;">
                        <h3 style="color: var(--text-primary);">Template Canvas</h3>
                        <div class="btn-group">
                            <button onclick="previewTemplate()" class="btn btn-secondary" style="padding: 0.5rem 1rem;">
                                👁️ Preview
                            </button>
                            <button onclick="saveTemplate()" class="btn btn-primary" style="padding: 0.5rem 1rem;">
                                💾 Save Template
                            </button>
                        </div>
                    </div>
                    
                    <div id="canvas-area" class="canvas-area">
                        <div style="text-align: center; color: var(--text-muted); padding: 4rem 2rem;">
                            <div style="font-size: 3rem; margin-bottom: 1rem;">📄</div>
                            <p>Drag and drop fields here to build your template</p>
                        </div>
                    </div>
                </div>

                <!-- Properties -->
                <div class="properties-panel">
                    <h3 style="color: var(--text-primary); margin-bottom: 1rem;">Element Properties</h3>
                    <p style="color: var(--text-muted); font-size: 0.85rem; margin-bottom: 1.5rem;">
                        Select an element to edit its properties
                    </p>
                    
                    <div id="properties-content">
                        <!-- Properties will be loaded here when an element is selected -->
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="js/app.js"></script>
    <script src="js/pdf-designer.js"></script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
