<?php
require_once '../includes/config.php';
requireLogin();

// Fetch locations for dropdown
$stmt = $pdo->query("SELECT id, location_name, city, postcode FROM locations ORDER BY location_name");
$locations = $stmt->fetchAll();

// Check if locations exist
$no_locations = empty($locations);

// Fetch vehicle types
$stmt = $pdo->query("SELECT id, type_name FROM vehicle_types WHERE is_active = 1 ORDER BY display_order");
$vehicle_types = $stmt->fetchAll();

// Fetch delivery volumes
$stmt = $pdo->query("SELECT id, volume_name FROM delivery_volumes WHERE is_active = 1 ORDER BY display_order");
$delivery_volumes = $stmt->fetchAll();

// Fetch risk categories and items
$stmt = $pdo->query("
    SELECT rc.id as category_id, rc.category_name, ri.id as item_id, ri.item_name, ri.description
    FROM risk_categories rc
    JOIN risk_items ri ON rc.id = ri.category_id
    WHERE rc.is_active = 1 AND ri.is_active = 1
    ORDER BY rc.display_order, ri.display_order
");
$risk_items = $stmt->fetchAll();

// Group by category
$grouped_risks = [];
foreach ($risk_items as $item) {
    $grouped_risks[$item['category_name']][] = $item;
}

// Generate unique assessment number
function generateUniqueAssessmentNumber($pdo) {
    $max_attempts = 10;
    $attempt = 0;
    
    while ($attempt < $max_attempts) {
        $assessment_number = 'RA-' . date('Y') . '-' . strtoupper(substr(md5(uniqid(mt_rand(), true)), 0, 6));
        
        // Check if this number already exists
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM assessments WHERE assessment_number = ?");
        $stmt->execute([$assessment_number]);
        
        if ($stmt->fetchColumn() == 0) {
            // Number is unique
            return $assessment_number;
        }
        
        $attempt++;
    }
    
    // Fallback: use timestamp-based number if all attempts fail
    return 'RA-' . date('Y') . '-' . strtoupper(substr(md5(microtime(true) . mt_rand()), 0, 6));
}

$assessment_number = generateUniqueAssessmentNumber($pdo);

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $pdo->beginTransaction();
        
        // If we get a duplicate entry error, regenerate the number and retry
        $max_retries = 3;
        $retry = 0;
        $success = false;
        $assessment_id = null;
        
        while ($retry < $max_retries && !$success) {
            try {
                // Use the submitted number on first try, regenerate on retry
                $number_to_use = ($retry == 0) ? $_POST['assessment_number'] : generateUniqueAssessmentNumber($pdo);
                
                // Insert assessment
                $stmt = $pdo->prepare("
                    INSERT INTO assessments (assessment_number, location_id, assessor_id, assessment_date, 
                                           assessment_time, weather_conditions, vehicle_type, delivery_volume,
                                           overall_risk_level, status, notes, recommendations)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $number_to_use,
                    $_POST['location_id'],
                    $_SESSION['user_id'],
                    $_POST['assessment_date'],
                    $_POST['assessment_time'],
                    $_POST['weather_conditions'] ?? null,
                    $_POST['vehicle_type'] ?? null,
                    $_POST['delivery_volume'] ?? null,
                    $_POST['overall_risk_level'],
                    $_POST['status'] ?? 'draft',
                    $_POST['notes'] ?? null,
                    $_POST['recommendations'] ?? null
                ]);
                
                $assessment_id = $pdo->lastInsertId();
                $success = true;
                
            } catch (PDOException $e) {
                // Check if it's a duplicate entry error
                if ($e->getCode() == 23000 && strpos($e->getMessage(), 'Duplicate entry') !== false) {
                    $retry++;
                    if ($retry >= $max_retries) {
                        throw new Exception('Unable to generate unique assessment number after multiple attempts. Please try again.');
                    }
                    // Continue loop to retry
                } else {
                    // Different error, throw it
                    throw $e;
                }
            }
        }
        
        // Insert assessment details
        if (isset($_POST['risk_items'])) {
            $stmt = $pdo->prepare("
                INSERT INTO assessment_details (assessment_id, risk_item_id, likelihood, severity,
                                               current_controls, additional_actions, action_owner, target_date)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            foreach ($_POST['risk_items'] as $item_id => $data) {
                if (!empty($data['likelihood']) && !empty($data['severity'])) {
                    $stmt->execute([
                        $assessment_id,
                        $item_id,
                        $data['likelihood'],
                        $data['severity'],
                        $data['current_controls'] ?? null,
                        $data['additional_actions'] ?? null,
                        $data['action_owner'] ?? null,
                        !empty($data['target_date']) ? $data['target_date'] : null
                    ]);
                }
            }
        }
        
        $pdo->commit();
        
        // Handle photo uploads if any
        if (!empty($_POST['uploaded_photos'])) {
            $photoData = json_decode($_POST['uploaded_photos'], true);
            if ($photoData) {
                $upload_dir = __DIR__ . '/../uploads/photos/';
                if (!file_exists($upload_dir)) {
                    mkdir($upload_dir, 0755, true);
                }
                
                foreach ($photoData as $index => $photo) {
                    // Photo data URLs are temporarily stored - in production, these would be uploaded separately
                    // For now, we'll just store the caption and placeholder
                    try {
                        $stmt = $pdo->prepare("
                            INSERT INTO photos (assessment_id, filename, original_filename, file_path, 
                                              file_size, mime_type, caption, display_order, uploaded_by)
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                        ");
                        
                        // Generate placeholder filename
                        $filename = 'pending_' . $assessment_id . '_' . $index . '.jpg';
                        
                        $stmt->execute([
                            $assessment_id,
                            $filename,
                            $photo['file'] ?? 'photo.jpg',
                            'uploads/photos/' . $filename,
                            0, // size unknown
                            'image/jpeg',
                            $photo['caption'] ?? '',
                            $index,
                            $_SESSION['user_id']
                        ]);
                    } catch (Exception $e) {
                        error_log('Photo save error: ' . $e->getMessage());
                    }
                }
            }
        }
        
        $_SESSION['success_message'] = 'Assessment created successfully!';
        header('Location: view-assessment.php?id=' . $assessment_id);
        exit;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = 'Error creating assessment: ' . $e->getMessage();
        error_log($error);
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>New Assessment - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .risk-item-card {
            background: var(--bg-tertiary);
            border: 1px solid var(--border-color);
            border-radius: 8px;
            padding: 1.5rem;
            margin-bottom: 1rem;
        }
        
        .risk-rating {
            display: grid;
            grid-template-columns: repeat(5, 1fr);
            gap: 0.5rem;
            margin-top: 0.5rem;
        }
        
        .risk-rating input[type="radio"] {
            display: none;
        }
        
        .risk-rating label {
            padding: 0.75rem;
            background: var(--bg-secondary);
            border: 2px solid var(--border-color);
            border-radius: 6px;
            text-align: center;
            cursor: pointer;
            transition: var(--transition);
            font-weight: 600;
        }
        
        .risk-rating input[type="radio"]:checked + label {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }
        
        .risk-rating label:hover {
            border-color: var(--primary);
        }
        
        .calculated-risk {
            padding: 1rem;
            background: var(--bg-card);
            border-radius: 6px;
            margin-top: 1rem;
            border: 2px solid var(--border-color);
        }
    </style>
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <h1 class="page-title">New Risk Assessment</h1>
            <p class="page-subtitle">Create a comprehensive safety assessment for delivery point</p>
        </div>

        <?php if (isset($error)): ?>
            <div class="alert alert-error">
                <span>❌</span>
                <span><?php echo sanitizeOutput($error); ?></span>
            </div>
        <?php endif; ?>

        <?php if ($no_locations): ?>
            <div class="alert" style="background: #fff3cd; border-left: 4px solid #ffc107; color: #856404;">
                <span>⚠️</span>
                <span>
                    <strong>No locations found!</strong> You need to add at least one delivery location before creating an assessment.
                    <a href="add-location.php" style="color: #856404; text-decoration: underline; font-weight: 600;">
                        Click here to add a location
                    </a>
                </span>
            </div>
        <?php endif; ?>

        <form method="POST" id="assessment-form" data-offline-support data-entity-type="assessment"
              <?php if ($no_locations): ?>disabled<?php endif; ?>>
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">Basic Information</h2>
                    <span id="autosave-indicator" style="display: none; color: var(--success); font-size: 0.9rem;"></span>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label required" for="assessment_number">Assessment Number</label>
                        <input type="text" id="assessment_number" name="assessment_number" class="form-control" 
                               value="<?php echo $assessment_number; ?>" readonly required>
                    </div>

                    <div class="form-group">
                        <label class="form-label required" for="location_id">Delivery Location</label>
                        <select id="location_id" name="location_id" class="form-select" required data-autosave>
                            <option value="">Select location...</option>
                            <?php foreach ($locations as $location): ?>
                                <option value="<?php echo $location['id']; ?>">
                                    <?php echo sanitizeOutput($location['location_name'] . ' - ' . $location['city'] . ', ' . $location['postcode']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label required" for="assessment_date">Assessment Date</label>
                        <input type="date" id="assessment_date" name="assessment_date" class="form-control" 
                               value="<?php echo date('Y-m-d'); ?>" required data-autosave>
                    </div>

                    <div class="form-group">
                        <label class="form-label required" for="assessment_time">Assessment Time</label>
                        <input type="time" id="assessment_time" name="assessment_time" class="form-control" 
                               value="<?php echo date('H:i'); ?>" required data-autosave>
                    </div>
                </div>

                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label" for="weather_conditions">Weather Conditions</label>
                        <select id="weather_conditions" name="weather_conditions" class="form-select" data-autosave>
                            <option value="">Select...</option>
                            <option value="Clear/Dry">Clear/Dry</option>
                            <option value="Overcast">Overcast</option>
                            <option value="Light Rain">Light Rain</option>
                            <option value="Heavy Rain">Heavy Rain</option>
                            <option value="Fog">Fog</option>
                            <option value="Ice/Snow">Ice/Snow</option>
                            <option value="High Wind">High Wind</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="vehicle_type">Vehicle Type</label>
                        <select id="vehicle_type" name="vehicle_type" class="form-select" data-autosave>
                            <option value="">Select...</option>
                            <?php foreach ($vehicle_types as $type): ?>
                                <option value="<?php echo sanitizeOutput($type['type_name']); ?>">
                                    <?php echo sanitizeOutput($type['type_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label class="form-label" for="delivery_volume">Delivery Volume</label>
                        <select id="delivery_volume" name="delivery_volume" class="form-select" data-autosave>
                            <option value="">Select...</option>
                            <?php foreach ($delivery_volumes as $volume): ?>
                                <option value="<?php echo sanitizeOutput($volume['volume_name']); ?>">
                                    <?php echo sanitizeOutput($volume['volume_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
            </div>

            <!-- Risk Assessment Items -->
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">Risk Assessment Details</h2>
                    <p style="color: var(--text-muted); font-size: 0.9rem; margin-top: 0.5rem;">
                        Rate each risk item by selecting likelihood (1-5) and severity (1-5)
                    </p>
                </div>

                <?php foreach ($grouped_risks as $category => $items): ?>
                    <div style="margin-bottom: 2rem;">
                        <h3 style="color: var(--primary); font-size: 1.3rem; margin-bottom: 1rem; padding: 0.5rem 0; border-bottom: 2px solid var(--primary);">
                            📋 <?php echo sanitizeOutput($category); ?>
                        </h3>

                        <?php foreach ($items as $item): ?>
                            <div class="risk-item-card" id="risk-item-<?php echo $item['item_id']; ?>">
                                <h4 style="font-size: 1.1rem; margin-bottom: 0.5rem; color: var(--text-primary);">
                                    <?php echo sanitizeOutput($item['item_name']); ?>
                                </h4>
                                <p style="color: var(--text-muted); font-size: 0.9rem; margin-bottom: 1rem;">
                                    <?php echo sanitizeOutput($item['description']); ?>
                                </p>

                                <div class="form-row" style="grid-template-columns: 1fr 1fr;">
                                    <div class="form-group">
                                        <label class="form-label">Likelihood (1=Rare, 5=Almost Certain)</label>
                                        <div class="risk-rating">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <input type="radio" id="likelihood_<?php echo $item['item_id']; ?>_<?php echo $i; ?>" 
                                                       name="risk_items[<?php echo $item['item_id']; ?>][likelihood]" 
                                                       value="<?php echo $i; ?>"
                                                       onchange="calculateRisk(<?php echo $item['item_id']; ?>)">
                                                <label for="likelihood_<?php echo $item['item_id']; ?>_<?php echo $i; ?>"><?php echo $i; ?></label>
                                            <?php endfor; ?>
                                        </div>
                                    </div>

                                    <div class="form-group">
                                        <label class="form-label">Severity (1=Negligible, 5=Catastrophic)</label>
                                        <div class="risk-rating">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <input type="radio" id="severity_<?php echo $item['item_id']; ?>_<?php echo $i; ?>" 
                                                       name="risk_items[<?php echo $item['item_id']; ?>][severity]" 
                                                       value="<?php echo $i; ?>"
                                                       onchange="calculateRisk(<?php echo $item['item_id']; ?>)">
                                                <label for="severity_<?php echo $item['item_id']; ?>_<?php echo $i; ?>"><?php echo $i; ?></label>
                                            <?php endfor; ?>
                                        </div>
                                    </div>
                                </div>

                                <div class="calculated-risk" id="risk-result-<?php echo $item['item_id']; ?>" style="display: none;">
                                    <div style="display: flex; justify-content: space-between; align-items: center;">
                                        <span><strong>Risk Score:</strong> <span id="score-<?php echo $item['item_id']; ?>"></span></span>
                                        <span class="badge" id="level-<?php echo $item['item_id']; ?>"></span>
                                    </div>
                                </div>

                                <div class="form-group mt-2">
                                    <label class="form-label" for="controls_<?php echo $item['item_id']; ?>">Current Controls</label>
                                    <textarea id="controls_<?php echo $item['item_id']; ?>" 
                                              name="risk_items[<?php echo $item['item_id']; ?>][current_controls]" 
                                              class="form-textarea" rows="2"
                                              placeholder="Describe existing safety measures..."></textarea>
                                </div>

                                <div class="form-group">
                                    <label class="form-label" for="actions_<?php echo $item['item_id']; ?>">Additional Actions Required</label>
                                    <textarea id="actions_<?php echo $item['item_id']; ?>" 
                                              name="risk_items[<?php echo $item['item_id']; ?>][additional_actions]" 
                                              class="form-textarea" rows="2"
                                              placeholder="What further actions are needed?"></textarea>
                                </div>

                                <div class="form-row">
                                    <div class="form-group">
                                        <label class="form-label" for="owner_<?php echo $item['item_id']; ?>">Action Owner</label>
                                        <input type="text" id="owner_<?php echo $item['item_id']; ?>" 
                                               name="risk_items[<?php echo $item['item_id']; ?>][action_owner]" 
                                               class="form-control" placeholder="Who is responsible?">
                                    </div>

                                    <div class="form-group">
                                        <label class="form-label" for="target_<?php echo $item['item_id']; ?>">Target Date</label>
                                        <input type="date" id="target_<?php echo $item['item_id']; ?>" 
                                               name="risk_items[<?php echo $item['item_id']; ?>][target_date]" 
                                               class="form-control">
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endforeach; ?>
            </div>

            <!-- Summary and Notes -->
            <div class="card mb-3">
                <div class="card-header">
                    <h2 class="card-title">Summary</h2>
                </div>

                <div class="form-group">
                    <label class="form-label required" for="overall_risk_level">Overall Risk Level</label>
                    <select id="overall_risk_level" name="overall_risk_level" class="form-select" required>
                        <option value="">Select overall risk...</option>
                        <option value="low">Low</option>
                        <option value="medium">Medium</option>
                        <option value="high">High</option>
                        <option value="critical">Critical</option>
                    </select>
                </div>

                <!-- Photo Upload Section -->
                <div class="form-group">
                    <label class="form-label">📷 Assessment Photos</label>
                    <p style="color: var(--text-muted); font-size: 0.9rem; margin-bottom: 1rem;">
                        Upload photos of the delivery point with captions
                    </p>
                    
                    <input type="file" 
                           id="photo-upload" 
                           accept="image/*" 
                           multiple 
                           class="form-control"
                           onchange="handlePhotoSelect(event)"
                           style="padding: 0.75rem;">
                    <small style="color: var(--text-muted); display: block; margin-top: 0.5rem;">
                        JPG, PNG, GIF, WebP | Max 10MB per photo | Add captions after upload
                    </small>
                </div>

                <div id="photo-preview-container" style="margin-top: 1rem;">
                    <!-- Photo previews will appear here -->
                </div>

                <input type="hidden" id="uploaded-photos" name="uploaded_photos" value="">

                <div class="form-group">
                    <label class="form-label" for="notes">General Notes</label>
                    <textarea id="notes" name="notes" class="form-textarea" rows="4"
                              placeholder="Any additional observations or notes..."></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label" for="recommendations">Recommendations</label>
                    <textarea id="recommendations" name="recommendations" class="form-textarea" rows="4"
                              placeholder="Overall recommendations for this delivery point..."></textarea>
                </div>

                <div class="form-group">
                    <label class="form-label required" for="status">Status</label>
                    <select id="status" name="status" class="form-select" required>
                        <option value="draft">Draft</option>
                        <option value="completed">Completed</option>
                    </select>
                </div>
            </div>

            <div class="btn-group">
                <button type="submit" class="btn btn-primary">💾 Save Assessment</button>
                <button type="button" class="btn btn-secondary" onclick="window.location.href='dashboard.php'">Cancel</button>
            </div>
        </form>
    </div>

    <div id="offline-indicator" class="offline-indicator">
        <span>📡</span>
        <span>You are offline. Assessment will be saved locally.</span>
    </div>

    <script src="js/app.js"></script>
    <script>
        function calculateRisk(itemId) {
            const likelihoodInput = document.querySelector(`input[name="risk_items[${itemId}][likelihood]"]:checked`);
            const severityInput = document.querySelector(`input[name="risk_items[${itemId}][severity]"]:checked`);
            
            if (likelihoodInput && severityInput) {
                const likelihood = parseInt(likelihoodInput.value);
                const severity = parseInt(severityInput.value);
                const score = likelihood * severity;
                
                let level = 'low';
                if (score > 16) level = 'critical';
                else if (score > 9) level = 'high';
                else if (score > 4) level = 'medium';
                
                const resultDiv = document.getElementById(`risk-result-${itemId}`);
                const scoreSpan = document.getElementById(`score-${itemId}`);
                const levelSpan = document.getElementById(`level-${itemId}`);
                
                resultDiv.style.display = 'block';
                scoreSpan.textContent = score;
                levelSpan.textContent = level.toUpperCase();
                levelSpan.className = `badge badge-${level}`;
            }
        }

        // Restore draft if exists
        window.addEventListener('DOMContentLoaded', () => {
            window.app.restoreDraft('assessment-form');
        });

        // Photo upload handling
        let uploadedPhotos = [];

        function handlePhotoSelect(event) {
            const files = Array.from(event.target.files);
            const container = document.getElementById('photo-preview-container');
            
            files.forEach((file, index) => {
                if (file.size > 10 * 1024 * 1024) {
                    window.app.showAlert(`File ${file.name} is too large. Maximum size is 10MB.`, 'error');
                    return;
                }
                
                if (!file.type.startsWith('image/')) {
                    window.app.showAlert(`File ${file.name} is not an image.`, 'error');
                    return;
                }
                
                const reader = new FileReader();
                reader.onload = function(e) {
                    const photoId = 'photo-' + Date.now() + '-' + index;
                    const photoData = {
                        id: photoId,
                        file: file,
                        dataUrl: e.target.result,
                        caption: ''
                    };
                    
                    uploadedPhotos.push(photoData);
                    addPhotoPreview(photoData);
                };
                reader.readAsDataURL(file);
            });
            
            // Clear input
            event.target.value = '';
        }

        function addPhotoPreview(photo) {
            const container = document.getElementById('photo-preview-container');
            
            const photoDiv = document.createElement('div');
            photoDiv.id = photo.id;
            photoDiv.className = 'photo-preview-item';
            photoDiv.style.cssText = `
                display: flex;
                gap: 1rem;
                padding: 1rem;
                background: var(--bg-tertiary);
                border-radius: 8px;
                margin-bottom: 1rem;
                border: 1px solid var(--border-color);
            `;
            
            photoDiv.innerHTML = `
                <div style="flex-shrink: 0;">
                    <img src="${photo.dataUrl}" 
                         alt="Preview" 
                         style="width: 150px; height: 150px; object-fit: cover; border-radius: 6px; border: 2px solid var(--border-color);">
                </div>
                <div style="flex-grow: 1;">
                    <div class="form-group" style="margin-bottom: 0.5rem;">
                        <label class="form-label" style="margin-bottom: 0.5rem;">Photo Caption / Description</label>
                        <textarea 
                            class="form-textarea" 
                            rows="3" 
                            placeholder="Describe what this photo shows (e.g., 'Loading bay entrance showing height restriction', 'Uneven surface near delivery area')..."
                            onchange="updatePhotoCaption('${photo.id}', this.value)"
                            style="resize: vertical;"
                        ></textarea>
                    </div>
                    <div style="display: flex; gap: 0.5rem; align-items: center;">
                        <span style="color: var(--text-muted); font-size: 0.85rem;">
                            ${formatFileSize(photo.file.size)} • ${photo.file.name}
                        </span>
                    </div>
                </div>
                <div style="flex-shrink: 0;">
                    <button type="button" 
                            class="btn btn-danger" 
                            onclick="removePhoto('${photo.id}')"
                            style="padding: 0.5rem 1rem;">
                        🗑️ Remove
                    </button>
                </div>
            `;
            
            container.appendChild(photoDiv);
        }

        function updatePhotoCaption(photoId, caption) {
            const photo = uploadedPhotos.find(p => p.id === photoId);
            if (photo) {
                photo.caption = caption;
            }
        }

        function removePhoto(photoId) {
            uploadedPhotos = uploadedPhotos.filter(p => p.id !== photoId);
            const element = document.getElementById(photoId);
            if (element) {
                element.remove();
            }
        }

        function formatFileSize(bytes) {
            if (bytes < 1024) return bytes + ' B';
            if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
            return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
        }

        // Intercept form submission to upload photos first
        document.getElementById('assessment-form').addEventListener('submit', async function(e) {
            if (uploadedPhotos.length > 0) {
                e.preventDefault();
                
                // Show uploading message
                const submitBtn = this.querySelector('button[type="submit"]');
                const originalText = submitBtn.textContent;
                submitBtn.disabled = true;
                submitBtn.textContent = '📤 Uploading photos...';
                
                try {
                    // Wait for photos to upload
                    // Note: Photos will be associated with assessment after it's created
                    // Store photo data temporarily
                    const photoData = uploadedPhotos.map(p => ({
                        file: p.file.name,
                        caption: p.caption,
                        dataUrl: p.dataUrl
                    }));
                    
                    document.getElementById('uploaded-photos').value = JSON.stringify(photoData);
                    
                    // Now submit the form
                    submitBtn.textContent = '💾 Saving assessment...';
                    this.submit();
                    
                } catch (error) {
                    console.error('Photo upload error:', error);
                    submitBtn.disabled = false;
                    submitBtn.textContent = originalText;
                    window.app.showAlert('Error uploading photos. Please try again.', 'error');
                }
            }
        });
    </script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
