<?php
require_once '../includes/config.php';

// Redirect if already logged in
if (isLoggedIn()) {
    header('Location: dashboard.php');
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';

    if (empty($username) || empty($password)) {
        $error = 'Please enter both username and password';
    } else {
        try {
            $stmt = $pdo->prepare("SELECT id, username, password, full_name, role, is_active FROM users WHERE username = ? OR email = ?");
            $stmt->execute([$username, $username]);
            $user = $stmt->fetch();

            if ($user && $user['is_active'] && password_verify($password, $user['password'])) {
                // Successful login
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['full_name'] = $user['full_name'];
                $_SESSION['role'] = $user['role'];
                $_SESSION['csrf_token'] = generateToken();

                // Update last login
                $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                $stmt->execute([$user['id']]);

                // Log the login (if table exists)
                try {
                    $stmt = $pdo->prepare("INSERT INTO audit_log (user_id, action, ip_address, user_agent) VALUES (?, 'login', ?, ?)");
                    $stmt->execute([$user['id'], $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);
                } catch (Exception $e) {
                    // Audit log table doesn't exist - skip logging
                }

                header('Location: dashboard.php');
                exit;
            } else {
                $error = 'Invalid username or password';
                
                // Log failed attempt (if table exists)
                try {
                    $stmt = $pdo->prepare("INSERT INTO audit_log (action, old_values, ip_address, user_agent) VALUES ('failed_login', ?, ?, ?)");
                    $stmt->execute([$username, $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);
                } catch (Exception $e) {
                    // Audit log table doesn't exist - skip logging
                }
            }
        } catch (PDOException $e) {
            $error = 'An error occurred. Please try again.';
            error_log($e->getMessage());
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
    <style>
        .login-container {
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }
        
        .login-box {
            background: var(--bg-card);
            border-radius: 16px;
            padding: 3rem;
            max-width: 450px;
            width: 100%;
            box-shadow: var(--shadow-lg);
            border: 1px solid var(--border-color);
        }
        
        .login-header {
            text-align: center;
            margin-bottom: 2rem;
        }
        
        .login-logo {
            width: 80px;
            height: 80px;
            margin: 0 auto 1.5rem;
            background: linear-gradient(135deg, var(--primary) 0%, var(--warning) 100%);
            border-radius: 16px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 3rem;
        }
        
        .login-title {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            background: linear-gradient(135deg, var(--primary-light) 0%, var(--primary) 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .login-subtitle {
            color: var(--text-muted);
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-box">
            <div class="login-header">
                <div class="login-logo">⚠️</div>
                <h1 class="login-title">Risk Assessment</h1>
                <p class="login-subtitle">Supermarket Delivery Safety System</p>
            </div>

            <?php if ($error): ?>
                <div class="alert alert-error">
                    <span>❌</span>
                    <span><?php echo sanitizeOutput($error); ?></span>
                </div>
            <?php endif; ?>

            <form method="POST" action="">
                <div class="form-group">
                    <label class="form-label required" for="username">Username or Email</label>
                    <input 
                        type="text" 
                        id="username" 
                        name="username" 
                        class="form-control" 
                        required 
                        autofocus
                        placeholder="Enter your username or email"
                        value="<?php echo sanitizeOutput($_POST['username'] ?? ''); ?>"
                    >
                </div>

                <div class="form-group">
                    <label class="form-label required" for="password">Password</label>
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        class="form-control" 
                        required
                        placeholder="Enter your password"
                    >
                </div>

                <button type="submit" class="btn btn-primary" style="width: 100%;">
                    🔐 Sign In
                </button>
            </form>

            <div style="margin-top: 2rem; padding-top: 2rem; border-top: 1px solid var(--border-color); text-align: center; color: var(--text-muted); font-size: 0.9rem;">
                <p><strong>Default Credentials:</strong></p>
                <p>Username: <code>admin</code></p>
                <p>Password: <code>Admin123!</code></p>
            </div>
        </div>
    </div>

    <script src="js/app.js"></script>
</body>
</html>
