/**
 * Theme Switcher JavaScript
 * Handles light/dark theme switching with localStorage persistence
 */

(function() {
    'use strict';
    
    const THEME_KEY = 'risk-assessment-theme';
    const THEME_LIGHT = 'light';
    const THEME_DARK = 'dark';
    
    /**
     * Theme Manager Class
     */
    class ThemeManager {
        constructor() {
            this.currentTheme = this.getStoredTheme() || this.getPreferredTheme();
            this.init();
        }
        
        /**
         * Initialize theme manager
         */
        init() {
            // Apply theme immediately (before page renders)
            this.applyTheme(this.currentTheme, false);
            
            // Wait for DOM to be ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', () => this.setupUI());
            } else {
                this.setupUI();
            }
            
            // Listen for system theme changes
            this.watchSystemTheme();
        }
        
        /**
         * Setup UI elements
         */
        setupUI() {
            // Create theme toggle button if it doesn't exist
            if (!document.querySelector('.theme-toggle')) {
                this.createToggleButton();
            }
            
            // Bind existing toggle buttons
            this.bindToggleButtons();
            
            // Update toggle button state
            this.updateToggleButtons();
        }
        
        /**
         * Create floating theme toggle button
         */
        createToggleButton() {
            const button = document.createElement('button');
            button.className = 'theme-toggle';
            button.setAttribute('aria-label', 'Toggle theme');
            button.setAttribute('title', 'Switch theme');
            button.innerHTML = `
                <span class="sun-icon">☀️</span>
                <span class="moon-icon">🌙</span>
            `;
            
            button.addEventListener('click', () => this.toggleTheme());
            
            document.body.appendChild(button);
        }
        
        /**
         * Bind click events to all toggle buttons
         */
        bindToggleButtons() {
            const buttons = document.querySelectorAll('.theme-toggle, [data-theme-toggle]');
            buttons.forEach(button => {
                button.addEventListener('click', (e) => {
                    e.preventDefault();
                    this.toggleTheme();
                });
            });
        }
        
        /**
         * Update toggle button states
         */
        updateToggleButtons() {
            const buttons = document.querySelectorAll('.theme-toggle, [data-theme-toggle]');
            buttons.forEach(button => {
                const label = this.currentTheme === THEME_DARK ? 'Switch to light theme' : 'Switch to dark theme';
                button.setAttribute('aria-label', label);
                button.setAttribute('title', label);
            });
        }
        
        /**
         * Toggle between light and dark theme
         */
        toggleTheme() {
            const newTheme = this.currentTheme === THEME_LIGHT ? THEME_DARK : THEME_LIGHT;
            this.setTheme(newTheme);
        }
        
        /**
         * Set specific theme
         * @param {string} theme - 'light' or 'dark'
         */
        setTheme(theme) {
            if (theme !== THEME_LIGHT && theme !== THEME_DARK) {
                console.warn(`Invalid theme: ${theme}. Using light theme.`);
                theme = THEME_LIGHT;
            }
            
            this.currentTheme = theme;
            this.applyTheme(theme, true);
            this.storeTheme(theme);
            this.updateToggleButtons();
            
            // Dispatch custom event
            window.dispatchEvent(new CustomEvent('themeChange', {
                detail: { theme }
            }));
        }
        
        /**
         * Apply theme to document
         * @param {string} theme
         * @param {boolean} animate - Whether to animate the transition
         */
        applyTheme(theme, animate = true) {
            const root = document.documentElement;
            const body = document.body;
            
            // Add transition class for smooth animation
            if (animate) {
                root.classList.add('theme-transitioning');
                setTimeout(() => root.classList.remove('theme-transitioning'), 300);
            }
            
            // Set theme attribute
            root.setAttribute('data-theme', theme);
            body.setAttribute('data-theme', theme);
            
            // Update meta theme-color for mobile browsers
            this.updateMetaThemeColor(theme);
        }
        
        /**
         * Update meta theme-color tag
         * @param {string} theme
         */
        updateMetaThemeColor(theme) {
            let metaTag = document.querySelector('meta[name="theme-color"]');
            
            if (!metaTag) {
                metaTag = document.createElement('meta');
                metaTag.name = 'theme-color';
                document.head.appendChild(metaTag);
            }
            
            const color = theme === THEME_DARK ? '#1a1f2e' : '#667eea';
            metaTag.content = color;
        }
        
        /**
         * Get stored theme from localStorage
         * @returns {string|null}
         */
        getStoredTheme() {
            try {
                return localStorage.getItem(THEME_KEY);
            } catch (e) {
                console.warn('localStorage not available:', e);
                return null;
            }
        }
        
        /**
         * Store theme in localStorage
         * @param {string} theme
         */
        storeTheme(theme) {
            try {
                localStorage.setItem(THEME_KEY, theme);
            } catch (e) {
                console.warn('Could not save theme preference:', e);
            }
        }
        
        /**
         * Get preferred theme from system
         * @returns {string}
         */
        getPreferredTheme() {
            // Check if user prefers dark mode
            if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
                return THEME_DARK;
            }
            return THEME_LIGHT;
        }
        
        /**
         * Watch for system theme changes
         */
        watchSystemTheme() {
            if (!window.matchMedia) return;
            
            const darkModeQuery = window.matchMedia('(prefers-color-scheme: dark)');
            
            // Listen for changes
            const handler = (e) => {
                // Only auto-switch if user hasn't set a preference
                if (!this.getStoredTheme()) {
                    const newTheme = e.matches ? THEME_DARK : THEME_LIGHT;
                    this.setTheme(newTheme);
                }
            };
            
            // Modern browsers
            if (darkModeQuery.addEventListener) {
                darkModeQuery.addEventListener('change', handler);
            } else if (darkModeQuery.addListener) {
                // Legacy browsers
                darkModeQuery.addListener(handler);
            }
        }
        
        /**
         * Get current theme
         * @returns {string}
         */
        getTheme() {
            return this.currentTheme;
        }
        
        /**
         * Check if dark theme is active
         * @returns {boolean}
         */
        isDarkTheme() {
            return this.currentTheme === THEME_DARK;
        }
        
        /**
         * Reset to system preference
         */
        resetToSystemPreference() {
            try {
                localStorage.removeItem(THEME_KEY);
            } catch (e) {
                console.warn('Could not clear theme preference:', e);
            }
            
            const systemTheme = this.getPreferredTheme();
            this.setTheme(systemTheme);
        }
    }
    
    // Add smooth transition CSS
    const style = document.createElement('style');
    style.textContent = `
        .theme-transitioning,
        .theme-transitioning * {
            transition: background-color 0.3s ease, 
                        color 0.3s ease, 
                        border-color 0.3s ease,
                        box-shadow 0.3s ease !important;
        }
    `;
    document.head.appendChild(style);
    
    // Initialize theme manager
    const themeManager = new ThemeManager();
    
    // Expose to global scope
    window.ThemeManager = themeManager;
    
    // Export functions for external use
    window.setTheme = (theme) => themeManager.setTheme(theme);
    window.toggleTheme = () => themeManager.toggleTheme();
    window.getTheme = () => themeManager.getTheme();
    window.isDarkTheme = () => themeManager.isDarkTheme();
    
    // Log initialization
    console.log('Theme Manager initialized. Current theme:', themeManager.getTheme());
    
})();

/**
 * Usage Examples:
 * 
 * // Toggle theme
 * toggleTheme();
 * 
 * // Set specific theme
 * setTheme('dark');
 * setTheme('light');
 * 
 * // Get current theme
 * const theme = getTheme();
 * 
 * // Check if dark theme
 * if (isDarkTheme()) {
 *     console.log('Dark theme is active');
 * }
 * 
 * // Listen for theme changes
 * window.addEventListener('themeChange', (e) => {
 *     console.log('Theme changed to:', e.detail.theme);
 * });
 * 
 * // Reset to system preference
 * ThemeManager.resetToSystemPreference();
 */
