/**
 * Risk Item Photos Component
 * Vue component for managing photos attached to individual risk items
 */

const RiskItemPhotos = {
    props: {
        assessmentId: { type: Number, required: true },
        riskItemId: { type: Number, required: true },
        assessmentDetailId: { type: Number, default: null },
        editable: { type: Boolean, default: true }
    },
    data() {
        return {
            photos: [],
            uploading: false,
            loading: false,
            selectedFile: null,
            caption: '',
            showUploader: false
        };
    },
    template: `
        <div class="risk-item-photos">
            <!-- Upload Section -->
            <div v-if="editable" class="photo-upload-section">
                <button 
                    @click="showUploader = !showUploader" 
                    class="btn btn-secondary"
                    style="width: 100%; margin-bottom: 1rem;"
                >
                    📷 {{ showUploader ? 'Hide' : 'Add Photo' }}
                </button>
                
                <div v-if="showUploader" class="upload-form">
                    <div class="form-group">
                        <input 
                            type="file" 
                            @change="handleFileSelect"
                            accept="image/jpeg,image/jpg,image/png,image/gif,image/webp"
                            class="form-control"
                        >
                        <small style="color: var(--text-muted);">
                            Max 5MB | JPG, PNG, GIF, WebP
                        </small>
                    </div>
                    
                    <div class="form-group">
                        <input 
                            type="text" 
                            v-model="caption"
                            placeholder="Photo caption (optional)"
                            class="form-control"
                        >
                    </div>
                    
                    <button 
                        @click="uploadPhoto"
                        :disabled="!selectedFile || uploading"
                        class="btn btn-primary"
                        style="width: 100%;"
                    >
                        {{ uploading ? '⏳ Uploading...' : '📤 Upload Photo' }}
                    </button>
                </div>
            </div>

            <!-- Photos Grid -->
            <div v-if="loading" style="text-align: center; padding: 1rem;">
                <div class="spinner" style="margin: 0 auto;"></div>
                <p style="color: var(--text-muted); margin-top: 0.5rem;">Loading photos...</p>
            </div>
            
            <div v-else-if="photos.length === 0" style="text-align: center; padding: 2rem; color: var(--text-muted);">
                📷 No photos attached to this risk item
            </div>
            
            <div v-else class="photos-grid">
                <div 
                    v-for="photo in photos" 
                    :key="photo.id"
                    class="photo-item"
                >
                    <img 
                        :src="photo.file_path" 
                        :alt="photo.caption || 'Risk photo'"
                        @click="viewPhoto(photo)"
                    >
                    
                    <div class="photo-info">
                        <div v-if="photo.caption" class="photo-caption">
                            {{ photo.caption }}
                        </div>
                        <div class="photo-meta">
                            <small>{{ formatDate(photo.uploaded_at) }}</small>
                            <small>by {{ photo.uploader_name }}</small>
                        </div>
                    </div>
                    
                    <div v-if="editable" class="photo-actions">
                        <button 
                            @click="deletePhoto(photo.id)"
                            class="btn-icon-danger"
                            title="Delete photo"
                        >
                            🗑️
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `,
    methods: {
        async loadPhotos() {
            this.loading = true;
            try {
                const params = new URLSearchParams({
                    assessment_id: this.assessmentId,
                    risk_item_id: this.riskItemId
                });
                
                const response = await axios.get(`api/risk-item-photos.php?${params}`);
                if (response.data.success) {
                    this.photos = response.data.photos;
                }
            } catch (error) {
                console.error('Error loading photos:', error);
            } finally {
                this.loading = false;
            }
        },
        
        handleFileSelect(event) {
            this.selectedFile = event.target.files[0];
        },
        
        async uploadPhoto() {
            if (!this.selectedFile) return;
            
            this.uploading = true;
            try {
                const formData = new FormData();
                formData.append('photo', this.selectedFile);
                formData.append('assessment_id', this.assessmentId);
                formData.append('risk_item_id', this.riskItemId);
                if (this.assessmentDetailId) {
                    formData.append('assessment_detail_id', this.assessmentDetailId);
                }
                if (this.caption) {
                    formData.append('caption', this.caption);
                }
                
                const response = await axios.post('api/risk-item-photos.php', formData, {
                    headers: { 'Content-Type': 'multipart/form-data' }
                });
                
                if (response.data.success) {
                    // Reset form
                    this.selectedFile = null;
                    this.caption = '';
                    this.showUploader = false;
                    
                    // Reload photos
                    await this.loadPhotos();
                    
                    this.$emit('photo-uploaded', response.data);
                }
            } catch (error) {
                console.error('Error uploading photo:', error);
                alert(error.response?.data?.error || 'Failed to upload photo');
            } finally {
                this.uploading = false;
            }
        },
        
        async deletePhoto(photoId) {
            if (!confirm('Delete this photo?')) return;
            
            try {
                const response = await axios.delete(`api/risk-item-photos.php?id=${photoId}`);
                if (response.data.success) {
                    await this.loadPhotos();
                    this.$emit('photo-deleted', photoId);
                }
            } catch (error) {
                console.error('Error deleting photo:', error);
                alert('Failed to delete photo');
            }
        },
        
        viewPhoto(photo) {
            // Open photo in new window/lightbox
            window.open(photo.file_path, '_blank');
        },
        
        formatDate(dateString) {
            return new Date(dateString).toLocaleDateString('en-GB', {
                day: '2-digit',
                month: 'short',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
        }
    },
    mounted() {
        this.loadPhotos();
    }
};

// Styles
const riskItemPhotosStyles = `
<style>
.risk-item-photos {
    margin-top: 1rem;
}

.photo-upload-section {
    margin-bottom: 1.5rem;
}

.upload-form {
    padding: 1rem;
    background: var(--bg-secondary);
    border-radius: 8px;
    border: 2px dashed var(--border-color);
}

.photos-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
    gap: 1rem;
}

.photo-item {
    position: relative;
    background: var(--bg-secondary);
    border-radius: 8px;
    overflow: hidden;
    transition: var(--transition);
}

.photo-item:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow);
}

.photo-item img {
    width: 100%;
    height: 200px;
    object-fit: cover;
    cursor: pointer;
}

.photo-info {
    padding: 0.75rem;
}

.photo-caption {
    font-size: 0.9rem;
    color: var(--text-primary);
    margin-bottom: 0.5rem;
    font-weight: 500;
}

.photo-meta {
    display: flex;
    justify-content: space-between;
    font-size: 0.75rem;
    color: var(--text-muted);
}

.photo-actions {
    position: absolute;
    top: 0.5rem;
    right: 0.5rem;
}

.btn-icon-danger {
    background: rgba(220, 38, 38, 0.9);
    color: white;
    border: none;
    padding: 0.5rem;
    border-radius: 4px;
    cursor: pointer;
    font-size: 1rem;
    transition: var(--transition);
}

.btn-icon-danger:hover {
    background: rgba(220, 38, 38, 1);
    transform: scale(1.1);
}
</style>
`;

// Add styles to page
if (typeof document !== 'undefined') {
    const styleEl = document.createElement('style');
    styleEl.innerHTML = riskItemPhotosStyles.replace(/<\/?style>/g, '');
    document.head.appendChild(styleEl);
}

// Export for use
window.RiskItemPhotos = RiskItemPhotos;
