/**
 * Responsive Navigation & Mobile Menu
 * Handles mobile menu toggle, touch gestures, and responsive behaviors
 */

(function() {
    'use strict';
    
    // ========================================================================
    // Mobile Menu Toggle
    // ========================================================================
    
    function initMobileMenu() {
        const toggle = document.querySelector('.navbar-toggle');
        const menu = document.querySelector('.navbar-menu');
        const backdrop = document.querySelector('.navbar-backdrop');
        const close = document.querySelector('.navbar-close');
        
        if (!toggle || !menu) return;
        
        // Create backdrop if it doesn't exist
        if (!backdrop) {
            const newBackdrop = document.createElement('div');
            newBackdrop.className = 'navbar-backdrop';
            document.body.appendChild(newBackdrop);
        }
        
        const backdropElement = document.querySelector('.navbar-backdrop');
        
        // Open menu
        toggle.addEventListener('click', function() {
            menu.classList.add('active');
            backdropElement.classList.add('active');
            document.body.style.overflow = 'hidden';
        });
        
        // Close menu
        function closeMenu() {
            menu.classList.remove('active');
            backdropElement.classList.remove('active');
            document.body.style.overflow = '';
        }
        
        if (close) {
            close.addEventListener('click', closeMenu);
        }
        
        if (backdropElement) {
            backdropElement.addEventListener('click', closeMenu);
        }
        
        // Close on escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && menu.classList.contains('active')) {
                closeMenu();
            }
        });
        
        // Close menu on window resize to desktop
        let resizeTimer;
        window.addEventListener('resize', function() {
            clearTimeout(resizeTimer);
            resizeTimer = setTimeout(function() {
                if (window.innerWidth >= 1024) {
                    closeMenu();
                }
            }, 250);
        });
    }
    
    // ========================================================================
    // Touch Swipe for Mobile Menu
    // ========================================================================
    
    function initSwipeGestures() {
        const menu = document.querySelector('.navbar-menu');
        if (!menu) return;
        
        let touchStartX = 0;
        let touchEndX = 0;
        
        menu.addEventListener('touchstart', function(e) {
            touchStartX = e.changedTouches[0].screenX;
        }, { passive: true });
        
        menu.addEventListener('touchend', function(e) {
            touchEndX = e.changedTouches[0].screenX;
            handleSwipe();
        }, { passive: true });
        
        function handleSwipe() {
            // Swipe left to close (threshold: 50px)
            if (touchStartX - touchEndX > 50) {
                menu.classList.remove('active');
                const backdrop = document.querySelector('.navbar-backdrop');
                if (backdrop) backdrop.classList.remove('active');
                document.body.style.overflow = '';
            }
        }
    }
    
    // ========================================================================
    // Responsive Tables
    // ========================================================================
    
    function initResponsiveTables() {
        const tables = document.querySelectorAll('.table-mobile-cards');
        
        tables.forEach(function(table) {
            const headers = table.querySelectorAll('thead th');
            const rows = table.querySelectorAll('tbody tr');
            
            rows.forEach(function(row) {
                const cells = row.querySelectorAll('td');
                cells.forEach(function(cell, index) {
                    if (headers[index]) {
                        cell.setAttribute('data-label', headers[index].textContent);
                    }
                });
            });
        });
    }
    
    // ========================================================================
    // Modal Management
    // ========================================================================
    
    function initModals() {
        // Open modal
        document.addEventListener('click', function(e) {
            if (e.target.matches('[data-modal-open]')) {
                const modalId = e.target.getAttribute('data-modal-open');
                const modal = document.getElementById(modalId);
                if (modal) {
                    openModal(modal);
                }
            }
        });
        
        // Close modal
        document.addEventListener('click', function(e) {
            if (e.target.matches('.modal-close') || e.target.matches('.modal-backdrop')) {
                const modal = e.target.closest('.modal');
                if (modal) {
                    closeModal(modal);
                }
            }
        });
        
        // Close on escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                const activeModal = document.querySelector('.modal.active');
                if (activeModal) {
                    closeModal(activeModal);
                }
            }
        });
    }
    
    function openModal(modal) {
        modal.classList.add('active');
        document.body.style.overflow = 'hidden';
    }
    
    function closeModal(modal) {
        modal.classList.remove('active');
        document.body.style.overflow = '';
    }
    
    // ========================================================================
    // Smooth Scroll
    // ========================================================================
    
    function initSmoothScroll() {
        document.addEventListener('click', function(e) {
            if (e.target.matches('a[href^="#"]')) {
                e.preventDefault();
                const targetId = e.target.getAttribute('href');
                if (targetId === '#') return;
                
                const target = document.querySelector(targetId);
                if (target) {
                    const offset = 80; // Navbar height
                    const targetPosition = target.getBoundingClientRect().top + window.pageYOffset - offset;
                    
                    window.scrollTo({
                        top: targetPosition,
                        behavior: 'smooth'
                    });
                }
            }
        });
    }
    
    // ========================================================================
    // Sticky Elements
    // ========================================================================
    
    function initStickyElements() {
        const stickyElements = document.querySelectorAll('[data-sticky]');
        
        stickyElements.forEach(function(element) {
            const offset = element.getAttribute('data-sticky-offset') || 0;
            
            window.addEventListener('scroll', function() {
                if (window.pageYOffset > offset) {
                    element.classList.add('is-stuck');
                } else {
                    element.classList.remove('is-stuck');
                }
            });
        });
    }
    
    // ========================================================================
    // Responsive Image Loading
    // ========================================================================
    
    function initLazyLoading() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver(function(entries, observer) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        img.src = img.getAttribute('data-src');
                        img.classList.remove('lazy');
                        observer.unobserve(img);
                    }
                });
            });
            
            const lazyImages = document.querySelectorAll('img.lazy');
            lazyImages.forEach(function(img) {
                imageObserver.observe(img);
            });
        }
    }
    
    // ========================================================================
    // Viewport Detection
    // ========================================================================
    
    function getViewport() {
        const width = window.innerWidth;
        
        if (width < 640) return 'mobile';
        if (width < 768) return 'sm';
        if (width < 1024) return 'md';
        if (width < 1280) return 'lg';
        return 'xl';
    }
    
    // Add viewport class to body
    function updateViewportClass() {
        const viewport = getViewport();
        document.body.setAttribute('data-viewport', viewport);
    }
    
    window.addEventListener('resize', function() {
        clearTimeout(window.viewportTimer);
        window.viewportTimer = setTimeout(updateViewportClass, 250);
    });
    
    // ========================================================================
    // Touch Device Detection
    // ========================================================================
    
    function detectTouch() {
        if ('ontouchstart' in window || navigator.maxTouchPoints > 0) {
            document.body.classList.add('touch-device');
        } else {
            document.body.classList.add('no-touch');
        }
    }
    
    // ========================================================================
    // Dropdown Management
    // ========================================================================
    
    function initDropdowns() {
        const dropdowns = document.querySelectorAll('[data-dropdown]');
        
        dropdowns.forEach(function(dropdown) {
            const trigger = dropdown.querySelector('[data-dropdown-trigger]');
            const menu = dropdown.querySelector('[data-dropdown-menu]');
            
            if (!trigger || !menu) return;
            
            trigger.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                // Close other dropdowns
                document.querySelectorAll('[data-dropdown-menu].active').forEach(function(otherMenu) {
                    if (otherMenu !== menu) {
                        otherMenu.classList.remove('active');
                    }
                });
                
                menu.classList.toggle('active');
            });
            
            // Close dropdown when clicking outside
            document.addEventListener('click', function(e) {
                if (!dropdown.contains(e.target)) {
                    menu.classList.remove('active');
                }
            });
        });
    }
    
    // ========================================================================
    // Form Validation Feedback
    // ========================================================================
    
    function initFormValidation() {
        const forms = document.querySelectorAll('form[data-validate]');
        
        forms.forEach(function(form) {
            const inputs = form.querySelectorAll('input, textarea, select');
            
            inputs.forEach(function(input) {
                input.addEventListener('blur', function() {
                    validateField(input);
                });
                
                input.addEventListener('input', function() {
                    if (input.classList.contains('is-invalid')) {
                        validateField(input);
                    }
                });
            });
            
            form.addEventListener('submit', function(e) {
                let isValid = true;
                
                inputs.forEach(function(input) {
                    if (!validateField(input)) {
                        isValid = false;
                    }
                });
                
                if (!isValid) {
                    e.preventDefault();
                    // Focus first invalid field
                    const firstInvalid = form.querySelector('.is-invalid');
                    if (firstInvalid) {
                        firstInvalid.focus();
                    }
                }
            });
        });
    }
    
    function validateField(field) {
        let isValid = true;
        const errorMessage = field.nextElementSibling;
        
        // Remove existing validation classes
        field.classList.remove('is-valid', 'is-invalid');
        if (errorMessage && errorMessage.classList.contains('error-message')) {
            errorMessage.remove();
        }
        
        // Check required
        if (field.hasAttribute('required') && !field.value.trim()) {
            isValid = false;
            showError(field, 'This field is required');
        }
        
        // Check email
        if (field.type === 'email' && field.value.trim()) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(field.value)) {
                isValid = false;
                showError(field, 'Please enter a valid email address');
            }
        }
        
        // Check minlength
        if (field.hasAttribute('minlength')) {
            const minLength = parseInt(field.getAttribute('minlength'));
            if (field.value.length < minLength && field.value.length > 0) {
                isValid = false;
                showError(field, `Minimum ${minLength} characters required`);
            }
        }
        
        if (isValid && field.value.trim()) {
            field.classList.add('is-valid');
        }
        
        return isValid;
    }
    
    function showError(field, message) {
        field.classList.add('is-invalid');
        
        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-message';
        errorDiv.style.color = 'var(--danger)';
        errorDiv.style.fontSize = 'var(--text-sm)';
        errorDiv.style.marginTop = 'var(--space-1)';
        errorDiv.textContent = message;
        
        field.parentNode.insertBefore(errorDiv, field.nextSibling);
    }
    
    // ========================================================================
    // Tooltips (Simple Implementation)
    // ========================================================================
    
    function initTooltips() {
        const tooltipElements = document.querySelectorAll('[data-tooltip]');
        
        tooltipElements.forEach(function(element) {
            element.addEventListener('mouseenter', function() {
                showTooltip(element);
            });
            
            element.addEventListener('mouseleave', function() {
                hideTooltip();
            });
        });
    }
    
    function showTooltip(element) {
        const text = element.getAttribute('data-tooltip');
        const tooltip = document.createElement('div');
        tooltip.className = 'tooltip';
        tooltip.textContent = text;
        tooltip.style.position = 'absolute';
        tooltip.style.zIndex = 'var(--z-tooltip)';
        tooltip.style.background = 'var(--gray-900)';
        tooltip.style.color = 'white';
        tooltip.style.padding = 'var(--space-2) var(--space-3)';
        tooltip.style.borderRadius = 'var(--radius)';
        tooltip.style.fontSize = 'var(--text-sm)';
        tooltip.style.whiteSpace = 'nowrap';
        
        document.body.appendChild(tooltip);
        
        const rect = element.getBoundingClientRect();
        tooltip.style.left = rect.left + (rect.width / 2) - (tooltip.offsetWidth / 2) + 'px';
        tooltip.style.top = rect.top - tooltip.offsetHeight - 8 + 'px';
    }
    
    function hideTooltip() {
        const tooltip = document.querySelector('.tooltip');
        if (tooltip) {
            tooltip.remove();
        }
    }
    
    // ========================================================================
    // Initialize All
    // ========================================================================
    
    function init() {
        initMobileMenu();
        initSwipeGestures();
        initResponsiveTables();
        initModals();
        initSmoothScroll();
        initStickyElements();
        initLazyLoading();
        updateViewportClass();
        detectTouch();
        initDropdowns();
        initFormValidation();
        initTooltips();
        
        // Emit custom event
        document.dispatchEvent(new CustomEvent('responsiveFrameworkReady'));
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
    
    // Export utility functions
    window.ResponsiveUtils = {
        getViewport: getViewport,
        openModal: openModal,
        closeModal: closeModal
    };
    
})();
