/**
 * Location Documents Component
 * Vue component for managing planning permissions and other location documents
 */

const LocationDocuments = {
    props: {
        locationId: { type: Number, required: true },
        editable: { type: Boolean, default: true }
    },
    data() {
        return {
            documents: [],
            loading: false,
            uploading: false,
            showUploader: false,
            selectedFile: null,
            documentType: 'planning_permission',
            title: '',
            description: '',
            documentDate: '',
            expiryDate: '',
            referenceNumber: '',
            filterType: 'all'
        };
    },
    computed: {
        filteredDocuments() {
            if (this.filterType === 'all') {
                return this.documents;
            }
            return this.documents.filter(doc => doc.document_type === this.filterType);
        },
        documentsByType() {
            const grouped = {};
            this.documents.forEach(doc => {
                if (!grouped[doc.document_type]) {
                    grouped[doc.document_type] = [];
                }
                grouped[doc.document_type].push(doc);
            });
            return grouped;
        },
        documentTypes() {
            return [
                { value: 'planning_permission', label: '📋 Planning Permission', icon: '📋' },
                { value: 'site_plan', label: '🗺️ Site Plan', icon: '🗺️' },
                { value: 'risk_assessment', label: '⚠️ Risk Assessment', icon: '⚠️' },
                { value: 'health_safety', label: '🏥 Health & Safety', icon: '🏥' },
                { value: 'delivery_instructions', label: '📦 Delivery Instructions', icon: '📦' },
                { value: 'contact_info', label: '📞 Contact Info', icon: '📞' },
                { value: 'access_permit', label: '🔑 Access Permit', icon: '🔑' },
                { value: 'insurance', label: '🛡️ Insurance', icon: '🛡️' },
                { value: 'other', label: '📄 Other', icon: '📄' }
            ];
        },
        expiringDocuments() {
            return this.documents.filter(doc => doc.status === 'expiring_soon' || doc.status === 'expired');
        }
    },
    template: `
        <div class="location-documents">
            <!-- Header with Filter -->
            <div class="documents-header">
                <h3 style="margin: 0;">📄 Location Documents</h3>
                
                <div style="display: flex; gap: 0.5rem; align-items: center;">
                    <select v-model="filterType" class="form-select" style="width: 200px;">
                        <option value="all">All Types</option>
                        <option v-for="type in documentTypes" :key="type.value" :value="type.value">
                            {{ type.label }}
                        </option>
                    </select>
                    
                    <button 
                        v-if="editable"
                        @click="showUploader = !showUploader" 
                        class="btn btn-primary"
                    >
                        {{ showUploader ? '✖ Close' : '➕ Upload Document' }}
                    </button>
                </div>
            </div>

            <!-- Expiry Warnings -->
            <div v-if="expiringDocuments.length > 0" class="alert alert-warning">
                ⚠️ <strong>{{ expiringDocuments.length }} document(s)</strong> expiring soon or expired
            </div>

            <!-- Upload Form -->
            <div v-if="showUploader && editable" class="upload-form">
                <h4>Upload New Document</h4>
                
                <div class="form-row">
                    <div class="form-group" style="flex: 2;">
                        <label class="form-label">Document Type *</label>
                        <select v-model="documentType" class="form-select" required>
                            <option v-for="type in documentTypes" :key="type.value" :value="type.value">
                                {{ type.label }}
                            </option>
                        </select>
                    </div>
                    
                    <div class="form-group" style="flex: 3;">
                        <label class="form-label">Document Title</label>
                        <input 
                            type="text" 
                            v-model="title"
                            placeholder="e.g., Planning Permission Ref. 12345"
                            class="form-control"
                        >
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="form-label">File *</label>
                    <input 
                        type="file" 
                        @change="handleFileSelect"
                        accept=".pdf,.doc,.docx,.xls,.xlsx,.jpg,.jpeg,.png,.gif,.txt"
                        class="form-control"
                        required
                    >
                    <small style="color: var(--text-muted);">
                        Max 10MB | PDF, Word, Excel, Images, Text
                    </small>
                </div>
                
                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea 
                        v-model="description"
                        placeholder="Additional notes about this document"
                        class="form-control"
                        rows="3"
                    ></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Document Date</label>
                        <input 
                            type="date" 
                            v-model="documentDate"
                            class="form-control"
                        >
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Expiry Date</label>
                        <input 
                            type="date" 
                            v-model="expiryDate"
                            class="form-control"
                        >
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label">Reference Number</label>
                        <input 
                            type="text" 
                            v-model="referenceNumber"
                            placeholder="e.g., PP/2024/12345"
                            class="form-control"
                        >
                    </div>
                </div>
                
                <div class="btn-group">
                    <button 
                        @click="uploadDocument"
                        :disabled="!selectedFile || uploading"
                        class="btn btn-primary"
                    >
                        {{ uploading ? '⏳ Uploading...' : '📤 Upload Document' }}
                    </button>
                    <button 
                        @click="resetForm"
                        :disabled="uploading"
                        class="btn btn-secondary"
                    >
                        Clear Form
                    </button>
                </div>
            </div>

            <!-- Documents List -->
            <div v-if="loading" style="text-align: center; padding: 2rem;">
                <div class="spinner" style="margin: 0 auto;"></div>
                <p style="color: var(--text-muted); margin-top: 0.5rem;">Loading documents...</p>
            </div>
            
            <div v-else-if="filteredDocuments.length === 0" class="empty-state">
                <div style="font-size: 3rem; margin-bottom: 1rem;">📄</div>
                <p style="color: var(--text-muted); margin: 0;">
                    {{ filterType === 'all' ? 'No documents uploaded yet' : 'No documents of this type' }}
                </p>
            </div>
            
            <div v-else class="documents-list">
                <div 
                    v-for="doc in filteredDocuments" 
                    :key="doc.id"
                    class="document-item"
                    :class="{ 'expired': doc.status === 'expired', 'expiring': doc.status === 'expiring_soon' }"
                >
                    <div class="document-icon">
                        {{ getDocumentIcon(doc.document_type) }}
                    </div>
                    
                    <div class="document-info">
                        <div class="document-title">
                            {{ doc.title || doc.original_filename }}
                        </div>
                        
                        <div class="document-meta">
                            <span>{{ formatDocumentType(doc.document_type) }}</span>
                            <span>{{ formatFileSize(doc.file_size) }}</span>
                            <span>Uploaded {{ formatDate(doc.uploaded_at) }}</span>
                            <span>by {{ doc.uploader_name }}</span>
                        </div>
                        
                        <div v-if="doc.description" class="document-description">
                            {{ doc.description }}
                        </div>
                        
                        <div class="document-details">
                            <span v-if="doc.document_date">
                                📅 Date: {{ formatDate(doc.document_date) }}
                            </span>
                            <span v-if="doc.expiry_date" :class="'status-' + doc.status">
                                ⏰ {{ doc.status_message }}
                            </span>
                            <span v-if="doc.reference_number">
                                🔖 Ref: {{ doc.reference_number }}
                            </span>
                        </div>
                    </div>
                    
                    <div class="document-actions">
                        <button 
                            @click="viewDocument(doc)"
                            class="btn-icon"
                            title="View document"
                        >
                            👁️
                        </button>
                        <button 
                            @click="downloadDocument(doc)"
                            class="btn-icon"
                            title="Download"
                        >
                            💾
                        </button>
                        <button 
                            v-if="editable"
                            @click="deleteDocument(doc.id)"
                            class="btn-icon-danger"
                            title="Delete"
                        >
                            🗑️
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `,
    methods: {
        async loadDocuments() {
            this.loading = true;
            try {
                const response = await axios.get(`api/location-documents.php?location_id=${this.locationId}`);
                if (response.data.success) {
                    this.documents = response.data.documents;
                }
            } catch (error) {
                console.error('Error loading documents:', error);
            } finally {
                this.loading = false;
            }
        },
        
        handleFileSelect(event) {
            this.selectedFile = event.target.files[0];
        },
        
        async uploadDocument() {
            if (!this.selectedFile) return;
            
            this.uploading = true;
            try {
                const formData = new FormData();
                formData.append('document', this.selectedFile);
                formData.append('location_id', this.locationId);
                formData.append('document_type', this.documentType);
                formData.append('title', this.title);
                formData.append('description', this.description);
                formData.append('document_date', this.documentDate);
                formData.append('expiry_date', this.expiryDate);
                formData.append('reference_number', this.referenceNumber);
                
                const response = await axios.post('api/location-documents.php', formData, {
                    headers: { 'Content-Type': 'multipart/form-data' }
                });
                
                if (response.data.success) {
                    this.resetForm();
                    this.showUploader = false;
                    await this.loadDocuments();
                    this.$emit('document-uploaded', response.data);
                }
            } catch (error) {
                console.error('Error uploading document:', error);
                alert(error.response?.data?.error || 'Failed to upload document');
            } finally {
                this.uploading = false;
            }
        },
        
        async deleteDocument(documentId) {
            if (!confirm('Delete this document? This cannot be undone.')) return;
            
            try {
                const response = await axios.delete(`api/location-documents.php?id=${documentId}`);
                if (response.data.success) {
                    await this.loadDocuments();
                    this.$emit('document-deleted', documentId);
                }
            } catch (error) {
                console.error('Error deleting document:', error);
                alert('Failed to delete document');
            }
        },
        
        viewDocument(doc) {
            window.open(doc.file_path, '_blank');
        },
        
        downloadDocument(doc) {
            const link = document.createElement('a');
            link.href = doc.file_path;
            link.download = doc.original_filename;
            link.click();
        },
        
        resetForm() {
            this.selectedFile = null;
            this.documentType = 'planning_permission';
            this.title = '';
            this.description = '';
            this.documentDate = '';
            this.expiryDate = '';
            this.referenceNumber = '';
            
            // Reset file input
            const fileInput = this.$el.querySelector('input[type="file"]');
            if (fileInput) fileInput.value = '';
        },
        
        getDocumentIcon(type) {
            const icons = {
                'planning_permission': '📋',
                'site_plan': '🗺️',
                'risk_assessment': '⚠️',
                'health_safety': '🏥',
                'delivery_instructions': '📦',
                'contact_info': '📞',
                'access_permit': '🔑',
                'insurance': '🛡️',
                'other': '📄'
            };
            return icons[type] || '📄';
        },
        
        formatDocumentType(type) {
            return type.split('_').map(word => 
                word.charAt(0).toUpperCase() + word.slice(1)
            ).join(' ');
        },
        
        formatFileSize(bytes) {
            if (bytes < 1024) return bytes + ' B';
            if (bytes < 1024 * 1024) return (bytes / 1024).toFixed(1) + ' KB';
            return (bytes / (1024 * 1024)).toFixed(1) + ' MB';
        },
        
        formatDate(dateString) {
            if (!dateString) return '';
            return new Date(dateString).toLocaleDateString('en-GB', {
                day: '2-digit',
                month: 'short',
                year: 'numeric'
            });
        }
    },
    mounted() {
        this.loadDocuments();
    }
};

// Styles
const locationDocumentsStyles = `
<style>
.location-documents {
    background: var(--bg-secondary);
    border-radius: 8px;
    padding: 1.5rem;
    margin-top: 1.5rem;
}

.documents-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    padding-bottom: 1rem;
    border-bottom: 2px solid var(--border-color);
}

.upload-form {
    background: var(--bg-primary);
    border: 2px solid var(--primary);
    border-radius: 8px;
    padding: 1.5rem;
    margin-bottom: 1.5rem;
}

.upload-form h4 {
    margin: 0 0 1rem 0;
    color: var(--primary);
}

.form-row {
    display: flex;
    gap: 1rem;
    margin-bottom: 1rem;
}

.form-row .form-group {
    flex: 1;
}

.documents-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.document-item {
    display: flex;
    gap: 1rem;
    padding: 1rem;
    background: var(--bg-primary);
    border: 1px solid var(--border-color);
    border-radius: 8px;
    transition: var(--transition);
    align-items: start;
}

.document-item:hover {
    box-shadow: var(--shadow);
    transform: translateY(-1px);
}

.document-item.expired {
    border-left: 4px solid var(--danger);
    background: rgba(220, 38, 38, 0.05);
}

.document-item.expiring {
    border-left: 4px solid var(--warning);
    background: rgba(245, 158, 11, 0.05);
}

.document-icon {
    font-size: 2.5rem;
    line-height: 1;
    flex-shrink: 0;
}

.document-info {
    flex: 1;
    min-width: 0;
}

.document-title {
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 0.5rem;
    word-break: break-word;
}

.document-meta {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
    font-size: 0.85rem;
    color: var(--text-muted);
    margin-bottom: 0.5rem;
}

.document-meta span {
    display: flex;
    align-items: center;
}

.document-description {
    font-size: 0.9rem;
    color: var(--text-secondary);
    margin: 0.5rem 0;
    line-height: 1.5;
}

.document-details {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
    font-size: 0.85rem;
    margin-top: 0.5rem;
}

.document-details span {
    padding: 0.25rem 0.5rem;
    background: var(--bg-secondary);
    border-radius: 4px;
}

.status-expired {
    background: rgba(220, 38, 38, 0.1) !important;
    color: var(--danger);
    font-weight: 600;
}

.status-expiring_soon {
    background: rgba(245, 158, 11, 0.1) !important;
    color: var(--warning);
    font-weight: 600;
}

.status-valid {
    background: rgba(5, 150, 105, 0.1) !important;
    color: var(--success);
}

.document-actions {
    display: flex;
    gap: 0.5rem;
    flex-shrink: 0;
}

.btn-icon, .btn-icon-danger {
    background: var(--bg-secondary);
    border: 1px solid var(--border-color);
    padding: 0.5rem;
    border-radius: 4px;
    cursor: pointer;
    font-size: 1.2rem;
    transition: var(--transition);
}

.btn-icon:hover {
    background: var(--primary);
    border-color: var(--primary);
    transform: scale(1.1);
}

.btn-icon-danger {
    border-color: var(--danger);
}

.btn-icon-danger:hover {
    background: var(--danger);
    transform: scale(1.1);
}

.empty-state {
    text-align: center;
    padding: 3rem;
}

.alert-warning {
    padding: 1rem;
    background: rgba(245, 158, 11, 0.1);
    border: 1px solid var(--warning);
    border-radius: 8px;
    color: var(--warning);
    margin-bottom: 1rem;
}
</style>
`;

// Add styles to page
if (typeof document !== 'undefined') {
    const styleEl = document.createElement('style');
    styleEl.innerHTML = locationDocumentsStyles.replace(/<\/?style>/g, '');
    document.head.appendChild(styleEl);
}

// Export for use
window.LocationDocuments = LocationDocuments;
