// Risk Assessment App - Main JavaScript
class RiskAssessmentApp {
    constructor() {
        this.isOnline = navigator.onLine;
        this.syncQueue = [];
        this.db = null;
        this.init();
    }

    async init() {
        this.setupEventListeners();
        this.setupOfflineDetection();
        this.initIndexedDB();
        this.registerServiceWorker();
        this.checkPendingSync();
    }

    setupEventListeners() {
        // Handle online/offline events
        window.addEventListener('online', () => this.handleOnline());
        window.addEventListener('offline', () => this.handleOffline());

        // Handle form submissions with offline support
        document.querySelectorAll('form[data-offline-support]').forEach(form => {
            form.addEventListener('submit', (e) => this.handleFormSubmit(e));
        });

        // Auto-save functionality
        document.querySelectorAll('[data-autosave]').forEach(input => {
            input.addEventListener('change', (e) => this.autoSave(e));
        });
    }

    setupOfflineDetection() {
        const indicator = document.getElementById('offline-indicator');
        if (!this.isOnline && indicator) {
            indicator.classList.add('active');
        }
    }

    handleOnline() {
        this.isOnline = true;
        const indicator = document.getElementById('offline-indicator');
        if (indicator) {
            indicator.classList.remove('active');
        }
        this.showAlert('You are back online. Syncing data...', 'success');
        this.syncOfflineData();
    }

    handleOffline() {
        this.isOnline = false;
        const indicator = document.getElementById('offline-indicator');
        if (indicator) {
            indicator.classList.add('active');
        }
        this.showAlert('You are offline. Data will be saved locally and synced when online.', 'warning');
    }

    // IndexedDB for offline storage
    initIndexedDB() {
        const request = indexedDB.open('RiskAssessmentDB', 1);

        request.onerror = () => {
            console.error('IndexedDB error:', request.error);
        };

        request.onsuccess = () => {
            this.db = request.result;
            console.log('IndexedDB initialized');
        };

        request.onupgradeneeded = (event) => {
            const db = event.target.result;

            // Create object stores
            if (!db.objectStoreNames.contains('assessments')) {
                const assessmentStore = db.createObjectStore('assessments', { keyPath: 'id', autoIncrement: true });
                assessmentStore.createIndex('sync_status', 'sync_status', { unique: false });
                assessmentStore.createIndex('assessment_number', 'assessment_number', { unique: true });
            }

            if (!db.objectStoreNames.contains('photos')) {
                const photoStore = db.createObjectStore('photos', { keyPath: 'id', autoIncrement: true });
                photoStore.createIndex('assessment_id', 'assessment_id', { unique: false });
            }

            if (!db.objectStoreNames.contains('locations')) {
                db.createObjectStore('locations', { keyPath: 'id', autoIncrement: true });
            }
        };
    }

    // Service Worker registration for PWA
    async registerServiceWorker() {
        if ('serviceWorker' in navigator) {
            try {
                const registration = await navigator.serviceWorker.register('/js/sw.js');
                console.log('Service Worker registered:', registration);
            } catch (error) {
                console.error('Service Worker registration failed:', error);
            }
        }
    }

    // Handle form submissions with offline support
    async handleFormSubmit(event) {
        const form = event.target;
        
        // Skip AJAX handling if form has data-no-ajax attribute or is assessment form
        if (form.hasAttribute('data-no-ajax') || form.id === 'assessment-form') {
            // Let the form submit normally to PHP
            return;
        }
        
        event.preventDefault();
        const formData = new FormData(form);
        const data = Object.fromEntries(formData.entries());

        if (this.isOnline) {
            // Online - submit normally
            this.submitForm(form, data);
        } else {
            // Offline - save to IndexedDB
            await this.saveOffline(form.dataset.entityType || 'assessment', data);
            this.showAlert('Data saved locally. Will sync when online.', 'success');
            if (form.dataset.redirect) {
                window.location.href = form.dataset.redirect;
            }
        }
    }

    async submitForm(form, data) {
        try {
            const response = await fetch(form.action, {
                method: form.method || 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });

            const result = await response.json();

            if (result.success) {
                this.showAlert(result.message || 'Saved successfully', 'success');
                if (form.dataset.redirect) {
                    setTimeout(() => {
                        window.location.href = form.dataset.redirect;
                    }, 1000);
                }
            } else {
                this.showAlert(result.message || 'Error saving data', 'error');
            }
        } catch (error) {
            console.error('Form submission error:', error);
            this.showAlert('Error submitting form. Please try again.', 'error');
        }
    }

    async saveOffline(entityType, data) {
        if (!this.db) return;

        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction([entityType + 's'], 'readwrite');
            const store = transaction.objectStore(entityType + 's');
            
            data.sync_status = 'pending';
            data.created_at_local = new Date().toISOString();
            
            const request = store.add(data);

            request.onsuccess = () => {
                console.log('Data saved offline:', request.result);
                resolve(request.result);
            };

            request.onerror = () => {
                console.error('Error saving offline:', request.error);
                reject(request.error);
            };
        });
    }

    async syncOfflineData() {
        if (!this.db || !this.isOnline) return;

        const transaction = this.db.transaction(['assessments'], 'readonly');
        const store = transaction.objectStore('assessments');
        const index = store.index('sync_status');
        const request = index.getAll('pending');

        request.onsuccess = async () => {
            const pendingItems = request.result;
            
            if (pendingItems.length === 0) {
                console.log('No pending items to sync');
                return;
            }

            this.showLoading(true);

            for (const item of pendingItems) {
                try {
                    await this.syncItem(item);
                    await this.updateSyncStatus(item.id, 'synced');
                } catch (error) {
                    console.error('Sync error for item:', item.id, error);
                    await this.updateSyncStatus(item.id, 'failed');
                }
            }

            this.showLoading(false);
            this.showAlert('All offline data synced successfully', 'success');
        };
    }

    async syncItem(item) {
        const response = await fetch('/api/sync.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(item)
        });

        if (!response.ok) {
            throw new Error('Sync failed');
        }

        return response.json();
    }

    async updateSyncStatus(id, status) {
        return new Promise((resolve, reject) => {
            const transaction = this.db.transaction(['assessments'], 'readwrite');
            const store = transaction.objectStore('assessments');
            const request = store.get(id);

            request.onsuccess = () => {
                const item = request.result;
                item.sync_status = status;
                const updateRequest = store.put(item);

                updateRequest.onsuccess = () => resolve();
                updateRequest.onerror = () => reject(updateRequest.error);
            };
        });
    }

    async checkPendingSync() {
        if (!this.db) {
            setTimeout(() => this.checkPendingSync(), 1000);
            return;
        }

        const transaction = this.db.transaction(['assessments'], 'readonly');
        const store = transaction.objectStore('assessments');
        const index = store.index('sync_status');
        const request = index.count('pending');

        request.onsuccess = () => {
            const count = request.result;
            if (count > 0) {
                this.showAlert(`You have ${count} item(s) pending sync`, 'info');
            }
        };
    }

    // Auto-save functionality
    async autoSave(event) {
        const input = event.target;
        const formId = input.closest('form')?.id;
        
        if (!formId) return;

        const formData = new FormData(document.getElementById(formId));
        const data = Object.fromEntries(formData.entries());

        // Save to localStorage as draft
        localStorage.setItem(`draft_${formId}`, JSON.stringify({
            data,
            timestamp: new Date().toISOString()
        }));

        // Show auto-save indicator
        const indicator = document.getElementById('autosave-indicator');
        if (indicator) {
            indicator.textContent = 'Draft saved';
            indicator.style.display = 'inline';
            setTimeout(() => {
                indicator.style.display = 'none';
            }, 2000);
        }
    }

    // Restore draft from localStorage
    restoreDraft(formId) {
        const draft = localStorage.getItem(`draft_${formId}`);
        if (!draft) return false;

        try {
            const { data, timestamp } = JSON.parse(draft);
            const draftAge = Date.now() - new Date(timestamp).getTime();
            
            // Only restore drafts less than 24 hours old
            if (draftAge > 24 * 60 * 60 * 1000) {
                localStorage.removeItem(`draft_${formId}`);
                return false;
            }

            const form = document.getElementById(formId);
            if (!form) return false;

            Object.entries(data).forEach(([key, value]) => {
                const input = form.elements[key];
                if (input) {
                    input.value = value;
                }
            });

            this.showAlert('Draft restored from ' + new Date(timestamp).toLocaleString(), 'info');
            return true;
        } catch (error) {
            console.error('Error restoring draft:', error);
            return false;
        }
    }

    clearDraft(formId) {
        localStorage.removeItem(`draft_${formId}`);
    }

    // UI Helper functions
    showAlert(message, type = 'info') {
        const alertContainer = document.getElementById('alert-container') || this.createAlertContainer();
        
        const alert = document.createElement('div');
        alert.className = `alert alert-${type}`;
        alert.innerHTML = `
            <span>📋</span>
            <span>${message}</span>
            <button onclick="this.parentElement.remove()" style="margin-left: auto; background: none; border: none; color: inherit; cursor: pointer; font-size: 1.2rem;">&times;</button>
        `;
        
        alertContainer.appendChild(alert);

        setTimeout(() => {
            alert.remove();
        }, 5000);
    }

    createAlertContainer() {
        const container = document.createElement('div');
        container.id = 'alert-container';
        container.style.cssText = 'position: fixed; top: 90px; right: 20px; z-index: 9999; width: 400px; max-width: 90%;';
        document.body.appendChild(container);
        return container;
    }

    showLoading(show = true) {
        const overlay = document.getElementById('loading-overlay');
        if (overlay) {
            overlay.classList.toggle('active', show);
        }
    }

    // Modal helpers
    openModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.add('active');
        }
    }

    closeModal(modalId) {
        const modal = document.getElementById(modalId);
        if (modal) {
            modal.classList.remove('active');
        }
    }

    // Risk calculation
    calculateRiskScore(likelihood, severity) {
        return likelihood * severity;
    }

    getRiskLevel(score) {
        if (score <= 4) return 'low';
        if (score <= 9) return 'medium';
        if (score <= 16) return 'high';
        return 'critical';
    }

    getRiskColor(level) {
        const colors = {
            low: '#059669',
            medium: '#F59E0B',
            high: '#EF4444',
            critical: '#DC2626'
        };
        return colors[level] || '#94A3B8';
    }

    // Photo handling
    async handlePhotoUpload(input, assessmentId) {
        const files = input.files;
        if (!files.length) return;

        const formData = new FormData();
        formData.append('assessment_id', assessmentId);
        
        for (let i = 0; i < files.length; i++) {
            formData.append('photos[]', files[i]);
        }

        try {
            if (this.isOnline) {
                const response = await fetch('/api/upload-photos.php', {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();
                
                if (result.success) {
                    this.showAlert('Photos uploaded successfully', 'success');
                    return result.photos;
                }
            } else {
                // Store photos in IndexedDB for offline mode
                await this.storePhotosOffline(files, assessmentId);
                this.showAlert('Photos saved locally. Will upload when online.', 'info');
            }
        } catch (error) {
            console.error('Photo upload error:', error);
            this.showAlert('Error uploading photos', 'error');
        }
    }

    async storePhotosOffline(files, assessmentId) {
        if (!this.db) return;

        for (const file of files) {
            const reader = new FileReader();
            reader.onload = async (e) => {
                const transaction = this.db.transaction(['photos'], 'readwrite');
                const store = transaction.objectStore('photos');
                
                await store.add({
                    assessment_id: assessmentId,
                    filename: file.name,
                    data: e.target.result,
                    sync_status: 'pending',
                    created_at: new Date().toISOString()
                });
            };
            reader.readAsDataURL(file);
        }
    }

    // Export/Print functionality
    async generatePDF(assessmentId) {
        this.showLoading(true);
        
        try {
            const response = await fetch(`/api/generate-pdf.php?id=${assessmentId}`);
            const blob = await response.blob();
            
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `risk-assessment-${assessmentId}.pdf`;
            document.body.appendChild(a);
            a.click();
            window.URL.revokeObjectURL(url);
            document.body.removeChild(a);
            
            this.showAlert('PDF generated successfully', 'success');
        } catch (error) {
            console.error('PDF generation error:', error);
            this.showAlert('Error generating PDF', 'error');
        } finally {
            this.showLoading(false);
        }
    }

    // Data validation
    validateForm(formId) {
        const form = document.getElementById(formId);
        if (!form) return false;

        const requiredFields = form.querySelectorAll('[required]');
        let isValid = true;

        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                field.classList.add('error');
                isValid = false;
            } else {
                field.classList.remove('error');
            }
        });

        if (!isValid) {
            this.showAlert('Please fill in all required fields', 'error');
        }

        return isValid;
    }
}

// Initialize app when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    window.app = new RiskAssessmentApp();

    // Setup modal close buttons
    document.querySelectorAll('.modal-close, [data-modal-close]').forEach(btn => {
        btn.addEventListener('click', (e) => {
            const modal = e.target.closest('.modal');
            if (modal) {
                modal.classList.remove('active');
            }
        });
    });

    // Close modal on overlay click
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                modal.classList.remove('active');
            }
        });
    });

    // Risk matrix interactive cells
    document.querySelectorAll('.risk-cell:not(.header)').forEach(cell => {
        cell.addEventListener('click', function() {
            const likelihood = this.dataset.likelihood;
            const severity = this.dataset.severity;
            
            if (likelihood && severity) {
                const score = likelihood * severity;
                const level = window.app.getRiskLevel(score);
                console.log(`Risk Score: ${score}, Level: ${level}`);
            }
        });
    });
});

// Helper function for date formatting
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-GB', {
        day: '2-digit',
        month: '2-digit',
        year: 'numeric'
    });
}

// Helper function for time formatting
function formatTime(timeString) {
    return timeString.substring(0, 5);
}
