<?php
require_once '../includes/config.php';
require_once '../includes/error_handler.php';
requireLogin();

// Only admins can view error logs
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$log_type = isset($_GET['log']) ? $_GET['log'] : 'errors';
$lines_to_show = isset($_GET['lines']) ? (int)$_GET['lines'] : 100;

// Handle log actions
if (isset($_POST['action'])) {
    if ($_POST['action'] === 'clear' && isset($_POST['log_file'])) {
        $log_file = $_POST['log_file'];
        $valid_logs = ['errors' => ERROR_LOG_FILE, 'debug' => DEBUG_LOG_FILE, 'security' => SECURITY_LOG_FILE];
        
        if (isset($valid_logs[$log_file])) {
            file_put_contents($valid_logs[$log_file], '');
            $success = "Log cleared successfully!";
        }
    } elseif ($_POST['action'] === 'cleanup') {
        clearOldLogs(30);
        $success = "Old logs cleaned up successfully!";
    }
}

// Get log file path
$log_files = [
    'errors' => ERROR_LOG_FILE,
    'debug' => DEBUG_LOG_FILE,
    'security' => SECURITY_LOG_FILE
];

$current_log_file = $log_files[$log_type] ?? ERROR_LOG_FILE;

// Read log file
$log_entries = [];
if (file_exists($current_log_file)) {
    $lines = file($current_log_file);
    $lines = array_reverse($lines); // Show newest first
    $lines = array_slice($lines, 0, $lines_to_show);
    
    foreach ($lines as $line) {
        if (preg_match('/\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\] (.+)/', $line, $matches)) {
            $log_entries[] = [
                'timestamp' => $matches[1],
                'message' => $matches[2],
                'full' => $line
            ];
        } else {
            $log_entries[] = [
                'timestamp' => '',
                'message' => $line,
                'full' => $line
            ];
        }
    }
}

// Get statistics
$stats = getErrorStats();

include '../templates/header.php';
?>

<div class="container">
    <div class="page-header">
        <div>
            <h1 class="page-title">📊 Error Logs</h1>
            <p class="page-subtitle">Monitor system errors and security events</p>
        </div>
        <div class="btn-group">
            <a href="admin-panel.php" class="btn btn-secondary">← Back to Admin</a>
        </div>
    </div>

    <?php if (isset($success)): ?>
    <div class="alert alert-success">
        ✅ <?php echo htmlspecialchars($success); ?>
    </div>
    <?php endif; ?>

    <!-- Statistics -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon" style="color: var(--danger);">📝</div>
            <div>
                <div class="stat-value"><?php echo number_format($stats['total_errors']); ?></div>
                <div class="stat-label">Total Errors</div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="color: var(--warning);">⚠️</div>
            <div>
                <div class="stat-value"><?php echo number_format($stats['recent_errors']); ?></div>
                <div class="stat-label">Last 24 Hours</div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="color: var(--info);">🔒</div>
            <div>
                <div class="stat-value"><?php echo number_format($stats['security_events']); ?></div>
                <div class="stat-label">Security Events</div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="color: var(--success);">📅</div>
            <div>
                <div class="stat-value">
                    <?php echo $stats['last_error'] ? date('H:i', strtotime($stats['last_error']['time'])) : 'None'; ?>
                </div>
                <div class="stat-label">Last Error</div>
            </div>
        </div>
    </div>

    <!-- Log Type Tabs -->
    <div class="log-tabs">
        <a href="?log=errors" class="log-tab <?php echo $log_type === 'errors' ? 'active' : ''; ?>">
            📝 System Errors
        </a>
        <a href="?log=debug" class="log-tab <?php echo $log_type === 'debug' ? 'active' : ''; ?>">
            🐛 Debug Log
        </a>
        <a href="?log=security" class="log-tab <?php echo $log_type === 'security' ? 'active' : ''; ?>">
            🔒 Security Events
        </a>
    </div>

    <!-- Controls -->
    <div class="card">
        <div class="card-body" style="padding: 1rem;">
            <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem;">
                <div style="display: flex; gap: 1rem; align-items: center;">
                    <label>Show:</label>
                    <select onchange="window.location.href='?log=<?php echo $log_type; ?>&lines='+this.value" class="form-control" style="width: auto;">
                        <option value="50" <?php echo $lines_to_show == 50 ? 'selected' : ''; ?>>50 lines</option>
                        <option value="100" <?php echo $lines_to_show == 100 ? 'selected' : ''; ?>>100 lines</option>
                        <option value="200" <?php echo $lines_to_show == 200 ? 'selected' : ''; ?>>200 lines</option>
                        <option value="500" <?php echo $lines_to_show == 500 ? 'selected' : ''; ?>>500 lines</option>
                    </select>
                </div>

                <div style="display: flex; gap: 0.5rem;">
                    <form method="POST" style="display: inline;" onsubmit="return confirm('Clear this log file?');">
                        <input type="hidden" name="action" value="clear">
                        <input type="hidden" name="log_file" value="<?php echo $log_type; ?>">
                        <button type="submit" class="btn btn-warning">🗑️ Clear Log</button>
                    </form>

                    <form method="POST" style="display: inline;" onsubmit="return confirm('Delete logs older than 30 days?');">
                        <input type="hidden" name="action" value="cleanup">
                        <button type="submit" class="btn btn-secondary">🧹 Cleanup Old</button>
                    </form>

                    <button onclick="window.location.reload()" class="btn btn-primary">🔄 Refresh</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Log Entries -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">
                <?php 
                $titles = [
                    'errors' => 'System Errors',
                    'debug' => 'Debug Log',
                    'security' => 'Security Events'
                ];
                echo $titles[$log_type];
                ?>
                (<?php echo count($log_entries); ?> entries)
            </h2>
        </div>
        <div class="card-body" style="padding: 0;">
            <?php if (empty($log_entries)): ?>
                <div style="padding: 3rem; text-align: center; color: var(--text-muted);">
                    <div style="font-size: 3rem; margin-bottom: 1rem;">✅</div>
                    <p>No log entries found</p>
                </div>
            <?php else: ?>
                <div class="log-container">
                    <?php foreach ($log_entries as $entry): ?>
                    <div class="log-entry <?php echo getLogEntryClass($entry['message']); ?>">
                        <?php if ($entry['timestamp']): ?>
                        <div class="log-timestamp">
                            <?php echo date('d M Y H:i:s', strtotime($entry['timestamp'])); ?>
                        </div>
                        <?php endif; ?>
                        <div class="log-message">
                            <?php echo htmlspecialchars($entry['message']); ?>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Log File Info -->
    <div class="card">
        <div class="card-body">
            <div style="display: grid; grid-template-columns: 150px 1fr; gap: 0.5rem; color: var(--text-muted); font-size: 0.9rem;">
                <div><strong>Log File:</strong></div>
                <div><code><?php echo basename($current_log_file); ?></code></div>
                
                <div><strong>File Size:</strong></div>
                <div>
                    <?php 
                    if (file_exists($current_log_file)) {
                        $size = filesize($current_log_file);
                        echo formatBytes($size);
                    } else {
                        echo 'File does not exist';
                    }
                    ?>
                </div>
                
                <div><strong>Last Modified:</strong></div>
                <div>
                    <?php 
                    if (file_exists($current_log_file)) {
                        echo date('d M Y H:i:s', filemtime($current_log_file));
                    } else {
                        echo 'N/A';
                    }
                    ?>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
.log-tabs {
    display: flex;
    gap: 0.5rem;
    margin-bottom: 1.5rem;
    border-bottom: 2px solid var(--border-color);
}

.log-tab {
    padding: 0.75rem 1.5rem;
    text-decoration: none;
    color: var(--text-secondary);
    border-bottom: 3px solid transparent;
    transition: var(--transition);
}

.log-tab:hover {
    color: var(--primary);
}

.log-tab.active {
    color: var(--primary);
    border-bottom-color: var(--primary);
    font-weight: 600;
}

.log-container {
    max-height: 600px;
    overflow-y: auto;
}

.log-entry {
    padding: 1rem;
    border-bottom: 1px solid var(--border-color);
    font-family: 'Courier New', monospace;
    font-size: 0.85rem;
}

.log-entry:last-child {
    border-bottom: none;
}

.log-entry.error {
    background: rgba(220, 38, 38, 0.1);
    border-left: 4px solid var(--danger);
}

.log-entry.warning {
    background: rgba(251, 191, 36, 0.1);
    border-left: 4px solid var(--warning);
}

.log-entry.security {
    background: rgba(239, 68, 68, 0.1);
    border-left: 4px solid #ef4444;
}

.log-timestamp {
    color: var(--text-muted);
    font-size: 0.8rem;
    margin-bottom: 0.25rem;
}

.log-message {
    color: var(--text-primary);
    white-space: pre-wrap;
    word-break: break-all;
}

.alert {
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1.5rem;
}

.alert-success {
    background: rgba(5, 150, 105, 0.1);
    border: 1px solid var(--success);
    color: var(--success);
}
</style>

<?php include '../templates/footer.php'; ?>

<?php
function getLogEntryClass($message) {
    if (stripos($message, 'ERROR') !== false || stripos($message, 'FATAL') !== false) {
        return 'error';
    } elseif (stripos($message, 'WARNING') !== false || stripos($message, 'NOTICE') !== false) {
        return 'warning';
    } elseif (stripos($message, 'SECURITY') !== false) {
        return 'security';
    }
    return '';
}

function formatBytes($bytes) {
    if ($bytes < 1024) return $bytes . ' B';
    if ($bytes < 1048576) return round($bytes / 1024, 2) . ' KB';
    return round($bytes / 1048576, 2) . ' MB';
}
?>
