<?php
require_once '../includes/config.php';
requireLogin();

// Fetch statistics
$stats = [
    'total_assessments' => 0,
    'high_risk' => 0,
    'pending_actions' => 0,
    'locations' => 0
];

try {
    $stmt = $pdo->query("SELECT COUNT(*) FROM assessments WHERE status != 'archived'");
    $stats['total_assessments'] = $stmt->fetchColumn();

    $stmt = $pdo->query("SELECT COUNT(*) FROM assessments WHERE overall_risk_level IN ('high', 'critical') AND status = 'completed'");
    $stats['high_risk'] = $stmt->fetchColumn();

    $stmt = $pdo->query("SELECT COUNT(DISTINCT ad.id) FROM assessment_details ad 
                         JOIN assessments a ON ad.assessment_id = a.id 
                         WHERE ad.target_date >= CURDATE() AND a.status = 'completed'");
    $stats['pending_actions'] = $stmt->fetchColumn();

    $stmt = $pdo->query("SELECT COUNT(*) FROM locations");
    $stats['locations'] = $stmt->fetchColumn();

    // Recent assessments
    $stmt = $pdo->prepare("
        SELECT a.*, l.location_name, l.city, l.postcode, u.full_name as assessor_name
        FROM assessments a
        JOIN locations l ON a.location_id = l.id
        JOIN users u ON a.assessor_id = u.id
        ORDER BY a.created_at DESC
        LIMIT 10
    ");
    $stmt->execute();
    $recent_assessments = $stmt->fetchAll();

    // Risk distribution
    $stmt = $pdo->query("
        SELECT overall_risk_level, COUNT(*) as count
        FROM assessments
        WHERE status = 'completed'
        GROUP BY overall_risk_level
    ");
    $risk_distribution = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

} catch (PDOException $e) {
    error_log($e->getMessage());
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
    <link rel="manifest" href="manifest.json">
    <meta name="theme-color" content="#D97706">
</head>
<body>
    <?php include '../templates/header.php'; ?>

    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Dashboard</h1>
            <p class="page-subtitle">Welcome back, <?php echo sanitizeOutput($_SESSION['full_name']); ?>!</p>
        </div>

        <!-- Statistics -->
        <div class="stats-grid">
            <div class="stat-card primary">
                <div class="stat-value"><?php echo $stats['total_assessments']; ?></div>
                <div class="stat-label">Total Assessments</div>
            </div>
            <div class="stat-card danger">
                <div class="stat-value"><?php echo $stats['high_risk']; ?></div>
                <div class="stat-label">High Risk Sites</div>
            </div>
            <div class="stat-card info">
                <div class="stat-value"><?php echo $stats['pending_actions']; ?></div>
                <div class="stat-label">Pending Actions</div>
            </div>
            <div class="stat-card success">
                <div class="stat-value"><?php echo $stats['locations']; ?></div>
                <div class="stat-label">Delivery Locations</div>
            </div>
        </div>

        <!-- Risk Distribution Chart -->
        <div class="card mb-3">
            <div class="card-header">
                <h2 class="card-title">Risk Level Distribution</h2>
            </div>
            <div style="padding: 2rem;">
                <div style="display: flex; gap: 2rem; align-items: flex-end; height: 300px;">
                    <?php 
                    $risk_levels = ['low' => 0, 'medium' => 0, 'high' => 0, 'critical' => 0];
                    $risk_levels = array_merge($risk_levels, $risk_distribution);
                    $max_count = max(array_values($risk_levels)) ?: 1;
                    
                    foreach ($risk_levels as $level => $count):
                        $height = ($count / $max_count) * 100;
                    ?>
                        <div style="flex: 1; display: flex; flex-direction: column; align-items: center; gap: 1rem;">
                            <div style="font-size: 1.5rem; font-weight: 700;"><?php echo $count; ?></div>
                            <div style="width: 100%; height: <?php echo $height; ?>%; min-height: 40px; background: var(--<?php echo $level === 'low' ? 'success' : ($level === 'medium' ? 'warning' : 'danger'); ?>); border-radius: 8px 8px 0 0; transition: all 0.3s;"></div>
                            <div style="text-transform: capitalize; font-weight: 600;"><?php echo $level; ?></div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <!-- Recent Assessments -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">Recent Assessments</h2>
                <a href="assessments.php" class="btn btn-secondary">View All</a>
            </div>
            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Assessment #</th>
                            <th>Location</th>
                            <th>Date</th>
                            <th>Assessor</th>
                            <th>Risk Level</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($recent_assessments)): ?>
                            <tr>
                                <td colspan="7" style="text-align: center; padding: 3rem; color: var(--text-muted);">
                                    No assessments found. <a href="new-assessment.php" style="color: var(--primary);">Create your first assessment</a>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($recent_assessments as $assessment): ?>
                                <tr>
                                    <td><?php echo sanitizeOutput($assessment['assessment_number']); ?></td>
                                    <td>
                                        <strong><?php echo sanitizeOutput($assessment['location_name']); ?></strong><br>
                                        <small style="color: var(--text-muted);"><?php echo sanitizeOutput($assessment['city'] . ', ' . $assessment['postcode']); ?></small>
                                    </td>
                                    <td><?php echo date('d/m/Y', strtotime($assessment['assessment_date'])); ?></td>
                                    <td><?php echo sanitizeOutput($assessment['assessor_name']); ?></td>
                                    <td>
                                        <span class="badge badge-<?php echo $assessment['overall_risk_level']; ?>">
                                            <?php echo strtoupper($assessment['overall_risk_level']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-<?php echo $assessment['status']; ?>">
                                            <?php echo strtoupper($assessment['status']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <div class="btn-group" style="gap: 0.5rem;">
                                            <a href="view-assessment.php?id=<?php echo $assessment['id']; ?>" class="btn btn-secondary" style="padding: 0.5rem 1rem;">📋 View</a>
                                            <a href="edit-assessment.php?id=<?php echo $assessment['id']; ?>" class="btn btn-secondary" style="padding: 0.5rem 1rem;">✏️ Edit</a>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Offline Indicator -->
    <div id="offline-indicator" class="offline-indicator">
        <span>📡</span>
        <span>You are offline. Changes will be saved locally.</span>
    </div>

    <!-- Loading Overlay -->
    <div id="loading-overlay" class="loading-overlay">
        <div class="spinner"></div>
    </div>

    <script src="js/app.js"></script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
