<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Risk Assessment System</title>
    <link rel="stylesheet" href="css/style.css">
    
    <!-- Vue 3 -->
    <script src="https://unpkg.com/vue@3.4.15/dist/vue.global.prod.js"></script>
    
    <!-- Axios for API calls -->
    <script src="https://unpkg.com/axios@1.6.5/dist/axios.min.js"></script>
</head>
<body>
    <?php 
    require_once '../includes/config.php';
    requireLogin();
    include '../templates/header.php'; 
    ?>

    <div id="app" class="container">
        <!-- Loading State -->
        <div v-if="loading" class="loading-state">
            <div style="text-align: center; padding: 3rem;">
                <div class="spinner"></div>
                <p style="margin-top: 1rem; color: var(--text-muted);">Loading dashboard...</p>
            </div>
        </div>

        <!-- Dashboard Content -->
        <div v-else>
            <!-- Page Header -->
            <div class="page-header">
                <div>
                    <h1 class="page-title">Dashboard</h1>
                    <p class="page-subtitle">Risk Assessment Overview</p>
                </div>
                <div class="btn-group">
                    <a href="new-assessment.php" class="btn btn-primary">
                        ➕ New Assessment
                    </a>
                </div>
            </div>

            <!-- Statistics Grid -->
            <div class="stats-grid">
                <stat-card
                    v-for="stat in stats"
                    :key="stat.label"
                    :label="stat.label"
                    :value="stat.value"
                    :icon="stat.icon"
                    :color="stat.color"
                    :link="stat.link"
                ></stat-card>
            </div>

            <!-- Risk Distribution Chart -->
            <div class="card" v-if="riskDistribution.length > 0">
                <div class="card-header">
                    <h2 class="card-title">📊 Risk Distribution</h2>
                </div>
                <div class="card-body">
                    <div class="risk-chart">
                        <div 
                            v-for="risk in riskDistribution" 
                            :key="risk.level"
                            class="risk-bar"
                            :style="{ width: risk.percentage + '%' }"
                        >
                            <span class="risk-label" :class="'risk-' + risk.level">
                                {{ risk.level }}: {{ risk.count }}
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Recent Assessments -->
            <div class="card">
                <div class="card-header" style="display: flex; justify-content: space-between; align-items: center;">
                    <h2 class="card-title">📋 Recent Assessments</h2>
                    <div style="display: flex; gap: 0.5rem;">
                        <input 
                            v-model="searchQuery" 
                            type="text" 
                            placeholder="Search assessments..."
                            class="form-control"
                            style="width: 250px;"
                        >
                        <select v-model="filterRisk" class="form-select" style="width: 150px;">
                            <option value="">All Risks</option>
                            <option value="low">Low</option>
                            <option value="medium">Medium</option>
                            <option value="high">High</option>
                            <option value="critical">Critical</option>
                        </select>
                    </div>
                </div>
                <div class="card-body" style="padding: 0;">
                    <div class="table-container">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Assessment #</th>
                                    <th>Location</th>
                                    <th>Date</th>
                                    <th>Risk Level</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr v-if="filteredAssessments.length === 0">
                                    <td colspan="6" style="text-align: center; padding: 2rem; color: var(--text-muted);">
                                        No assessments found
                                    </td>
                                </tr>
                                <tr v-for="assessment in filteredAssessments" :key="assessment.id">
                                    <td>
                                        <strong>{{ assessment.assessment_number }}</strong>
                                    </td>
                                    <td>
                                        {{ assessment.location_name }}<br>
                                        <small style="color: var(--text-muted);">
                                            {{ assessment.city }}
                                        </small>
                                    </td>
                                    <td>{{ formatDate(assessment.assessment_date) }}</td>
                                    <td>
                                        <span :class="'badge badge-' + getRiskBadge(assessment.overall_risk_level)">
                                            {{ assessment.overall_risk_level }}
                                        </span>
                                    </td>
                                    <td>
                                        <span :class="'badge badge-' + getStatusBadge(assessment.status)">
                                            {{ assessment.status }}
                                        </span>
                                    </td>
                                    <td>
                                        <a :href="'view-assessment.php?id=' + assessment.id" class="btn btn-secondary" style="padding: 0.4rem 0.8rem;">
                                            View
                                        </a>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <style>
        .loading-state {
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 400px;
        }

        .spinner {
            width: 50px;
            height: 50px;
            border: 4px solid var(--border-color);
            border-top-color: var(--primary);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        .risk-chart {
            display: flex;
            flex-direction: column;
            gap: 1rem;
            padding: 1rem;
        }

        .risk-bar {
            background: var(--bg-tertiary);
            padding: 0.75rem;
            border-radius: 8px;
            min-width: 100px;
            transition: var(--transition);
        }

        .risk-bar:hover {
            transform: translateX(5px);
            box-shadow: var(--shadow);
        }

        .risk-label {
            font-weight: 600;
            text-transform: uppercase;
            font-size: 0.85rem;
        }

        [v-cloak] {
            display: none;
        }
    </style>

    <script>
        const { createApp } = Vue;

        // Stat Card Component
        const StatCard = {
            props: ['label', 'value', 'icon', 'color', 'link'],
            template: `
                <div class="stat-card" @click="navigate">
                    <div class="stat-icon" :style="{ color: color }">{{ icon }}</div>
                    <div>
                        <div class="stat-value">{{ value }}</div>
                        <div class="stat-label">{{ label }}</div>
                    </div>
                </div>
            `,
            methods: {
                navigate() {
                    if (this.link) {
                        window.location.href = this.link;
                    }
                }
            }
        };

        // Main App
        createApp({
            components: {
                StatCard
            },
            data() {
                return {
                    loading: true,
                    stats: [],
                    recentAssessments: [],
                    riskDistribution: [],
                    searchQuery: '',
                    filterRisk: ''
                };
            },
            computed: {
                filteredAssessments() {
                    let filtered = this.recentAssessments;

                    // Filter by search query
                    if (this.searchQuery) {
                        const query = this.searchQuery.toLowerCase();
                        filtered = filtered.filter(a => 
                            a.assessment_number.toLowerCase().includes(query) ||
                            a.location_name.toLowerCase().includes(query) ||
                            a.city.toLowerCase().includes(query)
                        );
                    }

                    // Filter by risk level
                    if (this.filterRisk) {
                        filtered = filtered.filter(a => 
                            a.overall_risk_level === this.filterRisk
                        );
                    }

                    return filtered;
                }
            },
            methods: {
                async fetchData() {
                    try {
                        this.loading = true;

                        // Fetch stats
                        const statsResponse = await axios.get('api/dashboard-stats.php');
                        this.stats = statsResponse.data.stats;
                        this.recentAssessments = statsResponse.data.recent;
                        this.riskDistribution = statsResponse.data.risk_distribution;

                    } catch (error) {
                        console.error('Error fetching dashboard data:', error);
                        alert('Error loading dashboard data. Please refresh the page.');
                    } finally {
                        this.loading = false;
                    }
                },
                formatDate(dateString) {
                    const date = new Date(dateString);
                    return date.toLocaleDateString('en-GB', {
                        day: '2-digit',
                        month: 'short',
                        year: 'numeric'
                    });
                },
                getRiskBadge(level) {
                    const badges = {
                        'low': 'success',
                        'medium': 'warning',
                        'high': 'danger',
                        'critical': 'danger'
                    };
                    return badges[level] || 'info';
                },
                getStatusBadge(status) {
                    const badges = {
                        'draft': 'warning',
                        'completed': 'success',
                        'archived': 'info'
                    };
                    return badges[status] || 'info';
                }
            },
            mounted() {
                this.fetchData();
                
                // Refresh data every 30 seconds
                setInterval(() => {
                    this.fetchData();
                }, 30000);
            }
        }).mount('#app');
    </script>
</body>
</html>

<?php include '../templates/footer.php'; ?>
