<?php
require_once '../includes/config.php';
requireLogin();

// Check admin access
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$success = '';
$error = '';
$tableExists = false;

// Check if custom_fields table exists
try {
    $stmt = $pdo->query("SHOW TABLES LIKE 'custom_fields'");
    $tableExists = $stmt->rowCount() > 0;
} catch (PDOException $e) {
    $error = "Database error checking tables: " . $e->getMessage();
}

// Create table if it doesn't exist
if (!$tableExists) {
    try {
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS custom_fields (
                id INT AUTO_INCREMENT PRIMARY KEY,
                entity_type VARCHAR(50) NOT NULL COMMENT 'location, assessment, user, etc',
                field_name VARCHAR(100) NOT NULL COMMENT 'Internal field name (no spaces)',
                field_label VARCHAR(200) NOT NULL COMMENT 'Display label',
                field_type VARCHAR(50) NOT NULL COMMENT 'text, textarea, select, number, date, email, phone, checkbox, radio',
                field_options TEXT COMMENT 'JSON array for select/radio/checkbox options',
                default_value TEXT COMMENT 'Default value for new records',
                is_required BOOLEAN DEFAULT FALSE,
                display_order INT DEFAULT 0,
                is_active BOOLEAN DEFAULT TRUE,
                validation_rules TEXT COMMENT 'JSON validation rules',
                help_text TEXT COMMENT 'Help text shown to users',
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY unique_entity_field (entity_type, field_name),
                INDEX idx_entity_type (entity_type),
                INDEX idx_display_order (display_order),
                INDEX idx_active (is_active)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS custom_field_values (
                id INT AUTO_INCREMENT PRIMARY KEY,
                field_id INT NOT NULL,
                entity_id INT NOT NULL COMMENT 'ID of the location/assessment/user record',
                field_value TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_field_entity (field_id, entity_id),
                INDEX idx_entity_id (entity_id),
                FOREIGN KEY (field_id) REFERENCES custom_fields(id) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ");
        
        $success = "Custom fields tables created successfully!";
        $tableExists = true;
    } catch (PDOException $e) {
        $error = "Could not create custom_fields table: " . $e->getMessage();
    }
}

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $tableExists) {
    try {
        if (isset($_POST['add_field'])) {
            $stmt = $pdo->prepare("
                INSERT INTO custom_fields (entity_type, field_name, field_label, field_type, field_options, is_required, display_order)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $_POST['entity_type'],
                $_POST['field_name'],
                $_POST['field_label'],
                $_POST['field_type'],
                $_POST['field_options'] ?? null,
                isset($_POST['is_required']) ? 1 : 0,
                $_POST['display_order'] ?? 0
            ]);
            $success = "Custom field added successfully!";
        } elseif (isset($_POST['delete_field'])) {
            $stmt = $pdo->prepare("DELETE FROM custom_fields WHERE id = ?");
            $stmt->execute([$_POST['field_id']]);
            $success = "Custom field deleted successfully!";
        }
    } catch (PDOException $e) {
        $error = "Database error: " . $e->getMessage();
        error_log("Custom fields error: " . $e->getMessage());
    }
}

// Get all custom fields
$fields = [];
if ($tableExists) {
    try {
        $stmt = $pdo->query("SELECT * FROM custom_fields ORDER BY entity_type, display_order");
        $fields = $stmt->fetchAll();
    } catch (PDOException $e) {
        $error = "Could not load custom fields: " . $e->getMessage();
    }
}

$pageTitle = "Custom Fields Management";
include '../templates/header.php';
?>

<div class="container">
    <div class="page-header">
        <h1>🔧 Custom Fields Management</h1>
        <a href="admin-panel.php" class="btn btn-secondary">← Back</a>
    </div>

    <?php if ($success): ?>
        <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
    <?php endif; ?>
    
    <?php if ($error): ?>
        <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
    <?php endif; ?>
    
    <?php if (!$tableExists): ?>
        <div class="card">
            <div class="card-header">
                <h2>⚠️ Custom Fields Table Missing</h2>
            </div>
            <div class="card-body">
                <p>The custom_fields table doesn't exist in your database yet.</p>
                <p><strong>To fix this:</strong></p>
                <ol>
                    <li>Run the complete schema: <code>mysql -u user -p database &lt; schema.sql</code></li>
                    <li>Or refresh this page - the system will try to create the table automatically</li>
                </ol>
                <form method="POST">
                    <button type="submit" class="btn btn-primary">Create Tables Now</button>
                </form>
            </div>
        </div>
    <?php else: ?>

    <div class="card">
        <div class="card-header">
            <h2>Add New Custom Field</h2>
        </div>
        <div class="card-body">
            <form method="POST">
                <div class="form-group">
                    <label>Entity Type *</label>
                    <select name="entity_type" class="form-control" required>
                        <option value="location">Location</option>
                        <option value="assessment">Assessment</option>
                        <option value="user">User</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Field Name *</label>
                    <input type="text" name="field_name" class="form-control" required pattern="[a-z_]+">
                </div>
                <div class="form-group">
                    <label>Field Label *</label>
                    <input type="text" name="field_label" class="form-control" required>
                </div>
                <div class="form-group">
                    <label>Field Type *</label>
                    <select name="field_type" class="form-control" required>
                        <option value="text">Text</option>
                        <option value="textarea">Text Area</option>
                        <option value="number">Number</option>
                        <option value="date">Date</option>
                    </select>
                </div>
                <button type="submit" name="add_field" class="btn btn-primary">Add Field</button>
            </form>
        </div>
    </div>

    <div class="card mt-4">
        <div class="card-header">
            <h2>Existing Fields</h2>
        </div>
        <div class="card-body">
            <?php if (empty($fields)): ?>
                <p>No custom fields defined.</p>
            <?php else: ?>
                <table class="table">
                    <thead>
                        <tr>
                            <th>Entity</th>
                            <th>Name</th>
                            <th>Label</th>
                            <th>Type</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($fields as $field): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($field['entity_type']); ?></td>
                                <td><?php echo htmlspecialchars($field['field_name']); ?></td>
                                <td><?php echo htmlspecialchars($field['field_label']); ?></td>
                                <td><?php echo htmlspecialchars($field['field_type']); ?></td>
                                <td>
                                    <form method="POST" style="display:inline;">
                                        <input type="hidden" name="field_id" value="<?php echo $field['id']; ?>">
                                        <button type="submit" name="delete_field" class="btn btn-sm btn-danger" 
                                                onclick="return confirm('Delete this field?')">Delete</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; // End tableExists check ?>
</div>

<?php include '../templates/footer.php'; ?>
