<?php
/**
 * Risk Item Photos API
 * Handle photo uploads for individual risk items
 */

require_once '../../includes/config.php';
requireLogin();

header('Content-Type: application/json');

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch($method) {
        case 'POST':
            // Upload photos for a risk item
            if (!isset($_POST['assessment_id']) || !isset($_POST['risk_item_id'])) {
                throw new Exception('Assessment ID and Risk Item ID are required');
            }
            
            $assessment_id = (int)$_POST['assessment_id'];
            $risk_item_id = (int)$_POST['risk_item_id'];
            $assessment_detail_id = isset($_POST['assessment_detail_id']) ? (int)$_POST['assessment_detail_id'] : null;
            $caption = $_POST['caption'] ?? '';
            
            // Validate assessment exists and user has access
            $stmt = $pdo->prepare("SELECT id FROM assessments WHERE id = ?");
            $stmt->execute([$assessment_id]);
            if (!$stmt->fetch()) {
                throw new Exception('Assessment not found');
            }
            
            // Handle file upload
            if (!isset($_FILES['photo']) || $_FILES['photo']['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('No photo uploaded or upload error occurred');
            }
            
            $file = $_FILES['photo'];
            
            // Validate file type
            $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $file['tmp_name']);
            finfo_close($finfo);
            
            if (!in_array($mimeType, $allowedTypes)) {
                throw new Exception('Invalid file type. Only JPG, PNG, GIF, and WebP images are allowed.');
            }
            
            // Validate file size (max 5MB)
            if ($file['size'] > 5 * 1024 * 1024) {
                throw new Exception('File too large. Maximum size is 5MB.');
            }
            
            // Create upload directory structure
            $uploadDir = __DIR__ . '/../uploads/risk-items/';
            $assessmentDir = $uploadDir . $assessment_id . '/';
            $riskDir = $assessmentDir . $risk_item_id . '/';
            
            if (!file_exists($riskDir)) {
                mkdir($riskDir, 0755, true);
            }
            
            // Generate unique filename
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = uniqid('risk_') . '_' . time() . '.' . $extension;
            $filepath = $riskDir . $filename;
            $relativePath = 'uploads/risk-items/' . $assessment_id . '/' . $risk_item_id . '/' . $filename;
            
            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $filepath)) {
                throw new Exception('Failed to save uploaded file');
            }
            
            // Get or create assessment_detail_id if not provided
            if (!$assessment_detail_id) {
                $stmt = $pdo->prepare("
                    SELECT id FROM assessment_details 
                    WHERE assessment_id = ? AND risk_item_id = ?
                ");
                $stmt->execute([$assessment_id, $risk_item_id]);
                $detail = $stmt->fetch();
                
                if ($detail) {
                    $assessment_detail_id = $detail['id'];
                } else {
                    // Create a basic assessment detail entry
                    $stmt = $pdo->prepare("
                        INSERT INTO assessment_details 
                        (assessment_id, risk_item_id, likelihood, severity, current_controls)
                        VALUES (?, ?, 1, 1, '')
                    ");
                    $stmt->execute([$assessment_id, $risk_item_id]);
                    $assessment_detail_id = $pdo->lastInsertId();
                }
            }
            
            // Get display order
            $stmt = $pdo->prepare("
                SELECT COALESCE(MAX(display_order), 0) + 1 as next_order
                FROM risk_item_photos
                WHERE assessment_detail_id = ?
            ");
            $stmt->execute([$assessment_detail_id]);
            $displayOrder = $stmt->fetchColumn();
            
            // Save to database
            $stmt = $pdo->prepare("
                INSERT INTO risk_item_photos 
                (assessment_detail_id, assessment_id, risk_item_id, file_path, file_name, 
                 file_size, mime_type, caption, uploaded_by, display_order)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $assessment_detail_id,
                $assessment_id,
                $risk_item_id,
                $relativePath,
                $file['name'],
                $file['size'],
                $mimeType,
                $caption,
                $_SESSION['user_id'],
                $displayOrder
            ]);
            
            $photoId = $pdo->lastInsertId();
            
            echo json_encode([
                'success' => true,
                'photo_id' => $photoId,
                'filename' => $filename,
                'path' => $relativePath,
                'message' => 'Photo uploaded successfully'
            ]);
            break;
        
        case 'GET':
            // Get photos for a risk item
            if (isset($_GET['assessment_detail_id'])) {
                $stmt = $pdo->prepare("
                    SELECT rip.*, u.full_name as uploader_name
                    FROM risk_item_photos rip
                    JOIN users u ON rip.uploaded_by = u.id
                    WHERE rip.assessment_detail_id = ?
                    ORDER BY rip.display_order, rip.uploaded_at
                ");
                $stmt->execute([$_GET['assessment_detail_id']]);
                $photos = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode([
                    'success' => true,
                    'photos' => $photos
                ]);
            } elseif (isset($_GET['assessment_id']) && isset($_GET['risk_item_id'])) {
                $stmt = $pdo->prepare("
                    SELECT rip.*, u.full_name as uploader_name
                    FROM risk_item_photos rip
                    JOIN users u ON rip.uploaded_by = u.id
                    WHERE rip.assessment_id = ? AND rip.risk_item_id = ?
                    ORDER BY rip.display_order, rip.uploaded_at
                ");
                $stmt->execute([$_GET['assessment_id'], $_GET['risk_item_id']]);
                $photos = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode([
                    'success' => true,
                    'photos' => $photos
                ]);
            } elseif (isset($_GET['assessment_id'])) {
                // Get all photos for an assessment grouped by risk item
                $stmt = $pdo->prepare("
                    SELECT rip.*, ri.item_name, rc.category_name, u.full_name as uploader_name
                    FROM risk_item_photos rip
                    JOIN risk_items ri ON rip.risk_item_id = ri.id
                    JOIN risk_categories rc ON ri.category_id = rc.id
                    JOIN users u ON rip.uploaded_by = u.id
                    WHERE rip.assessment_id = ?
                    ORDER BY rc.display_order, ri.display_order, rip.display_order
                ");
                $stmt->execute([$_GET['assessment_id']]);
                $photos = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode([
                    'success' => true,
                    'photos' => $photos
                ]);
            } else {
                throw new Exception('Missing required parameters');
            }
            break;
        
        case 'DELETE':
            // Delete a photo
            if (!isset($_GET['id'])) {
                throw new Exception('Photo ID required');
            }
            
            $photoId = (int)$_GET['id'];
            
            // Get photo info
            $stmt = $pdo->prepare("SELECT * FROM risk_item_photos WHERE id = ?");
            $stmt->execute([$photoId]);
            $photo = $stmt->fetch();
            
            if (!$photo) {
                throw new Exception('Photo not found');
            }
            
            // Check permissions (admin or uploader)
            if ($_SESSION['role'] !== 'admin' && $_SESSION['user_id'] != $photo['uploaded_by']) {
                throw new Exception('Permission denied');
            }
            
            // Delete file
            $filePath = __DIR__ . '/../' . $photo['file_path'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            
            // Delete from database
            $stmt = $pdo->prepare("DELETE FROM risk_item_photos WHERE id = ?");
            $stmt->execute([$photoId]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Photo deleted successfully'
            ]);
            break;
        
        case 'PUT':
            // Update photo caption
            parse_str(file_get_contents('php://input'), $_PUT);
            
            if (!isset($_PUT['id'])) {
                throw new Exception('Photo ID required');
            }
            
            $photoId = (int)$_PUT['id'];
            $caption = $_PUT['caption'] ?? '';
            
            // Get photo to check permissions
            $stmt = $pdo->prepare("SELECT uploaded_by FROM risk_item_photos WHERE id = ?");
            $stmt->execute([$photoId]);
            $photo = $stmt->fetch();
            
            if (!$photo) {
                throw new Exception('Photo not found');
            }
            
            // Check permissions
            if ($_SESSION['role'] !== 'admin' && $_SESSION['user_id'] != $photo['uploaded_by']) {
                throw new Exception('Permission denied');
            }
            
            // Update caption
            $stmt = $pdo->prepare("UPDATE risk_item_photos SET caption = ? WHERE id = ?");
            $stmt->execute([$caption, $photoId]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Caption updated successfully'
            ]);
            break;
        
        default:
            throw new Exception('Method not allowed');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
