<?php
/**
 * Location Documents API
 * Handle document uploads for locations (planning permissions, etc.)
 */

require_once '../../includes/config.php';
requireLogin();

header('Content-Type: application/json');

$method = $_SERVER['REQUEST_METHOD'];

try {
    switch($method) {
        case 'POST':
            // Upload document
            if (!isset($_POST['location_id'])) {
                throw new Exception('Location ID is required');
            }
            
            $location_id = (int)$_POST['location_id'];
            $document_type = $_POST['document_type'] ?? 'planning_permission';
            $title = $_POST['title'] ?? '';
            $description = $_POST['description'] ?? '';
            $document_date = $_POST['document_date'] ?? null;
            $expiry_date = $_POST['expiry_date'] ?? null;
            $reference_number = $_POST['reference_number'] ?? '';
            
            // Validate location exists
            $stmt = $pdo->prepare("SELECT id, location_name FROM locations WHERE id = ?");
            $stmt->execute([$location_id]);
            $location = $stmt->fetch();
            
            if (!$location) {
                throw new Exception('Location not found');
            }
            
            // Handle file upload
            if (!isset($_FILES['document']) || $_FILES['document']['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('No document uploaded or upload error occurred');
            }
            
            $file = $_FILES['document'];
            
            // Validate file type (documents only)
            $allowedTypes = [
                'application/pdf',
                'application/msword',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                'application/vnd.ms-excel',
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'image/jpeg',
                'image/jpg',
                'image/png',
                'image/gif',
                'text/plain'
            ];
            
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mimeType = finfo_file($finfo, $file['tmp_name']);
            finfo_close($finfo);
            
            if (!in_array($mimeType, $allowedTypes)) {
                throw new Exception('Invalid file type. Allowed: PDF, Word, Excel, Images, Text files.');
            }
            
            // Validate file size (max 10MB)
            if ($file['size'] > 10 * 1024 * 1024) {
                throw new Exception('File too large. Maximum size is 10MB.');
            }
            
            // Create upload directory structure
            $uploadDir = __DIR__ . '/../uploads/locations/';
            $locationDir = $uploadDir . $location_id . '/';
            $documentsDir = $locationDir . 'documents/';
            
            if (!file_exists($documentsDir)) {
                mkdir($documentsDir, 0755, true);
            }
            
            // Generate unique filename
            $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
            $filename = uniqid('doc_') . '_' . time() . '.' . $extension;
            $filepath = $documentsDir . $filename;
            $relativePath = 'uploads/locations/' . $location_id . '/documents/' . $filename;
            
            // Move uploaded file
            if (!move_uploaded_file($file['tmp_name'], $filepath)) {
                throw new Exception('Failed to save uploaded file');
            }
            
            // Get display order
            $stmt = $pdo->prepare("
                SELECT COALESCE(MAX(display_order), 0) + 1 as next_order
                FROM location_documents
                WHERE location_id = ?
            ");
            $stmt->execute([$location_id]);
            $displayOrder = $stmt->fetchColumn();
            
            // Save to database
            $stmt = $pdo->prepare("
                INSERT INTO location_documents 
                (location_id, document_type, file_path, file_name, original_filename,
                 file_size, mime_type, title, description, document_date, expiry_date,
                 reference_number, uploaded_by, display_order)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $location_id,
                $document_type,
                $relativePath,
                $filename,
                $file['name'],
                $file['size'],
                $mimeType,
                $title,
                $description,
                $document_date,
                $expiry_date,
                $reference_number,
                $_SESSION['user_id'],
                $displayOrder
            ]);
            
            $documentId = $pdo->lastInsertId();
            
            echo json_encode([
                'success' => true,
                'document_id' => $documentId,
                'filename' => $filename,
                'path' => $relativePath,
                'message' => 'Document uploaded successfully'
            ]);
            break;
        
        case 'GET':
            // Get documents for a location
            if (isset($_GET['location_id'])) {
                $location_id = (int)$_GET['location_id'];
                $document_type = $_GET['document_type'] ?? null;
                
                $sql = "
                    SELECT ld.*, u.full_name as uploader_name
                    FROM location_documents ld
                    JOIN users u ON ld.uploaded_by = u.id
                    WHERE ld.location_id = ? AND ld.is_active = 1
                ";
                
                $params = [$location_id];
                
                if ($document_type) {
                    $sql .= " AND ld.document_type = ?";
                    $params[] = $document_type;
                }
                
                $sql .= " ORDER BY ld.display_order, ld.uploaded_at DESC";
                
                $stmt = $pdo->prepare($sql);
                $stmt->execute($params);
                $documents = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                // Check for expiring documents
                foreach ($documents as &$doc) {
                    if ($doc['expiry_date']) {
                        $expiry = new DateTime($doc['expiry_date']);
                        $now = new DateTime();
                        $interval = $now->diff($expiry);
                        
                        if ($expiry < $now) {
                            $doc['status'] = 'expired';
                            $doc['status_message'] = 'Expired';
                        } elseif ($interval->days <= 30) {
                            $doc['status'] = 'expiring_soon';
                            $doc['status_message'] = 'Expires in ' . $interval->days . ' days';
                        } else {
                            $doc['status'] = 'valid';
                            $doc['status_message'] = 'Valid';
                        }
                    } else {
                        $doc['status'] = 'no_expiry';
                        $doc['status_message'] = 'No expiry';
                    }
                }
                
                echo json_encode([
                    'success' => true,
                    'documents' => $documents
                ]);
                
            } elseif (isset($_GET['id'])) {
                // Get single document
                $stmt = $pdo->prepare("
                    SELECT ld.*, u.full_name as uploader_name, l.location_name
                    FROM location_documents ld
                    JOIN users u ON ld.uploaded_by = u.id
                    JOIN locations l ON ld.location_id = l.id
                    WHERE ld.id = ?
                ");
                $stmt->execute([$_GET['id']]);
                $document = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$document) {
                    throw new Exception('Document not found');
                }
                
                echo json_encode([
                    'success' => true,
                    'document' => $document
                ]);
            } else {
                throw new Exception('Missing required parameters');
            }
            break;
        
        case 'DELETE':
            // Delete a document
            if (!isset($_GET['id'])) {
                throw new Exception('Document ID required');
            }
            
            $documentId = (int)$_GET['id'];
            
            // Get document info
            $stmt = $pdo->prepare("SELECT * FROM location_documents WHERE id = ?");
            $stmt->execute([$documentId]);
            $document = $stmt->fetch();
            
            if (!$document) {
                throw new Exception('Document not found');
            }
            
            // Check permissions (admin or uploader)
            if ($_SESSION['role'] !== 'admin' && $_SESSION['user_id'] != $document['uploaded_by']) {
                throw new Exception('Permission denied');
            }
            
            // Delete file
            $filePath = __DIR__ . '/../' . $document['file_path'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            
            // Delete from database
            $stmt = $pdo->prepare("DELETE FROM location_documents WHERE id = ?");
            $stmt->execute([$documentId]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Document deleted successfully'
            ]);
            break;
        
        case 'PUT':
            // Update document metadata
            parse_str(file_get_contents('php://input'), $_PUT);
            
            if (!isset($_PUT['id'])) {
                throw new Exception('Document ID required');
            }
            
            $documentId = (int)$_PUT['id'];
            
            // Get document to check permissions
            $stmt = $pdo->prepare("SELECT uploaded_by FROM location_documents WHERE id = ?");
            $stmt->execute([$documentId]);
            $document = $stmt->fetch();
            
            if (!$document) {
                throw new Exception('Document not found');
            }
            
            // Check permissions
            if ($_SESSION['role'] !== 'admin' && $_SESSION['user_id'] != $document['uploaded_by']) {
                throw new Exception('Permission denied');
            }
            
            // Build update query
            $updates = [];
            $params = [];
            
            $allowedFields = ['title', 'description', 'document_date', 'expiry_date', 'reference_number', 'document_type'];
            
            foreach ($allowedFields as $field) {
                if (isset($_PUT[$field])) {
                    $updates[] = "$field = ?";
                    $params[] = $_PUT[$field];
                }
            }
            
            if (empty($updates)) {
                throw new Exception('No fields to update');
            }
            
            $params[] = $documentId;
            $sql = "UPDATE location_documents SET " . implode(', ', $updates) . " WHERE id = ?";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            
            echo json_encode([
                'success' => true,
                'message' => 'Document updated successfully'
            ]);
            break;
        
        default:
            throw new Exception('Method not allowed');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
