#!/usr/bin/env python3
"""
Enhanced PDF Generator with Template Support
Generates PDF reports based on custom templates
"""

import sys
import json
import mysql.connector
from reportlab.lib.pagesizes import A4, LETTER, LEGAL, A3
from reportlab.lib import colors
from reportlab.lib.units import mm
from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph, Spacer, PageBreak, Image
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.enums import TA_LEFT, TA_CENTER, TA_RIGHT
from datetime import datetime
import os

# Database configuration
DB_CONFIG = {
    'host': 'localhost',
    'database': 'scubatricky_risk',
    'user': 'scubatricky_risk',
    'password': ''
}

class PDFGenerator:
    def __init__(self, assessment_id, template_id=None, output_path=None):
        self.assessment_id = assessment_id
        self.template_id = template_id
        self.output_path = output_path or f'assessment_{assessment_id}.pdf'
        self.data = {}
        self.template = None
        
        # Connect to database
        self.conn = mysql.connector.connect(**DB_CONFIG)
        self.cursor = self.conn.cursor(dictionary=True)
        
        # Load data
        self.load_assessment_data()
        if template_id:
            self.load_template()
        
    def load_assessment_data(self):
        """Load all assessment data from database"""
        # Main assessment
        self.cursor.execute("""
            SELECT a.*, l.*, u.full_name as assessor_name, u.email as assessor_email
            FROM assessments a
            JOIN locations l ON a.location_id = l.id
            JOIN users u ON a.assessor_id = u.id
            WHERE a.id = %s
        """, (self.assessment_id,))
        self.data['assessment'] = self.cursor.fetchone()
        
        if not self.data['assessment']:
            raise Exception(f'Assessment {self.assessment_id} not found')
        
        # Assessment details with risk items
        self.cursor.execute("""
            SELECT ad.*, ri.item_name, ri.description as item_description,
                   rc.category_name
            FROM assessment_details ad
            JOIN risk_items ri ON ad.risk_item_id = ri.id
            JOIN risk_categories rc ON ri.category_id = rc.id
            WHERE ad.assessment_id = %s
            ORDER BY rc.display_order, ri.display_order
        """, (self.assessment_id,))
        self.data['details'] = self.cursor.fetchall()
        
        # Photos
        self.cursor.execute("""
            SELECT * FROM photos 
            WHERE assessment_id = %s 
            ORDER BY display_order
        """, (self.assessment_id,))
        self.data['photos'] = self.cursor.fetchall()
        
        # Custom fields for location
        location_id = self.data['assessment']['location_id']
        self.cursor.execute("""
            SELECT * FROM custom_field_definitions 
            WHERE is_active = 1 
            ORDER BY display_order
        """)
        custom_fields_def = self.cursor.fetchall()
        
        self.data['custom_fields'] = {}
        if self.data['assessment'].get('custom_fields'):
            try:
                custom_data = json.loads(self.data['assessment']['custom_fields'])
                for field_def in custom_fields_def:
                    field_name = field_def['field_name']
                    if field_name in custom_data:
                        self.data['custom_fields'][field_def['field_label']] = custom_data[field_name]
            except:
                pass
    
    def load_template(self):
        """Load PDF template configuration"""
        self.cursor.execute("""
            SELECT * FROM pdf_templates WHERE id = %s AND is_active = 1
        """, (self.template_id,))
        template = self.cursor.fetchone()
        
        if not template:
            raise Exception(f'Template {self.template_id} not found')
        
        self.template = template
        try:
            self.template['config'] = json.loads(template['template_config'])
        except:
            self.template['config'] = {}
    
    def get_page_size(self):
        """Get page size based on template"""
        if not self.template:
            return A4
        
        size_map = {
            'A4': A4,
            'Letter': LETTER,
            'Legal': LEGAL,
            'A3': A3
        }
        return size_map.get(self.template['page_size'], A4)
    
    def generate(self):
        """Generate PDF based on template or default format"""
        page_size = self.get_page_size()
        
        # Create PDF document
        doc = SimpleDocTemplate(
            self.output_path,
            pagesize=page_size,
            rightMargin=20*mm,
            leftMargin=20*mm,
            topMargin=20*mm,
            bottomMargin=20*mm
        )
        
        # Build story
        story = []
        styles = getSampleStyleSheet()
        
        if self.template and self.template['config'].get('sections'):
            # Use template
            story = self.build_from_template(styles)
        else:
            # Use default format
            story = self.build_default_format(styles)
        
        # Build PDF
        doc.build(story)
        
        return self.output_path
    
    def build_from_template(self, styles):
        """Build PDF content from template configuration"""
        story = []
        config = self.template['config']
        
        # Header if enabled
        if config.get('header', {}).get('enabled'):
            header = config['header']
            title_style = ParagraphStyle(
                'CustomTitle',
                parent=styles['Heading1'],
                fontSize=18,
                textColor=colors.HexColor('#2563eb'),
                spaceAfter=20
            )
            story.append(Paragraph(header.get('title', 'Report'), title_style))
            story.append(Spacer(1, 10*mm))
        
        # Process sections
        for section in config.get('sections', []):
            section_type = section.get('type')
            
            if section_type == 'heading':
                level = section.get('level', 1)
                style_name = f'Heading{level}'
                story.append(Paragraph(section.get('text', ''), styles[style_name]))
                story.append(Spacer(1, 5*mm))
            
            elif section_type == 'field':
                value = self.get_field_value(section.get('source'), section.get('field'))
                label = section.get('label', section.get('field'))
                
                if section.get('showLabel', True):
                    text = f"<b>{label}:</b> {value}"
                else:
                    text = str(value)
                
                story.append(Paragraph(text, styles['Normal']))
                story.append(Spacer(1, 3*mm))
            
            elif section_type == 'table':
                table = self.build_table_section(section)
                if table:
                    story.append(table)
                    story.append(Spacer(1, 5*mm))
        
        return story
    
    def build_default_format(self, styles):
        """Build PDF with default comprehensive format"""
        story = []
        assessment = self.data['assessment']
        
        # Title
        title_style = ParagraphStyle(
            'CustomTitle',
            parent=styles['Heading1'],
            fontSize=20,
            textColor=colors.HexColor('#2563eb'),
            spaceAfter=20
        )
        story.append(Paragraph('Risk Assessment Report', title_style))
        story.append(Spacer(1, 10*mm))
        
        # Assessment Information
        story.append(Paragraph('<b>Assessment Information</b>', styles['Heading2']))
        info_data = [
            ['Assessment Number:', assessment['assessment_number']],
            ['Date:', assessment['assessment_date'].strftime('%d/%m/%Y') if assessment.get('assessment_date') else 'N/A'],
            ['Time:', str(assessment.get('assessment_time', 'N/A'))],
            ['Assessor:', assessment.get('assessor_name', 'N/A')],
            ['Overall Risk Level:', assessment.get('overall_risk_level', 'N/A').upper()],
            ['Status:', assessment.get('status', 'N/A').upper()]
        ]
        
        info_table = Table(info_data, colWidths=[60*mm, 120*mm])
        info_table.setStyle(TableStyle([
            ('FONT', (0, 0), (0, -1), 'Helvetica-Bold'),
            ('FONT', (1, 0), (1, -1), 'Helvetica'),
            ('FONTSIZE', (0, 0), (-1, -1), 10),
            ('TEXTCOLOR', (0, 0), (0, -1), colors.HexColor('#64748b')),
            ('VALIGN', (0, 0), (-1, -1), 'TOP'),
            ('BOTTOMPADDING', (0, 0), (-1, -1), 8),
        ]))
        story.append(info_table)
        story.append(Spacer(1, 10*mm))
        
        # Location Information
        story.append(Paragraph('<b>Location Details</b>', styles['Heading2']))
        location_data = [
            ['Location Name:', assessment.get('location_name', 'N/A')],
            ['Address:', assessment.get('address', 'N/A')],
            ['City:', assessment.get('city', 'N/A')],
            ['Postcode:', assessment.get('postcode', 'N/A')],
            ['Contact:', assessment.get('contact_name', 'N/A')],
            ['Phone:', assessment.get('contact_phone', 'N/A')]
        ]
        
        location_table = Table(location_data, colWidths=[60*mm, 120*mm])
        location_table.setStyle(TableStyle([
            ('FONT', (0, 0), (0, -1), 'Helvetica-Bold'),
            ('FONT', (1, 0), (1, -1), 'Helvetica'),
            ('FONTSIZE', (0, 0), (-1, -1), 10),
            ('TEXTCOLOR', (0, 0), (0, -1), colors.HexColor('#64748b')),
            ('VALIGN', (0, 0), (-1, -1), 'TOP'),
            ('BOTTOMPADDING', (0, 0), (-1, -1), 8),
        ]))
        story.append(location_table)
        story.append(Spacer(1, 10*mm))
        
        # Risk Assessment Details
        if self.data['details']:
            story.append(Paragraph('<b>Risk Evaluation</b>', styles['Heading2']))
            
            # Group by category
            by_category = {}
            for detail in self.data['details']:
                cat = detail['category_name']
                if cat not in by_category:
                    by_category[cat] = []
                by_category[cat].append(detail)
            
            for category, items in by_category.items():
                story.append(Paragraph(f'<b>{category}</b>', styles['Heading3']))
                
                # Build risk table
                table_data = [['Risk Item', 'L', 'S', 'Score', 'Current Controls']]
                for item in items:
                    table_data.append([
                        item['item_name'],
                        str(item['likelihood']),
                        str(item['severity']),
                        str(item['risk_score']),
                        item.get('current_controls', '')[:50] + ('...' if len(item.get('current_controls', '')) > 50 else '')
                    ])
                
                risk_table = Table(table_data, colWidths=[50*mm, 10*mm, 10*mm, 15*mm, 95*mm])
                risk_table.setStyle(TableStyle([
                    ('BACKGROUND', (0, 0), (-1, 0), colors.HexColor('#2563eb')),
                    ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),
                    ('FONT', (0, 0), (-1, 0), 'Helvetica-Bold'),
                    ('FONTSIZE', (0, 0), (-1, 0), 10),
                    ('FONT', (0, 1), (-1, -1), 'Helvetica'),
                    ('FONTSIZE', (0, 1), (-1, -1), 9),
                    ('GRID', (0, 0), (-1, -1), 0.5, colors.grey),
                    ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                    ('ROWBACKGROUNDS', (0, 1), (-1, -1), [colors.white, colors.HexColor('#f8f9fa')])
                ]))
                story.append(risk_table)
                story.append(Spacer(1, 5*mm))
        
        return story
    
    def get_field_value(self, source, field):
        """Get field value from loaded data"""
        if source == 'assessments':
            return self.data['assessment'].get(field, 'N/A')
        elif source == 'locations':
            return self.data['assessment'].get(field, 'N/A')
        elif source == 'users':
            if field == 'full_name':
                return self.data['assessment'].get('assessor_name', 'N/A')
            return self.data['assessment'].get(f'assessor_{field}', 'N/A')
        return 'N/A'
    
    def build_table_section(self, section):
        """Build a table from assessment details"""
        if not self.data['details']:
            return None
        
        columns = section.get('columns', [])
        headers = section.get('headers', columns)
        
        # Build table data
        table_data = [headers]
        for detail in self.data['details']:
            row = []
            for col in columns:
                if col == 'risk_item':
                    row.append(detail.get('item_name', ''))
                else:
                    row.append(str(detail.get(col, '')))
            table_data.append(row)
        
        # Create table
        table = Table(table_data)
        table.setStyle(TableStyle([
            ('BACKGROUND', (0, 0), (-1, 0), colors.HexColor('#2563eb')),
            ('TEXTCOLOR', (0, 0), (-1, 0), colors.whitesmoke),
            ('FONT', (0, 0), (-1, 0), 'Helvetica-Bold'),
            ('GRID', (0, 0), (-1, -1), 0.5, colors.grey),
            ('VALIGN', (0, 0), (-1, -1), 'TOP'),
        ]))
        
        return table
    
    def __del__(self):
        """Close database connection"""
        if hasattr(self, 'cursor'):
            self.cursor.close()
        if hasattr(self, 'conn'):
            self.conn.close()

def main():
    """Main entry point"""
    if len(sys.argv) < 2:
        print(json.dumps({'success': False, 'error': 'Assessment ID required'}))
        sys.exit(1)
    
    assessment_id = sys.argv[1]
    template_id = sys.argv[2] if len(sys.argv) > 2 else None
    output_path = sys.argv[3] if len(sys.argv) > 3 else None
    
    try:
        generator = PDFGenerator(assessment_id, template_id, output_path)
        pdf_path = generator.generate()
        
        print(json.dumps({
            'success': True,
            'file': pdf_path,
            'filename': os.path.basename(pdf_path)
        }))
        
    except Exception as e:
        print(json.dumps({
            'success': False,
            'error': str(e)
        }))
        sys.exit(1)

if __name__ == '__main__':
    main()
