#!/usr/bin/env python3
"""
Risk Assessment PDF Generator
Generates professional PDF reports with customizable styling
"""

import sys
import json
from datetime import datetime
from reportlab.lib.pagesizes import A4
from reportlab.lib import colors
from reportlab.lib.units import mm, inch
from reportlab.platypus import SimpleDocTemplate, Table, TableStyle, Paragraph, Spacer, PageBreak, Image
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.enums import TA_CENTER, TA_LEFT, TA_RIGHT
from reportlab.pdfgen import canvas

class RiskAssessmentPDF:
    def __init__(self, data_file, output_file):
        self.output_file = output_file
        
        # Load data
        with open(data_file, 'r') as f:
            data = json.load(f)
            self.assessment = data['assessment']
            self.details = data['details']
        
        # Setup document
        self.doc = SimpleDocTemplate(
            output_file,
            pagesize=A4,
            rightMargin=20*mm,
            leftMargin=20*mm,
            topMargin=25*mm,
            bottomMargin=25*mm
        )
        
        # Setup styles
        self.styles = getSampleStyleSheet()
        self.setup_custom_styles()
        
        # Story elements
        self.story = []
    
    def setup_custom_styles(self):
        """Create custom paragraph styles"""
        # Title style
        self.styles.add(ParagraphStyle(
            name='CustomTitle',
            parent=self.styles['Heading1'],
            fontSize=24,
            textColor=colors.HexColor('#D97706'),
            spaceAfter=30,
            alignment=TA_CENTER,
            fontName='Helvetica-Bold'
        ))
        
        # Subtitle style
        self.styles.add(ParagraphStyle(
            name='CustomSubtitle',
            parent=self.styles['Normal'],
            fontSize=12,
            textColor=colors.HexColor('#64748B'),
            spaceAfter=20,
            alignment=TA_CENTER
        ))
        
        # Section header
        self.styles.add(ParagraphStyle(
            name='SectionHeader',
            parent=self.styles['Heading2'],
            fontSize=16,
            textColor=colors.HexColor('#1E293B'),
            spaceAfter=12,
            spaceBefore=20,
            fontName='Helvetica-Bold',
            borderWidth=1,
            borderColor=colors.HexColor('#D97706'),
            borderPadding=5,
            backColor=colors.HexColor('#FEF3C7')
        ))
        
        # Risk level styles
        risk_colors = {
            'low': '#059669',
            'medium': '#F59E0B',
            'high': '#EF4444',
            'critical': '#DC2626'
        }
        
        for level, color in risk_colors.items():
            self.styles.add(ParagraphStyle(
                name=f'Risk{level.capitalize()}',
                parent=self.styles['Normal'],
                fontSize=11,
                textColor=colors.white,
                backColor=colors.HexColor(color),
                borderRadius=3,
                alignment=TA_CENTER
            ))
    
    def add_header(self):
        """Add document header"""
        # Title
        title = Paragraph("RISK ASSESSMENT REPORT", self.styles['CustomTitle'])
        self.story.append(title)
        
        # Subtitle
        subtitle = Paragraph(
            f"Supermarket Delivery Point Assessment<br/>{self.assessment['location_name']}",
            self.styles['CustomSubtitle']
        )
        self.story.append(subtitle)
        self.story.append(Spacer(1, 20))
        
        # Assessment info box
        info_data = [
            ['Assessment Number:', self.assessment['assessment_number']],
            ['Assessment Date:', datetime.strptime(self.assessment['assessment_date'], '%Y-%m-%d').strftime('%d/%m/%Y')],
            ['Assessor:', self.assessment['assessor_name']],
            ['Location:', f"{self.assessment['location_name']}, {self.assessment['city']}"],
            ['Overall Risk Level:', self.get_risk_badge(self.assessment['overall_risk_level'])],
            ['Status:', self.assessment['status'].upper()]
        ]
        
        info_table = Table(info_data, colWidths=[50*mm, 120*mm])
        info_table.setStyle(TableStyle([
            ('BACKGROUND', (0, 0), (0, -1), colors.HexColor('#F1F5F9')),
            ('TEXTCOLOR', (0, 0), (-1, -1), colors.HexColor('#1E293B')),
            ('ALIGN', (0, 0), (0, -1), 'RIGHT'),
            ('ALIGN', (1, 0), (1, -1), 'LEFT'),
            ('FONTNAME', (0, 0), (0, -1), 'Helvetica-Bold'),
            ('FONTNAME', (1, 0), (1, -1), 'Helvetica'),
            ('FONTSIZE', (0, 0), (-1, -1), 10),
            ('GRID', (0, 0), (-1, -1), 1, colors.HexColor('#CBD5E1')),
            ('VALIGN', (0, 0), (-1, -1), 'MIDDLE'),
            ('ROWBACKGROUNDS', (0, 0), (-1, -1), [colors.white, colors.HexColor('#F8FAFC')]),
            ('TOPPADDING', (0, 0), (-1, -1), 8),
            ('BOTTOMPADDING', (0, 0), (-1, -1), 8),
        ]))
        
        self.story.append(info_table)
        self.story.append(Spacer(1, 20))
    
    def add_location_details(self):
        """Add location details section"""
        self.story.append(Paragraph("LOCATION DETAILS", self.styles['SectionHeader']))
        self.story.append(Spacer(1, 10))
        
        location_text = f"""
        <b>Address:</b> {self.assessment['address_line1']}<br/>
        {self.assessment['address_line2'] or ''}<br/>
        {self.assessment['city']}, {self.assessment['postcode']}<br/>
        {self.assessment['region'] or ''}<br/><br/>
        
        <b>Contact Information:</b><br/>
        Name: {self.assessment['contact_name'] or 'N/A'}<br/>
        Phone: {self.assessment['contact_phone'] or 'N/A'}<br/>
        Email: {self.assessment['contact_email'] or 'N/A'}
        """
        
        self.story.append(Paragraph(location_text, self.styles['Normal']))
        self.story.append(Spacer(1, 20))
    
    def add_risk_matrix(self):
        """Add risk assessment matrix"""
        self.story.append(Paragraph("RISK MATRIX", self.styles['SectionHeader']))
        self.story.append(Spacer(1, 10))
        
        # Create risk matrix
        matrix_data = [
            ['', 'Severity →', '1', '2', '3', '4', '5'],
            ['Likelihood ↓', '5', '5', '10', '15', '20', '25'],
            ['', '4', '4', '8', '12', '16', '20'],
            ['', '3', '3', '6', '9', '12', '15'],
            ['', '2', '2', '4', '6', '8', '10'],
            ['', '1', '1', '2', '3', '4', '5']
        ]
        
        matrix = Table(matrix_data, colWidths=[30*mm, 25*mm, 20*mm, 20*mm, 20*mm, 20*mm, 20*mm])
        
        # Color coding for risk levels
        def get_risk_color(score):
            score = int(score) if score.isdigit() else 0
            if score <= 4:
                return colors.HexColor('#D1FAE5')  # Low - green
            elif score <= 9:
                return colors.HexColor('#FEF3C7')  # Medium - yellow
            elif score <= 16:
                return colors.HexColor('#FED7AA')  # High - orange
            else:
                return colors.HexColor('#FECACA')  # Critical - red
        
        # Apply styling
        style_commands = [
            ('BACKGROUND', (0, 0), (1, 0), colors.HexColor('#1E293B')),
            ('BACKGROUND', (0, 1), (1, 1), colors.HexColor('#1E293B')),
            ('TEXTCOLOR', (0, 0), (1, 0), colors.white),
            ('TEXTCOLOR', (0, 1), (1, -1), colors.white),
            ('FONTNAME', (0, 0), (-1, -1), 'Helvetica-Bold'),
            ('FONTSIZE', (0, 0), (-1, -1), 10),
            ('ALIGN', (0, 0), (-1, -1), 'CENTER'),
            ('VALIGN', (0, 0), (-1, -1), 'MIDDLE'),
            ('GRID', (0, 0), (-1, -1), 1, colors.HexColor('#94A3B8'))
        ]
        
        # Color cells based on risk score
        for row in range(1, 6):
            for col in range(2, 7):
                score = matrix_data[row][col]
                color = get_risk_color(score)
                style_commands.append(('BACKGROUND', (col, row), (col, row), color))
        
        matrix.setStyle(TableStyle(style_commands))
        self.story.append(matrix)
        self.story.append(Spacer(1, 10))
        
        # Legend
        legend_text = """
        <b>Risk Levels:</b> 1-4: Low (Green) | 5-9: Medium (Yellow) | 10-16: High (Orange) | 17-25: Critical (Red)
        """
        self.story.append(Paragraph(legend_text, self.styles['Normal']))
        self.story.append(Spacer(1, 20))
    
    def add_assessment_details(self):
        """Add detailed risk assessment items"""
        self.story.append(Paragraph("DETAILED RISK ASSESSMENT", self.styles['SectionHeader']))
        self.story.append(Spacer(1, 10))
        
        if not self.details:
            self.story.append(Paragraph("No risk assessment details recorded.", self.styles['Normal']))
            return
        
        # Group by category
        current_category = None
        
        for detail in self.details:
            # Add category header if changed
            if detail['category_name'] != current_category:
                current_category = detail['category_name']
                self.story.append(Spacer(1, 10))
                category_style = ParagraphStyle(
                    name='CategoryTemp',
                    parent=self.styles['Heading3'],
                    fontSize=13,
                    textColor=colors.HexColor('#D97706'),
                    spaceBefore=10,
                    spaceAfter=8
                )
                self.story.append(Paragraph(f"• {current_category}", category_style))
            
            # Create detail table
            risk_score = detail['risk_score']
            risk_level = self.calculate_risk_level(risk_score)
            
            detail_data = [
                ['Risk Item:', detail['item_name']],
                ['Likelihood:', f"{detail['likelihood']}/5"],
                ['Severity:', f"{detail['severity']}/5"],
                ['Risk Score:', f"{risk_score} ({risk_level.upper()})"],
                ['Current Controls:', detail['current_controls'] or 'None specified'],
                ['Additional Actions:', detail['additional_actions'] or 'None required'],
            ]
            
            if detail['action_owner']:
                detail_data.append(['Action Owner:', detail['action_owner']])
            if detail['target_date']:
                detail_data.append(['Target Date:', datetime.strptime(detail['target_date'], '%Y-%m-%d').strftime('%d/%m/%Y')])
            
            detail_table = Table(detail_data, colWidths=[45*mm, 125*mm])
            detail_table.setStyle(TableStyle([
                ('BACKGROUND', (0, 0), (0, -1), colors.HexColor('#F8FAFC')),
                ('TEXTCOLOR', (0, 0), (-1, -1), colors.HexColor('#1E293B')),
                ('ALIGN', (0, 0), (0, -1), 'RIGHT'),
                ('ALIGN', (1, 0), (1, -1), 'LEFT'),
                ('FONTNAME', (0, 0), (0, -1), 'Helvetica-Bold'),
                ('FONTNAME', (1, 0), (1, -1), 'Helvetica'),
                ('FONTSIZE', (0, 0), (-1, -1), 9),
                ('GRID', (0, 0), (-1, -1), 0.5, colors.HexColor('#CBD5E1')),
                ('VALIGN', (0, 0), (-1, -1), 'TOP'),
                ('TOPPADDING', (0, 0), (-1, -1), 6),
                ('BOTTOMPADDING', (0, 0), (-1, -1), 6),
                ('LEFTPADDING', (0, 0), (-1, -1), 8),
                ('RIGHTPADDING', (0, 0), (-1, -1), 8),
            ]))
            
            self.story.append(detail_table)
            self.story.append(Spacer(1, 8))
    
    def add_recommendations(self):
        """Add recommendations section"""
        if self.assessment.get('recommendations'):
            self.story.append(PageBreak())
            self.story.append(Paragraph("RECOMMENDATIONS", self.styles['SectionHeader']))
            self.story.append(Spacer(1, 10))
            self.story.append(Paragraph(self.assessment['recommendations'], self.styles['Normal']))
            self.story.append(Spacer(1, 20))
    
    def add_footer_info(self):
        """Add footer information"""
        self.story.append(Spacer(1, 30))
        
        footer_text = f"""
        <para alignment="center">
        <b>Report Generated:</b> {datetime.now().strftime('%d/%m/%Y %H:%M')}<br/>
        <b>Generated By:</b> Risk Assessment System v1.0<br/>
        <i>This report is confidential and for internal use only.</i>
        </para>
        """
        
        self.story.append(Paragraph(footer_text, self.styles['Normal']))
    
    def get_risk_badge(self, level):
        """Get colored risk level badge"""
        colors_map = {
            'low': '#059669',
            'medium': '#F59E0B',
            'high': '#EF4444',
            'critical': '#DC2626'
        }
        color = colors_map.get(level, '#64748B')
        return f'<font color="{color}"><b>{level.upper()}</b></font>'
    
    def calculate_risk_level(self, score):
        """Calculate risk level from score"""
        if score <= 4:
            return 'low'
        elif score <= 9:
            return 'medium'
        elif score <= 16:
            return 'high'
        else:
            return 'critical'
    
    def add_page_number(self, canvas, doc):
        """Add page numbers to each page"""
        canvas.saveState()
        canvas.setFont('Helvetica', 9)
        page_num = canvas.getPageNumber()
        text = f"Page {page_num}"
        canvas.drawRightString(200*mm - 20*mm, 15*mm, text)
        
        # Add company branding
        canvas.setFillColorRGB(0.85, 0.47, 0.02)  # Primary color
        canvas.setFont('Helvetica-Bold', 10)
        canvas.drawString(20*mm, 15*mm, "Risk Assessment System")
        
        canvas.restoreState()
    
    def generate(self):
        """Generate the PDF report"""
        # Add all sections
        self.add_header()
        self.add_location_details()
        self.add_risk_matrix()
        self.add_assessment_details()
        self.add_recommendations()
        self.add_footer_info()
        
        # Build PDF
        self.doc.build(self.story, onFirstPage=self.add_page_number, onLaterPages=self.add_page_number)
        
        print(f"PDF generated successfully: {self.output_file}")
        return True

def main():
    if len(sys.argv) < 3:
        print("Usage: python generate_pdf.py <data_file> <output_file>")
        sys.exit(1)
    
    data_file = sys.argv[1]
    output_file = sys.argv[2]
    
    try:
        pdf_generator = RiskAssessmentPDF(data_file, output_file)
        pdf_generator.generate()
        sys.exit(0)
    except Exception as e:
        print(f"Error generating PDF: {str(e)}")
        import traceback
        traceback.print_exc()
        sys.exit(1)

if __name__ == "__main__":
    main()
