<?php
/**
 * Safe Admin Panel
 * This version has comprehensive error handling
 */

// Enable error handling
ini_set('display_errors', 0); // Hide errors from users
error_reporting(E_ALL);

// Log errors instead
ini_set('log_errors', 1);
ini_set('error_log', '../logs/admin-panel-errors.log');

try {
    require_once '../includes/config.php';
    requireLogin();

    // Only admins can access
    if ($_SESSION['role'] !== 'admin') {
        header('Location: dashboard.php');
        exit;
    }

    // Safe function to get count from table
    function safeCount($pdo, $table, $where = '') {
        try {
            $query = "SELECT COUNT(*) FROM $table";
            if ($where) $query .= " WHERE $where";
            $stmt = $pdo->query($query);
            return $stmt->fetchColumn();
        } catch (Exception $e) {
            error_log("Error counting $table: " . $e->getMessage());
            return 0;
        }
    }

    // Get statistics with safe error handling
    $stats = [
        'total_users' => safeCount($pdo, 'users'),
        'active_users' => safeCount($pdo, 'users', 'is_active = 1'),
        'total_locations' => safeCount($pdo, 'locations'),
        'total_assessments' => safeCount($pdo, 'assessments'),
        'custom_fields' => safeCount($pdo, 'custom_fields', 'is_active = 1'),
        'vehicle_types' => safeCount($pdo, 'vehicle_types', 'is_active = 1'),
        'total_photos' => safeCount($pdo, 'photos'),
        'pdf_templates' => safeCount($pdo, 'pdf_templates', 'is_active = 1')
    ];

    // Recent activity
    $recent_logins = [];
    try {
        $stmt = $pdo->query("
            SELECT u.username, u.last_login 
            FROM users u 
            WHERE u.last_login IS NOT NULL 
            ORDER BY u.last_login DESC 
            LIMIT 5
        ");
        $recent_logins = $stmt->fetchAll();
    } catch (Exception $e) {
        error_log("Error fetching recent logins: " . $e->getMessage());
    }

    include '../templates/header.php';
    
} catch (Exception $e) {
    // Fatal error - show user-friendly message
    echo "<!DOCTYPE html><html><head><title>Error</title></head><body>";
    echo "<h1>System Error</h1>";
    echo "<p>The admin panel encountered an error. Please contact your system administrator.</p>";
    echo "<p><a href='dashboard.php'>Return to Dashboard</a></p>";
    echo "<!-- Error: " . htmlspecialchars($e->getMessage()) . " -->";
    echo "</body></html>";
    error_log("Admin panel fatal error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
    exit;
}
?>

<div class="container">
    <div class="page-header">
        <div>
            <h1 class="page-title">⚙️ Admin Panel</h1>
            <p class="page-subtitle">System administration and configuration</p>
        </div>
    </div>

    <!-- Quick Stats -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon" style="color: var(--primary);">👥</div>
            <div>
                <div class="stat-value"><?php echo number_format($stats['total_users']); ?></div>
                <div class="stat-label">Total Users</div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="color: var(--success);">📍</div>
            <div>
                <div class="stat-value"><?php echo number_format($stats['total_locations']); ?></div>
                <div class="stat-label">Locations</div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="color: var(--info);">📋</div>
            <div>
                <div class="stat-value"><?php echo number_format($stats['total_assessments']); ?></div>
                <div class="stat-label">Assessments</div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="color: var(--warning);">📸</div>
            <div>
                <div class="stat-value"><?php echo number_format($stats['total_photos']); ?></div>
                <div class="stat-label">Photos</div>
            </div>
        </div>
    </div>

    <!-- Admin Cards Grid -->
    <div class="admin-grid">
        <!-- Users -->
        <div class="admin-card" onclick="window.location.href='users.php'">
            <span class="admin-card-icon">👥</span>
            <h3 class="admin-card-title">User Management</h3>
            <p class="admin-card-description">Manage user accounts, roles, and permissions</p>
            <div class="admin-card-stat"><?php echo $stats['active_users']; ?> active users</div>
        </div>

        <!-- Locations -->
        <div class="admin-card" onclick="window.location.href='locations.php'">
            <span class="admin-card-icon">📍</span>
            <h3 class="admin-card-title">Location Management</h3>
            <p class="admin-card-description">Manage delivery locations and store information</p>
            <div class="admin-card-stat"><?php echo $stats['total_locations']; ?> locations</div>
        </div>

        <!-- Risk Categories -->
        <div class="admin-card" onclick="window.location.href='risk-categories.php'">
            <span class="admin-card-icon">⚠️</span>
            <h3 class="admin-card-title">Risk Categories</h3>
            <p class="admin-card-description">Configure risk categories and items for assessments</p>
            <div class="admin-card-stat">Manage risks</div>
        </div>

        <!-- Custom Fields -->
        <div class="admin-card" onclick="window.location.href='custom-fields.php'">
            <span class="admin-card-icon">📝</span>
            <h3 class="admin-card-title">Custom Fields</h3>
            <p class="admin-card-description">Add custom fields to assessment forms</p>
            <div class="admin-card-stat"><?php echo $stats['custom_fields']; ?> fields</div>
        </div>

        <!-- Vehicle Types -->
        <div class="admin-card" onclick="window.location.href='vehicle-types.php'">
            <span class="admin-card-icon">🚚</span>
            <h3 class="admin-card-title">Vehicle Types</h3>
            <p class="admin-card-description">Configure vehicle types for assessments</p>
            <div class="admin-card-stat"><?php echo $stats['vehicle_types']; ?> types</div>
        </div>

        <!-- PDF Templates -->
        <div class="admin-card" onclick="window.location.href='pdf-templates.php'">
            <span class="admin-card-icon">📄</span>
            <h3 class="admin-card-title">PDF Templates</h3>
            <p class="admin-card-description">Customize PDF report templates and layouts</p>
            <div class="admin-card-stat"><?php echo $stats['pdf_templates']; ?> templates</div>
        </div>

        <!-- System Customization -->
        <div class="admin-card" onclick="window.location.href='system-customization.php'">
            <span class="admin-card-icon">🎨</span>
            <h3 class="admin-card-title">System Customization</h3>
            <p class="admin-card-description">Customize colors, upload logo, and personalize appearance</p>
            <div class="admin-card-stat">Customize</div>
        </div>

        <!-- Permissions -->
        <div class="admin-card" onclick="window.location.href='permissions-management.php'">
            <span class="admin-card-icon">🔐</span>
            <h3 class="admin-card-title">Permissions Management</h3>
            <p class="admin-card-description">Control user access to features and manage roles</p>
            <div class="admin-card-stat">Manage Access</div>
        </div>

        <!-- Version Control -->
        <div class="admin-card" onclick="window.location.href='version-control.php'">
            <span class="admin-card-icon">🔄</span>
            <h3 class="admin-card-title">Version Control</h3>
            <p class="admin-card-description">Track file changes, view version history, and rollback</p>
            <div class="admin-card-stat">Manage Versions</div>
        </div>

        <!-- Error Logs -->
        <div class="admin-card" onclick="window.location.href='error-logs.php'">
            <span class="admin-card-icon">📊</span>
            <h3 class="admin-card-title">Error Logs</h3>
            <p class="admin-card-description">Monitor system errors, security events, and debug logs</p>
            <div class="admin-card-stat">View Logs</div>
        </div>

        <!-- SMTP Settings -->
        <div class="admin-card" onclick="window.location.href='smtp-settings.php'">
            <span class="admin-card-icon">📧</span>
            <h3 class="admin-card-title">SMTP Email Settings</h3>
            <p class="admin-card-description">Configure email server settings for automated reports</p>
            <div class="admin-card-stat">Configure Email</div>
        </div>

        <!-- Reports -->
        <div class="admin-card" onclick="window.location.href='reports.php'">
            <span class="admin-card-icon">📊</span>
            <h3 class="admin-card-title">Reports & Analytics</h3>
            <p class="admin-card-description">Generate reports and view analytics on assessments</p>
            <div class="admin-card-stat">View reports</div>
        </div>
    </div>

    <?php if (!empty($recent_logins)): ?>
    <!-- Recent Activity -->
    <div class="card">
        <div class="card-header">
            <h2 class="card-title">Recent Login Activity</h2>
        </div>
        <div class="card-body">
            <div class="table-container">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Username</th>
                            <th>Last Login</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($recent_logins as $login): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($login['username']); ?></td>
                            <td><?php echo $login['last_login'] ? date('d M Y H:i', strtotime($login['last_login'])) : 'Never'; ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<style>
.admin-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.admin-card {
    background: var(--bg-card);
    border-radius: 12px;
    padding: 1.5rem;
    cursor: pointer;
    transition: var(--transition);
    border: 2px solid var(--border-color);
}

.admin-card:hover {
    transform: translateY(-4px);
    box-shadow: var(--shadow-lg);
    border-color: var(--primary);
}

.admin-card-icon {
    font-size: 3rem;
    display: block;
    margin-bottom: 1rem;
}

.admin-card-title {
    font-size: 1.25rem;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 0.5rem;
}

.admin-card-description {
    color: var(--text-muted);
    font-size: 0.9rem;
    margin-bottom: 1rem;
    line-height: 1.5;
}

.admin-card-stat {
    color: var(--primary);
    font-weight: 600;
    font-size: 0.9rem;
}
</style>

<?php include '../templates/footer.php'; ?>
