<?php
require_once '../includes/config.php';
require_once '../includes/permissions.php';
requireLogin();

// Check permission
requirePermission('view_reports');

$location_id = isset($_GET['location_id']) ? (int)$_GET['location_id'] : null;
$status_filter = isset($_GET['status']) ? $_GET['status'] : 'all';
$priority_filter = isset($_GET['priority']) ? $_GET['priority'] : 'all';

// Fetch all locations for dropdown
$stmt = $pdo->query("SELECT id, location_name, city FROM locations ORDER BY location_name");
$locations = $stmt->fetchAll();

// Build query based on filters
$where_conditions = ["a.status = 'completed'"];
$params = [];

if ($location_id) {
    $where_conditions[] = "a.location_id = ?";
    $params[] = $location_id;
}

// Fetch action items
$sql = "
    SELECT 
        ad.id as detail_id,
        ad.additional_actions,
        ad.action_owner,
        ad.target_date,
        ad.risk_score,
        ad.likelihood,
        ad.severity,
        ri.item_name,
        rc.category_name,
        rc.icon as category_icon,
        a.id as assessment_id,
        a.assessment_number,
        a.assessment_date,
        a.overall_risk_level,
        l.id as location_id,
        l.location_name,
        l.city,
        l.postcode,
        u.full_name as assessor_name,
        CASE 
            WHEN ad.target_date IS NULL THEN 'no_date'
            WHEN ad.target_date < CURDATE() THEN 'overdue'
            WHEN ad.target_date <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) THEN 'due_soon'
            ELSE 'on_track'
        END as action_status,
        DATEDIFF(ad.target_date, CURDATE()) as days_until_due
    FROM assessment_details ad
    JOIN assessments a ON ad.assessment_id = a.id
    JOIN risk_items ri ON ad.risk_item_id = ri.id
    JOIN risk_categories rc ON ri.category_id = rc.id
    JOIN locations l ON a.location_id = l.id
    JOIN users u ON a.assessor_id = u.id
    WHERE " . implode(' AND ', $where_conditions) . "
    AND ad.additional_actions IS NOT NULL 
    AND ad.additional_actions != ''
";

// Add status filter
if ($status_filter !== 'all') {
    if ($status_filter === 'overdue') {
        $sql .= " AND ad.target_date < CURDATE()";
    } elseif ($status_filter === 'due_soon') {
        $sql .= " AND ad.target_date >= CURDATE() AND ad.target_date <= DATE_ADD(CURDATE(), INTERVAL 7 DAY)";
    } elseif ($status_filter === 'on_track') {
        $sql .= " AND ad.target_date > DATE_ADD(CURDATE(), INTERVAL 7 DAY)";
    } elseif ($status_filter === 'no_date') {
        $sql .= " AND ad.target_date IS NULL";
    }
}

// Add priority filter (based on risk score)
if ($priority_filter !== 'all') {
    if ($priority_filter === 'critical') {
        $sql .= " AND ad.risk_score >= 20";
    } elseif ($priority_filter === 'high') {
        $sql .= " AND ad.risk_score >= 15 AND ad.risk_score < 20";
    } elseif ($priority_filter === 'medium') {
        $sql .= " AND ad.risk_score >= 10 AND ad.risk_score < 15";
    } elseif ($priority_filter === 'low') {
        $sql .= " AND ad.risk_score < 10";
    }
}

$sql .= " ORDER BY 
    CASE 
        WHEN ad.target_date < CURDATE() THEN 1
        WHEN ad.target_date <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) THEN 2
        WHEN ad.target_date IS NULL THEN 4
        ELSE 3
    END,
    ad.risk_score DESC,
    ad.target_date ASC
";

$stmt = $pdo->prepare($sql);
$stmt->execute($params);
$action_items = $stmt->fetchAll();

// Calculate statistics
$total_actions = count($action_items);
$overdue = count(array_filter($action_items, fn($item) => $item['action_status'] === 'overdue'));
$due_soon = count(array_filter($action_items, fn($item) => $item['action_status'] === 'due_soon'));
$on_track = count(array_filter($action_items, fn($item) => $item['action_status'] === 'on_track'));
$no_date = count(array_filter($action_items, fn($item) => $item['action_status'] === 'no_date'));

// Group by location if showing all
$by_location = [];
foreach ($action_items as $item) {
    $loc_id = $item['location_id'];
    if (!isset($by_location[$loc_id])) {
        $by_location[$loc_id] = [
            'location_name' => $item['location_name'],
            'city' => $item['city'],
            'items' => []
        ];
    }
    $by_location[$loc_id]['items'][] = $item;
}

include '../templates/header.php';
?>

<div class="container">
    <div class="page-header">
        <div>
            <h1 class="page-title">📋 Action Items Report</h1>
            <p class="page-subtitle">Track and manage pending actions from risk assessments</p>
        </div>
        <div class="btn-group">
            <button onclick="exportToExcel()" class="btn btn-secondary">📊 Export to Excel</button>
            <button onclick="window.print()" class="btn btn-secondary">🖨️ Print Report</button>
            <a href="reports.php" class="btn btn-secondary">← Back to Reports</a>
        </div>
    </div>

    <!-- Filters -->
    <div class="card">
        <div class="card-body" style="padding: 1rem;">
            <form method="GET" class="filters-form">
                <div class="filter-row">
                    <div class="form-group">
                        <label class="form-label">Location</label>
                        <select name="location_id" class="form-select" onchange="this.form.submit()">
                            <option value="">All Locations</option>
                            <?php foreach ($locations as $loc): ?>
                            <option value="<?php echo $loc['id']; ?>" <?php echo $location_id == $loc['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($loc['location_name']); ?> - <?php echo htmlspecialchars($loc['city']); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-select" onchange="this.form.submit()">
                            <option value="all" <?php echo $status_filter === 'all' ? 'selected' : ''; ?>>All Status</option>
                            <option value="overdue" <?php echo $status_filter === 'overdue' ? 'selected' : ''; ?>>⚠️ Overdue</option>
                            <option value="due_soon" <?php echo $status_filter === 'due_soon' ? 'selected' : ''; ?>>🔔 Due Soon (7 days)</option>
                            <option value="on_track" <?php echo $status_filter === 'on_track' ? 'selected' : ''; ?>>✅ On Track</option>
                            <option value="no_date" <?php echo $status_filter === 'no_date' ? 'selected' : ''; ?>>❓ No Due Date</option>
                        </select>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Priority</label>
                        <select name="priority" class="form-select" onchange="this.form.submit()">
                            <option value="all" <?php echo $priority_filter === 'all' ? 'selected' : ''; ?>>All Priorities</option>
                            <option value="critical" <?php echo $priority_filter === 'critical' ? 'selected' : ''; ?>>🔴 Critical (20-25)</option>
                            <option value="high" <?php echo $priority_filter === 'high' ? 'selected' : ''; ?>>🟠 High (15-19)</option>
                            <option value="medium" <?php echo $priority_filter === 'medium' ? 'selected' : ''; ?>>🟡 Medium (10-14)</option>
                            <option value="low" <?php echo $priority_filter === 'low' ? 'selected' : ''; ?>>🟢 Low (1-9)</option>
                        </select>
                    </div>

                    <div class="form-group" style="display: flex; align-items: flex-end;">
                        <a href="action-items-report.php" class="btn btn-secondary">Clear Filters</a>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Statistics -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon" style="color: var(--primary);">📋</div>
            <div>
                <div class="stat-value"><?php echo $total_actions; ?></div>
                <div class="stat-label">Total Actions</div>
            </div>
        </div>

        <div class="stat-card" style="<?php echo $overdue > 0 ? 'border-left: 4px solid var(--danger);' : ''; ?>">
            <div class="stat-icon" style="color: var(--danger);">⚠️</div>
            <div>
                <div class="stat-value"><?php echo $overdue; ?></div>
                <div class="stat-label">Overdue</div>
            </div>
        </div>

        <div class="stat-card" style="<?php echo $due_soon > 0 ? 'border-left: 4px solid var(--warning);' : ''; ?>">
            <div class="stat-icon" style="color: var(--warning);">🔔</div>
            <div>
                <div class="stat-value"><?php echo $due_soon; ?></div>
                <div class="stat-label">Due Soon (7 days)</div>
            </div>
        </div>

        <div class="stat-card">
            <div class="stat-icon" style="color: var(--success);">✅</div>
            <div>
                <div class="stat-value"><?php echo $on_track; ?></div>
                <div class="stat-label">On Track</div>
            </div>
        </div>
    </div>

    <!-- Action Items -->
    <?php if ($location_id): ?>
        <!-- Single Location View -->
        <div class="card">
            <div class="card-header">
                <h2 class="card-title">
                    <?php 
                    $selected_location = array_values(array_filter($locations, fn($l) => $l['id'] == $location_id))[0];
                    echo htmlspecialchars($selected_location['location_name']); 
                    ?>
                </h2>
            </div>
            <div class="card-body" style="padding: 0;">
                <?php if (empty($action_items)): ?>
                    <div style="text-align: center; padding: 3rem; color: var(--text-muted);">
                        <div style="font-size: 3rem; margin-bottom: 1rem;">✅</div>
                        <p>No pending actions for this location</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($action_items as $index => $item): ?>
                        <?php renderActionItem($item, $index); ?>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    <?php else: ?>
        <!-- All Locations View -->
        <?php if (empty($action_items)): ?>
            <div class="card">
                <div class="card-body" style="text-align: center; padding: 3rem;">
                    <div style="font-size: 3rem; margin-bottom: 1rem;">✅</div>
                    <h3 style="color: var(--text-primary);">No Pending Actions</h3>
                    <p style="color: var(--text-muted);">All actions are completed or no actions have been assigned.</p>
                </div>
            </div>
        <?php else: ?>
            <?php foreach ($by_location as $loc_id => $location_data): ?>
            <div class="card">
                <div class="card-header">
                    <h2 class="card-title">
                        📍 <?php echo htmlspecialchars($location_data['location_name']); ?>
                        <small style="color: var(--text-muted); font-weight: normal;">
                            (<?php echo count($location_data['items']); ?> actions)
                        </small>
                    </h2>
                    <a href="?location_id=<?php echo $loc_id; ?>" class="btn btn-secondary">View Details</a>
                </div>
                <div class="card-body" style="padding: 0;">
                    <?php foreach ($location_data['items'] as $index => $item): ?>
                        <?php renderActionItem($item, $index); ?>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endforeach; ?>
        <?php endif; ?>
    <?php endif; ?>
</div>

<style>
.filters-form {
    display: flex;
}

.filter-row {
    display: flex;
    gap: 1rem;
    align-items: flex-end;
    width: 100%;
}

.filter-row .form-group {
    flex: 1;
}

.action-item {
    padding: 1.5rem;
    border-bottom: 1px solid var(--border-color);
    transition: var(--transition);
}

.action-item:last-child {
    border-bottom: none;
}

.action-item:hover {
    background: var(--bg-secondary);
}

.action-item.overdue {
    border-left: 4px solid var(--danger);
    background: rgba(220, 38, 38, 0.05);
}

.action-item.due_soon {
    border-left: 4px solid var(--warning);
    background: rgba(245, 158, 11, 0.05);
}

.action-header {
    display: flex;
    justify-content: space-between;
    align-items: start;
    margin-bottom: 1rem;
}

.action-title {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 1.1rem;
    font-weight: 600;
    color: var(--text-primary);
}

.action-meta {
    display: flex;
    gap: 1.5rem;
    flex-wrap: wrap;
    font-size: 0.85rem;
    color: var(--text-muted);
    margin-bottom: 1rem;
}

.action-content {
    background: var(--bg-secondary);
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1rem;
}

.action-footer {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.priority-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 12px;
    font-size: 0.85rem;
    font-weight: 600;
}

.priority-critical {
    background: rgba(220, 38, 38, 0.1);
    color: var(--danger);
}

.priority-high {
    background: rgba(245, 158, 11, 0.1);
    color: var(--warning);
}

.priority-medium {
    background: rgba(234, 179, 8, 0.1);
    color: #EAB308;
}

.priority-low {
    background: rgba(5, 150, 105, 0.1);
    color: var(--success);
}

@media print {
    .page-header .btn-group,
    .filters-form,
    .action-footer .btn {
        display: none !important;
    }
}
</style>

<script>
function exportToExcel() {
    // Create CSV content
    let csv = 'Location,Risk Category,Risk Item,Action Required,Owner,Target Date,Status,Priority,Assessment #\n';
    
    <?php foreach ($action_items as $item): ?>
    csv += '<?php echo addslashes($item['location_name']); ?>,';
    csv += '<?php echo addslashes($item['category_name']); ?>,';
    csv += '<?php echo addslashes($item['item_name']); ?>,';
    csv += '<?php echo addslashes($item['additional_actions']); ?>,';
    csv += '<?php echo addslashes($item['action_owner'] ?? 'Not assigned'); ?>,';
    csv += '<?php echo $item['target_date'] ? date('d/m/Y', strtotime($item['target_date'])) : 'No date'; ?>,';
    csv += '<?php echo ucfirst(str_replace('_', ' ', $item['action_status'])); ?>,';
    csv += '<?php echo getPriorityLabel($item['risk_score']); ?>,';
    csv += '<?php echo addslashes($item['assessment_number']); ?>\n';
    <?php endforeach; ?>
    
    // Download CSV
    const blob = new Blob([csv], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'action-items-report-<?php echo date('Y-m-d'); ?>.csv';
    a.click();
}
</script>

<?php include '../templates/footer.php'; ?>

<?php
function renderActionItem($item, $index) {
    $status_class = $item['action_status'];
    $status_icons = [
        'overdue' => '⚠️',
        'due_soon' => '🔔',
        'on_track' => '✅',
        'no_date' => '❓'
    ];
    $status_labels = [
        'overdue' => 'Overdue',
        'due_soon' => 'Due Soon',
        'on_track' => 'On Track',
        'no_date' => 'No Due Date'
    ];
    
    $priority_class = getPriorityClass($item['risk_score']);
    $priority_label = getPriorityLabel($item['risk_score']);
    ?>
    <div class="action-item <?php echo $status_class; ?>">
        <div class="action-header">
            <div class="action-title">
                <?php echo $item['category_icon']; ?>
                <?php echo htmlspecialchars($item['item_name']); ?>
            </div>
            <span class="badge badge-<?php echo getRiskBadgeClass($item['overall_risk_level']); ?>">
                <?php echo strtoupper($item['overall_risk_level']); ?>
            </span>
        </div>

        <div class="action-meta">
            <span><strong>Category:</strong> <?php echo htmlspecialchars($item['category_name']); ?></span>
            <span><strong>Assessment:</strong> <?php echo htmlspecialchars($item['assessment_number']); ?></span>
            <span><strong>Date:</strong> <?php echo date('d M Y', strtotime($item['assessment_date'])); ?></span>
            <span><strong>Assessor:</strong> <?php echo htmlspecialchars($item['assessor_name']); ?></span>
        </div>

        <div class="action-content">
            <strong>Action Required:</strong>
            <p style="margin: 0.5rem 0 0 0; line-height: 1.6;">
                <?php echo nl2br(htmlspecialchars($item['additional_actions'])); ?>
            </p>
        </div>

        <div class="action-footer">
            <div style="display: flex; gap: 1rem; align-items: center;">
                <span class="priority-badge <?php echo $priority_class; ?>">
                    <?php echo $priority_label; ?> (<?php echo $item['risk_score']; ?>)
                </span>
                
                <?php if ($item['action_owner']): ?>
                <span style="font-size: 0.9rem;">
                    👤 <strong>Owner:</strong> <?php echo htmlspecialchars($item['action_owner']); ?>
                </span>
                <?php endif; ?>

                <?php if ($item['target_date']): ?>
                <span style="font-size: 0.9rem;" class="<?php echo $status_class === 'overdue' ? 'text-danger' : ($status_class === 'due_soon' ? 'text-warning' : ''); ?>">
                    📅 <strong>Due:</strong> <?php echo date('d M Y', strtotime($item['target_date'])); ?>
                    <?php if ($item['days_until_due'] !== null): ?>
                        (<?php echo $item['days_until_due'] >= 0 ? $item['days_until_due'] . ' days' : abs($item['days_until_due']) . ' days overdue'; ?>)
                    <?php endif; ?>
                </span>
                <?php else: ?>
                <span style="font-size: 0.9rem; color: var(--text-muted);">
                    ❓ No due date set
                </span>
                <?php endif; ?>

                <span class="badge badge-<?php echo getStatusBadgeClass($status_class); ?>">
                    <?php echo $status_icons[$status_class]; ?> <?php echo $status_labels[$status_class]; ?>
                </span>
            </div>

            <a href="view-assessment.php?id=<?php echo $item['assessment_id']; ?>" class="btn btn-secondary">
                View Assessment →
            </a>
        </div>
    </div>
    <?php
}

function getPriorityClass($score) {
    if ($score >= 20) return 'priority-critical';
    if ($score >= 15) return 'priority-high';
    if ($score >= 10) return 'priority-medium';
    return 'priority-low';
}

function getPriorityLabel($score) {
    if ($score >= 20) return 'Critical';
    if ($score >= 15) return 'High';
    if ($score >= 10) return 'Medium';
    return 'Low';
}

function getRiskBadgeClass($level) {
    $classes = [
        'low' => 'success',
        'medium' => 'warning',
        'high' => 'danger',
        'critical' => 'danger'
    ];
    return $classes[$level] ?? 'info';
}

function getStatusBadgeClass($status) {
    $classes = [
        'overdue' => 'danger',
        'due_soon' => 'warning',
        'on_track' => 'success',
        'no_date' => 'secondary'
    ];
    return $classes[$status] ?? 'info';
}
?>
