<?php
/**
 * System Files Version Control Helper Functions
 * Automatic versioning, tracking, and rollback for system files
 */

/**
 * Save a new version of a file
 */
function saveFileVersion($pdo, $file_path, $file_content, $change_type, $change_summary, $user_id, $description = '', $breaking = false) {
    try {
        $pdo->beginTransaction();
        
        // Get or create system file record
        $stmt = $pdo->prepare("SELECT id, current_version FROM system_files WHERE file_path = ?");
        $stmt->execute([$file_path]);
        $file_record = $stmt->fetch();
        
        if (!$file_record) {
            // Register new file
            $file_info = pathinfo($file_path);
            $extension = strtolower($file_info['extension'] ?? 'other');
            $type_map = [
                'php' => 'php',
                'js' => 'js',
                'css' => 'css',
                'sql' => 'sql',
                'html' => 'html'
            ];
            $file_type = $type_map[$extension] ?? 'other';
            
            // Determine category from path
            if (strpos($file_path, 'public/') === 0) $category = 'public';
            elseif (strpos($file_path, 'includes/') === 0) $category = 'includes';
            elseif (strpos($file_path, 'templates/') === 0) $category = 'templates';
            elseif (strpos($file_path, 'api/') !== false) $category = 'api';
            else $category = 'asset';
            
            $stmt = $pdo->prepare("
                INSERT INTO system_files (file_path, file_type, file_category, file_name, current_version)
                VALUES (?, ?, ?, ?, 1)
            ");
            $stmt->execute([$file_path, $file_type, $category, $file_info['basename']]);
            $file_id = $pdo->lastInsertId();
            $version_number = 1;
        } else {
            $file_id = $file_record['id'];
            $version_number = $file_record['current_version'] + 1;
        }
        
        // Calculate file hash
        $file_hash = hash('sha256', $file_content);
        
        // Check if content actually changed
        if ($file_record) {
            $stmt = $pdo->prepare("
                SELECT file_hash FROM file_versions 
                WHERE system_file_id = ? AND version_number = ?
            ");
            $stmt->execute([$file_id, $file_record['current_version']]);
            $current_hash = $stmt->fetchColumn();
            
            if ($current_hash === $file_hash) {
                $pdo->rollBack();
                return ['success' => true, 'message' => 'No changes detected', 'version' => $file_record['current_version']];
            }
        }
        
        // Save new version
        $stmt = $pdo->prepare("
            INSERT INTO file_versions 
            (system_file_id, version_number, file_content, file_size, file_hash, 
             change_type, change_summary, change_description, breaking_changes, changed_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $file_id,
            $version_number,
            $file_content,
            strlen($file_content),
            $file_hash,
            $change_type,
            $change_summary,
            $description,
            $breaking ? 1 : 0,
            $user_id
        ]);
        
        // Update current version
        $stmt = $pdo->prepare("UPDATE system_files SET current_version = ? WHERE id = ?");
        $stmt->execute([$version_number, $file_id]);
        
        $pdo->commit();
        
        return [
            'success' => true,
            'file_id' => $file_id,
            'version' => $version_number,
            'message' => 'Version saved successfully'
        ];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Save file version error: " . $e->getMessage());
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Get file version history
 */
function getFileHistory($pdo, $file_path, $limit = 10) {
    $stmt = $pdo->prepare("
        SELECT fv.*, u.full_name as changed_by_name
        FROM file_versions fv
        JOIN system_files sf ON fv.system_file_id = sf.id
        JOIN users u ON fv.changed_by = u.id
        WHERE sf.file_path = ?
        ORDER BY fv.version_number DESC
        LIMIT ?
    ");
    $stmt->execute([$file_path, $limit]);
    return $stmt->fetchAll();
}

/**
 * Get specific file version content
 */
function getFileVersion($pdo, $file_path, $version_number) {
    $stmt = $pdo->prepare("
        SELECT fv.*
        FROM file_versions fv
        JOIN system_files sf ON fv.system_file_id = sf.id
        WHERE sf.file_path = ? AND fv.version_number = ?
    ");
    $stmt->execute([$file_path, $version_number]);
    return $stmt->fetch();
}

/**
 * Rollback file to previous version
 */
function rollbackFileVersion($pdo, $file_path, $target_version, $user_id, $reason = '') {
    try {
        $pdo->beginTransaction();
        
        // Get file record
        $stmt = $pdo->prepare("SELECT id, current_version FROM system_files WHERE file_path = ?");
        $stmt->execute([$file_path]);
        $file_record = $stmt->fetch();
        
        if (!$file_record) {
            throw new Exception('File not found');
        }
        
        // Get target version content
        $target = getFileVersion($pdo, $file_path, $target_version);
        if (!$target) {
            throw new Exception('Target version not found');
        }
        
        // Create new version with rolled-back content
        $new_version = $file_record['current_version'] + 1;
        
        $stmt = $pdo->prepare("
            INSERT INTO file_versions 
            (system_file_id, version_number, file_content, file_size, file_hash,
             change_type, change_summary, change_description, changed_by)
            VALUES (?, ?, ?, ?, ?, 'update', ?, ?, ?)
        ");
        $stmt->execute([
            $file_record['id'],
            $new_version,
            $target['file_content'],
            $target['file_size'],
            $target['file_hash'],
            "Rolled back to version $target_version",
            $reason,
            $user_id
        ]);
        
        // Update current version
        $stmt = $pdo->prepare("UPDATE system_files SET current_version = ? WHERE id = ?");
        $stmt->execute([$new_version, $file_record['id']]);
        
        // Log rollback
        $stmt = $pdo->prepare("
            INSERT INTO version_rollback_log 
            (system_file_id, from_version, to_version, rollback_reason, performed_by, success)
            VALUES (?, ?, ?, ?, ?, 1)
        ");
        $stmt->execute([
            $file_record['id'],
            $file_record['current_version'],
            $target_version,
            $reason,
            $user_id
        ]);
        
        // Write file to disk
        $full_path = __DIR__ . '/../' . $file_path;
        file_put_contents($full_path, $target['file_content']);
        
        $pdo->commit();
        
        return [
            'success' => true,
            'new_version' => $new_version,
            'message' => "Rolled back to version $target_version"
        ];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        
        // Log failed rollback
        if (isset($file_record)) {
            $stmt = $pdo->prepare("
                INSERT INTO version_rollback_log 
                (system_file_id, from_version, to_version, rollback_reason, performed_by, success, error_message)
                VALUES (?, ?, ?, ?, ?, 0, ?)
            ");
            $stmt->execute([
                $file_record['id'],
                $file_record['current_version'],
                $target_version,
                $reason,
                $user_id,
                $e->getMessage()
            ]);
        }
        
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Compare two file versions
 */
function compareFileVersions($pdo, $file_path, $version1, $version2) {
    $v1 = getFileVersion($pdo, $file_path, $version1);
    $v2 = getFileVersion($pdo, $file_path, $version2);
    
    if (!$v1 || !$v2) {
        return false;
    }
    
    return [
        'version1' => $v1,
        'version2' => $v2,
        'diff' => generateDiff($v1['file_content'], $v2['file_content'])
    ];
}

/**
 * Simple diff generator
 */
function generateDiff($content1, $content2) {
    $lines1 = explode("\n", $content1);
    $lines2 = explode("\n", $content2);
    
    $diff = [];
    $max = max(count($lines1), count($lines2));
    
    for ($i = 0; $i < $max; $i++) {
        $line1 = $lines1[$i] ?? null;
        $line2 = $lines2[$i] ?? null;
        
        if ($line1 === $line2) {
            $diff[] = ['type' => 'same', 'content' => $line1];
        } elseif ($line1 === null) {
            $diff[] = ['type' => 'added', 'content' => $line2];
        } elseif ($line2 === null) {
            $diff[] = ['type' => 'removed', 'content' => $line1];
        } else {
            $diff[] = ['type' => 'changed', 'old' => $line1, 'new' => $line2];
        }
    }
    
    return $diff;
}

/**
 * Create a system release
 */
function createRelease($pdo, $version, $name, $notes, $type, $user_id, $file_versions = []) {
    try {
        $pdo->beginTransaction();
        
        $stmt = $pdo->prepare("
            INSERT INTO system_releases (release_version, release_name, release_notes, release_type, released_by)
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$version, $name, $notes, $type, $user_id]);
        $release_id = $pdo->lastInsertId();
        
        // Link file versions to release
        if (!empty($file_versions)) {
            $stmt = $pdo->prepare("
                INSERT INTO release_files (release_id, file_version_id, is_breaking_change, notes)
                VALUES (?, ?, ?, ?)
            ");
            
            foreach ($file_versions as $fv) {
                $stmt->execute([
                    $release_id,
                    $fv['version_id'],
                    $fv['breaking'] ?? false,
                    $fv['notes'] ?? null
                ]);
            }
        }
        
        $pdo->commit();
        return ['success' => true, 'release_id' => $release_id];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        return ['success' => false, 'error' => $e->getMessage()];
    }
}

/**
 * Auto-version file on save (hook into file writes)
 */
function autoVersionFile($file_path, $content, $change_summary = 'Automatic save') {
    global $pdo;
    
    if (!isset($_SESSION['user_id'])) {
        return false;
    }
    
    return saveFileVersion(
        $pdo,
        $file_path,
        $content,
        'update',
        $change_summary,
        $_SESSION['user_id']
    );
}

/**
 * Get all tracked files
 */
function getTrackedFiles($pdo) {
    $stmt = $pdo->query("
        SELECT sf.*, fv.version_number as current_version, fv.change_date as last_modified
        FROM system_files sf
        LEFT JOIN file_versions fv ON sf.id = fv.system_file_id AND sf.current_version = fv.version_number
        WHERE sf.is_tracked = TRUE
        ORDER BY sf.file_category, sf.file_name
    ");
    return $stmt->fetchAll();
}

/**
 * Get recent changes
 */
function getRecentChanges($pdo, $limit = 20) {
    $stmt = $pdo->prepare("
        SELECT sf.file_path, sf.file_name, fv.version_number, fv.change_type, 
               fv.change_summary, u.full_name as changed_by, fv.change_date
        FROM file_versions fv
        JOIN system_files sf ON fv.system_file_id = sf.id
        JOIN users u ON fv.changed_by = u.id
        ORDER BY fv.change_date DESC
        LIMIT ?
    ");
    $stmt->execute([$limit]);
    return $stmt->fetchAll();
}
