<?php
/**
 * Permission System Helper Functions
 * Role-Based Access Control (RBAC) implementation
 */

/**
 * Check if user has a specific permission
 */
function hasPermission($pdo, $user_id, $permission_key, $log = true) {
    // Check if permission system is enabled
    if (!isPermissionSystemEnabled($pdo)) {
        // Fallback to role-based check
        return checkRolePermission($_SESSION['role'] ?? 'viewer', $permission_key);
    }
    
    try {
        // Get user info
        $stmt = $pdo->prepare("SELECT role FROM users WHERE id = ? AND is_active = 1");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        
        if (!$user) return false;
        
        // Check user-specific permission override first
        $stmt = $pdo->prepare("
            SELECT up.is_granted
            FROM user_permissions up
            JOIN permissions p ON up.permission_id = p.id
            WHERE up.user_id = ? AND p.permission_key = ?
        ");
        $stmt->execute([$user_id, $permission_key]);
        $user_perm = $stmt->fetch();
        
        if ($user_perm !== false) {
            $result = (bool)$user_perm['is_granted'];
            if ($log) logPermissionCheck($pdo, $user_id, 'check', $permission_key, $user['role'], $result);
            return $result;
        }
        
        // Check role-based permission
        $stmt = $pdo->prepare("
            SELECT rp.id
            FROM role_permissions rp
            JOIN permissions p ON rp.permission_id = p.id
            WHERE rp.role = ? AND p.permission_key = ? AND p.is_active = 1
        ");
        $stmt->execute([$user['role'], $permission_key]);
        $result = $stmt->fetch() !== false;
        
        if ($log) logPermissionCheck($pdo, $user_id, 'check', $permission_key, $user['role'], $result);
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Permission check error: " . $e->getMessage());
        return checkRolePermission($user['role'] ?? 'viewer', $permission_key);
    }
}

/**
 * Fallback role-based permission check
 */
function checkRolePermission($role, $permission_key) {
    $rolePermissions = [
        'admin' => ['*'], // Admin has all permissions
        'assessor' => [
            'view_dashboard', 'view_analytics', 'view_assessments', 'create_assessments',
            'edit_assessments', 'export_assessments', 'view_locations', 'create_locations',
            'edit_locations', 'manage_location_documents', 'upload_photos', 'upload_risk_item_photos',
            'view_reports', 'generate_reports', 'export_reports', 'view_versions',
            'create_revisions', 'compare_versions'
        ],
        'viewer' => [
            'view_dashboard', 'view_assessments', 'view_locations', 'view_reports',
            'view_versions', 'compare_versions'
        ]
    ];
    
    if ($role === 'admin') return true;
    return in_array($permission_key, $rolePermissions[$role] ?? []);
}

/**
 * Require permission - redirect if not granted
 */
function requirePermission($permission_key, $redirect_url = 'dashboard.php') {
    global $pdo;
    
    if (!isset($_SESSION['user_id'])) {
        header('Location: login.php');
        exit;
    }
    
    if (!hasPermission($pdo, $_SESSION['user_id'], $permission_key)) {
        $_SESSION['error'] = 'You do not have permission to access this page.';
        header('Location: ' . $redirect_url);
        exit;
    }
}

/**
 * Check if user can perform action on resource
 */
function canPerformAction($pdo, $user_id, $action, $resource, $resource_owner_id = null) {
    $permission_key = $action . '_' . $resource;
    
    if (!hasPermission($pdo, $user_id, $permission_key, false)) {
        return false;
    }
    
    // Additional check: can only edit/delete own resources (unless admin)
    if (($action === 'edit' || $action === 'delete') && $resource_owner_id) {
        $stmt = $pdo->prepare("SELECT role FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        
        if ($user['role'] !== 'admin' && $user_id != $resource_owner_id) {
            return false;
        }
    }
    
    return true;
}

/**
 * Get all permissions for a user
 */
function getUserPermissions($pdo, $user_id) {
    if (!isPermissionSystemEnabled($pdo)) {
        return [];
    }
    
    $stmt = $pdo->prepare("
        SELECT DISTINCT permission_key
        FROM user_effective_permissions
        WHERE user_id = ? AND has_permission = 1
    ");
    $stmt->execute([$user_id]);
    
    return array_column($stmt->fetchAll(), 'permission_key');
}

/**
 * Grant permission to user
 */
function grantUserPermission($pdo, $user_id, $permission_key, $granted_by, $notes = '') {
    try {
        $stmt = $pdo->prepare("SELECT id FROM permissions WHERE permission_key = ?");
        $stmt->execute([$permission_key]);
        $permission = $stmt->fetch();
        
        if (!$permission) return false;
        
        $stmt = $pdo->prepare("
            INSERT INTO user_permissions (user_id, permission_id, is_granted, granted_by, notes)
            VALUES (?, ?, 1, ?, ?)
            ON DUPLICATE KEY UPDATE is_granted = 1, granted_by = ?, granted_at = CURRENT_TIMESTAMP, notes = ?
        ");
        $stmt->execute([$user_id, $permission['id'], $granted_by, $notes, $granted_by, $notes]);
        
        logPermissionCheck($pdo, $user_id, 'grant', $permission_key, null, true);
        return true;
    } catch (Exception $e) {
        error_log("Grant permission error: " . $e->getMessage());
        return false;
    }
}

/**
 * Revoke permission from user
 */
function revokeUserPermission($pdo, $user_id, $permission_key, $granted_by, $notes = '') {
    try {
        $stmt = $pdo->prepare("SELECT id FROM permissions WHERE permission_key = ?");
        $stmt->execute([$permission_key]);
        $permission = $stmt->fetch();
        
        if (!$permission) return false;
        
        $stmt = $pdo->prepare("
            INSERT INTO user_permissions (user_id, permission_id, is_granted, granted_by, notes)
            VALUES (?, ?, 0, ?, ?)
            ON DUPLICATE KEY UPDATE is_granted = 0, granted_by = ?, granted_at = CURRENT_TIMESTAMP, notes = ?
        ");
        $stmt->execute([$user_id, $permission['id'], $granted_by, $notes, $granted_by, $notes]);
        
        logPermissionCheck($pdo, $user_id, 'revoke', $permission_key, null, true);
        return true;
    } catch (Exception $e) {
        error_log("Revoke permission error: " . $e->getMessage());
        return false;
    }
}

/**
 * Log permission check/action
 */
function logPermissionCheck($pdo, $user_id, $action, $permission_key, $role, $result) {
    if (!isPermissionSystemEnabled($pdo)) return;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO permission_audit_log 
            (user_id, action, permission_key, role, result, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $user_id,
            $action,
            $permission_key,
            $role,
            $result ? 'allowed' : 'denied',
            $_SERVER['REMOTE_ADDR'] ?? null,
            $_SERVER['HTTP_USER_AGENT'] ?? null
        ]);
    } catch (Exception $e) {
        error_log("Permission log error: " . $e->getMessage());
    }
}

/**
 * Get permissions grouped by group
 */
function getPermissionsGrouped($pdo) {
    $stmt = $pdo->query("
        SELECT * FROM permissions 
        WHERE is_active = 1 
        ORDER BY permission_group, permission_name
    ");
    
    $grouped = [];
    foreach ($stmt->fetchAll() as $permission) {
        $group = $permission['permission_group'];
        if (!isset($grouped[$group])) {
            $grouped[$group] = [];
        }
        $grouped[$group][] = $permission;
    }
    
    return $grouped;
}

/**
 * Get role permissions
 */
function getRolePermissions($pdo, $role) {
    $stmt = $pdo->prepare("
        SELECT p.permission_key
        FROM role_permissions rp
        JOIN permissions p ON rp.permission_id = p.id
        WHERE rp.role = ? AND p.is_active = 1
    ");
    $stmt->execute([$role]);
    
    return array_column($stmt->fetchAll(), 'permission_key');
}

/**
 * Update role permissions
 */
function updateRolePermissions($pdo, $role, $permission_keys) {
    try {
        $pdo->beginTransaction();
        
        $stmt = $pdo->prepare("DELETE FROM role_permissions WHERE role = ?");
        $stmt->execute([$role]);
        
        $stmt = $pdo->prepare("
            INSERT INTO role_permissions (role, permission_id)
            SELECT ?, id FROM permissions WHERE permission_key = ?
        ");
        
        foreach ($permission_keys as $permission_key) {
            $stmt->execute([$role, $permission_key]);
        }
        
        $pdo->commit();
        return true;
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Update role permissions error: " . $e->getMessage());
        return false;
    }
}

/**
 * Check if permission system is enabled
 */
function isPermissionSystemEnabled($pdo) {
    try {
        $stmt = $pdo->query("SHOW TABLES LIKE 'permissions'");
        return $stmt->rowCount() > 0;
    } catch (Exception $e) {
        return false;
    }
}
