<?php
/**
 * Permissions Helper Functions
 * Role-Based Access Control (RBAC) utilities
 */

/**
 * Check if user has a specific permission
 * @param PDO $pdo Database connection
 * @param string $permission_key Permission to check
 * @param int|null $user_id User ID (defaults to current session user)
 * @return bool True if user has permission
 */
function hasPermission($pdo, $permission_key, $user_id = null) {
    if ($user_id === null) {
        $user_id = $_SESSION['user_id'] ?? null;
    }
    
    if (!$user_id) {
        return false;
    }
    
    // Get user role
    $stmt = $pdo->prepare("SELECT role FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    
    if (!$user) {
        return false;
    }
    
    $role = $user['role'];
    
    // Admin has all permissions
    if ($role === 'admin') {
        return true;
    }
    
    // Check user-specific permission override (takes precedence)
    $stmt = $pdo->prepare("
        SELECT up.is_granted 
        FROM user_permissions up
        JOIN permissions p ON up.permission_id = p.id
        WHERE up.user_id = ? 
        AND p.permission_key = ?
        AND (up.expires_at IS NULL OR up.expires_at > NOW())
    ");
    $stmt->execute([$user_id, $permission_key]);
    $user_perm = $stmt->fetch();
    
    if ($user_perm !== false) {
        return (bool)$user_perm['is_granted'];
    }
    
    // Check role-based permission
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as count
        FROM role_permissions rp
        JOIN permissions p ON rp.permission_id = p.id
        WHERE rp.role = ? 
        AND p.permission_key = ?
        AND p.is_active = 1
    ");
    $stmt->execute([$role, $permission_key]);
    $result = $stmt->fetch();
    
    return $result['count'] > 0;
}

/**
 * Check if user has any of the specified permissions
 * @param PDO $pdo Database connection
 * @param array $permission_keys Array of permission keys
 * @param int|null $user_id User ID (defaults to current session user)
 * @return bool True if user has at least one permission
 */
function hasAnyPermission($pdo, $permission_keys, $user_id = null) {
    foreach ($permission_keys as $key) {
        if (hasPermission($pdo, $key, $user_id)) {
            return true;
        }
    }
    return false;
}

/**
 * Check if user has all specified permissions
 * @param PDO $pdo Database connection
 * @param array $permission_keys Array of permission keys
 * @param int|null $user_id User ID (defaults to current session user)
 * @return bool True if user has all permissions
 */
function hasAllPermissions($pdo, $permission_keys, $user_id = null) {
    foreach ($permission_keys as $key) {
        if (!hasPermission($pdo, $key, $user_id)) {
            return false;
        }
    }
    return true;
}

/**
 * Require permission or redirect/exit
 * @param PDO $pdo Database connection
 * @param string $permission_key Permission required
 * @param string $redirect_url URL to redirect if permission denied (default: dashboard)
 */
function requirePermission($pdo, $permission_key, $redirect_url = 'dashboard.php') {
    if (!hasPermission($pdo, $permission_key)) {
        $_SESSION['error_message'] = 'You do not have permission to access this feature.';
        header('Location: ' . $redirect_url);
        exit;
    }
}

/**
 * Check if a feature is enabled system-wide
 * @param PDO $pdo Database connection
 * @param string $feature_key Feature to check
 * @return bool True if feature is enabled
 */
function isFeatureEnabled($pdo, $feature_key) {
    try {
        $stmt = $pdo->prepare("SELECT is_enabled FROM feature_toggles WHERE feature_key = ?");
        $stmt->execute([$feature_key]);
        $result = $stmt->fetch();
        
        return $result ? (bool)$result['is_enabled'] : true; // Default to enabled if not found
    } catch (Exception $e) {
        // Table might not exist yet
        return true;
    }
}

/**
 * Require feature to be enabled or redirect
 * @param PDO $pdo Database connection
 * @param string $feature_key Feature required
 * @param string $redirect_url URL to redirect if feature disabled
 */
function requireFeature($pdo, $feature_key, $redirect_url = 'dashboard.php') {
    if (!isFeatureEnabled($pdo, $feature_key)) {
        $_SESSION['error_message'] = 'This feature is currently disabled.';
        header('Location: ' . $redirect_url);
        exit;
    }
}

/**
 * Get all permissions for a user
 * @param PDO $pdo Database connection
 * @param int|null $user_id User ID (defaults to current session user)
 * @return array Array of permission keys
 */
function getUserPermissions($pdo, $user_id = null) {
    if ($user_id === null) {
        $user_id = $_SESSION['user_id'] ?? null;
    }
    
    if (!$user_id) {
        return [];
    }
    
    // Get user role
    $stmt = $pdo->prepare("SELECT role FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    
    if (!$user) {
        return [];
    }
    
    $role = $user['role'];
    
    // Admin has all permissions
    if ($role === 'admin') {
        $stmt = $pdo->query("SELECT permission_key FROM permissions WHERE is_active = 1");
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }
    
    // Get role permissions
    $stmt = $pdo->prepare("
        SELECT p.permission_key 
        FROM role_permissions rp
        JOIN permissions p ON rp.permission_id = p.id
        WHERE rp.role = ? AND p.is_active = 1
    ");
    $stmt->execute([$role]);
    $permissions = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Get user-specific permission overrides
    $stmt = $pdo->prepare("
        SELECT p.permission_key, up.is_granted
        FROM user_permissions up
        JOIN permissions p ON up.permission_id = p.id
        WHERE up.user_id = ?
        AND (up.expires_at IS NULL OR up.expires_at > NOW())
    ");
    $stmt->execute([$user_id]);
    $overrides = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Apply overrides
    foreach ($overrides as $override) {
        if ($override['is_granted']) {
            if (!in_array($override['permission_key'], $permissions)) {
                $permissions[] = $override['permission_key'];
            }
        } else {
            $permissions = array_diff($permissions, [$override['permission_key']]);
        }
    }
    
    return $permissions;
}

/**
 * Grant permission to a user
 * @param PDO $pdo Database connection
 * @param int $user_id User to grant permission to
 * @param string $permission_key Permission to grant
 * @param int|null $granted_by User granting the permission
 * @param string|null $expires_at Expiry date (optional)
 * @return bool Success
 */
function grantPermission($pdo, $user_id, $permission_key, $granted_by = null, $expires_at = null) {
    if ($granted_by === null) {
        $granted_by = $_SESSION['user_id'] ?? null;
    }
    
    // Get permission ID
    $stmt = $pdo->prepare("SELECT id FROM permissions WHERE permission_key = ?");
    $stmt->execute([$permission_key]);
    $permission = $stmt->fetch();
    
    if (!$permission) {
        return false;
    }
    
    // Insert or update user permission
    $stmt = $pdo->prepare("
        INSERT INTO user_permissions (user_id, permission_id, is_granted, granted_by, expires_at)
        VALUES (?, ?, 1, ?, ?)
        ON DUPLICATE KEY UPDATE is_granted = 1, granted_by = ?, expires_at = ?, updated_at = CURRENT_TIMESTAMP
    ");
    $stmt->execute([$user_id, $permission['id'], $granted_by, $expires_at, $granted_by, $expires_at]);
    
    return true;
}

/**
 * Revoke permission from a user
 * @param PDO $pdo Database connection
 * @param int $user_id User to revoke permission from
 * @param string $permission_key Permission to revoke
 * @param int|null $revoked_by User revoking the permission
 * @return bool Success
 */
function revokePermission($pdo, $user_id, $permission_key, $revoked_by = null) {
    if ($revoked_by === null) {
        $revoked_by = $_SESSION['user_id'] ?? null;
    }
    
    // Get permission ID
    $stmt = $pdo->prepare("SELECT id FROM permissions WHERE permission_key = ?");
    $stmt->execute([$permission_key]);
    $permission = $stmt->fetch();
    
    if (!$permission) {
        return false;
    }
    
    // Insert or update user permission (denied)
    $stmt = $pdo->prepare("
        INSERT INTO user_permissions (user_id, permission_id, is_granted, granted_by)
        VALUES (?, ?, 0, ?)
        ON DUPLICATE KEY UPDATE is_granted = 0, granted_by = ?, updated_at = CURRENT_TIMESTAMP
    ");
    $stmt->execute([$user_id, $permission['id'], $revoked_by, $revoked_by]);
    
    return true;
}

/**
 * Get all permissions grouped by category
 * @param PDO $pdo Database connection
 * @return array Permissions grouped by category
 */
function getAllPermissions($pdo) {
    $stmt = $pdo->query("
        SELECT * FROM permissions 
        WHERE is_active = 1 
        ORDER BY permission_category, permission_name
    ");
    $permissions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $grouped = [];
    foreach ($permissions as $perm) {
        $category = $perm['permission_category'];
        if (!isset($grouped[$category])) {
            $grouped[$category] = [];
        }
        $grouped[$category][] = $perm;
    }
    
    return $grouped;
}

/**
 * Get permissions for a specific role
 * @param PDO $pdo Database connection
 * @param string $role Role name
 * @return array Array of permission keys
 */
function getRolePermissions($pdo, $role) {
    $stmt = $pdo->prepare("
        SELECT p.permission_key 
        FROM role_permissions rp
        JOIN permissions p ON rp.permission_id = p.id
        WHERE rp.role = ? AND p.is_active = 1
    ");
    $stmt->execute([$role]);
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

/**
 * Check if current user can manage permissions
 * @param PDO $pdo Database connection
 * @return bool
 */
function canManagePermissions($pdo) {
    return hasPermission($pdo, 'user.manage_permissions');
}

/**
 * Log permission change
 * @param PDO $pdo Database connection
 * @param string $action Action performed (grant/revoke)
 * @param int $target_user_id User affected
 * @param string $permission_key Permission changed
 * @param int|null $changed_by User making the change
 */
function logPermissionChange($pdo, $action, $target_user_id, $permission_key, $changed_by = null) {
    if ($changed_by === null) {
        $changed_by = $_SESSION['user_id'] ?? null;
    }
    
    // This could be extended to write to a dedicated audit log table
    error_log("Permission Change: $action - User $target_user_id - Permission $permission_key - By User $changed_by");
}
