<?php
/**
 * Error Reporting and Logging System
 * Centralized error handling for the Risk Assessment System
 */

// Error log file location
define('ERROR_LOG_FILE', __DIR__ . '/../logs/system-errors.log');
define('DEBUG_LOG_FILE', __DIR__ . '/../logs/debug.log');
define('SECURITY_LOG_FILE', __DIR__ . '/../logs/security.log');

// Ensure logs directory exists
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

/**
 * Initialize error reporting based on environment
 */
function initErrorReporting() {
    // Detect environment
    $isProduction = (isset($_SERVER['SERVER_NAME']) && 
                     strpos($_SERVER['SERVER_NAME'], 'localhost') === false &&
                     strpos($_SERVER['SERVER_NAME'], '127.0.0.1') === false);
    
    if ($isProduction) {
        // Production: Log errors, don't display
        ini_set('display_errors', 0);
        ini_set('display_startup_errors', 0);
        error_reporting(E_ALL);
    } else {
        // Development: Display and log errors
        ini_set('display_errors', 1);
        ini_set('display_startup_errors', 1);
        error_reporting(E_ALL);
    }
    
    // Set error log file
    ini_set('log_errors', 1);
    ini_set('error_log', ERROR_LOG_FILE);
    
    // Set custom error and exception handlers
    set_error_handler('customErrorHandler');
    set_exception_handler('customExceptionHandler');
    register_shutdown_function('fatalErrorHandler');
}

/**
 * Custom error handler
 */
function customErrorHandler($errno, $errstr, $errfile, $errline) {
    // Don't handle suppressed errors (@)
    if (!(error_reporting() & $errno)) {
        return false;
    }
    
    $errorTypes = [
        E_ERROR => 'ERROR',
        E_WARNING => 'WARNING',
        E_PARSE => 'PARSE ERROR',
        E_NOTICE => 'NOTICE',
        E_CORE_ERROR => 'CORE ERROR',
        E_CORE_WARNING => 'CORE WARNING',
        E_COMPILE_ERROR => 'COMPILE ERROR',
        E_COMPILE_WARNING => 'COMPILE WARNING',
        E_USER_ERROR => 'USER ERROR',
        E_USER_WARNING => 'USER WARNING',
        E_USER_NOTICE => 'USER NOTICE',
        E_STRICT => 'STRICT NOTICE',
        E_RECOVERABLE_ERROR => 'RECOVERABLE ERROR',
        E_DEPRECATED => 'DEPRECATED',
        E_USER_DEPRECATED => 'USER DEPRECATED'
    ];
    
    $errorType = $errorTypes[$errno] ?? 'UNKNOWN ERROR';
    
    // Format error message
    $message = sprintf(
        "[%s] %s: %s in %s on line %d",
        date('Y-m-d H:i:s'),
        $errorType,
        $errstr,
        basename($errfile),
        $errline
    );
    
    // Add context
    $message .= "\nRequest: " . ($_SERVER['REQUEST_URI'] ?? 'CLI');
    $message .= "\nUser: " . ($_SESSION['user_id'] ?? 'Not logged in');
    $message .= "\n" . str_repeat('-', 80) . "\n";
    
    // Log to file
    error_log($message, 3, ERROR_LOG_FILE);
    
    // For critical errors, also log to security log
    if (in_array($errno, [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR])) {
        error_log($message, 3, SECURITY_LOG_FILE);
    }
    
    // Don't execute PHP internal error handler
    return true;
}

/**
 * Custom exception handler
 */
function customExceptionHandler($exception) {
    $message = sprintf(
        "[%s] EXCEPTION: %s in %s on line %d\nStack trace:\n%s\n%s\n",
        date('Y-m-d H:i:s'),
        $exception->getMessage(),
        basename($exception->getFile()),
        $exception->getLine(),
        $exception->getTraceAsString(),
        str_repeat('-', 80)
    );
    
    // Add context
    $message .= "Request: " . ($_SERVER['REQUEST_URI'] ?? 'CLI') . "\n";
    $message .= "User: " . ($_SESSION['user_id'] ?? 'Not logged in') . "\n";
    
    error_log($message, 3, ERROR_LOG_FILE);
    
    // Display user-friendly error page
    displayErrorPage('An unexpected error occurred', 
                     'The system encountered an error. Please try again or contact support.');
}

/**
 * Fatal error handler
 */
function fatalErrorHandler() {
    $error = error_get_last();
    
    if ($error !== null && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        $message = sprintf(
            "[%s] FATAL: %s in %s on line %d\n%s\n",
            date('Y-m-d H:i:s'),
            $error['message'],
            basename($error['file']),
            $error['line'],
            str_repeat('-', 80)
        );
        
        error_log($message, 3, ERROR_LOG_FILE);
        
        // Display user-friendly error page
        displayErrorPage('System Error', 
                         'A critical error occurred. Please contact your system administrator.');
    }
}

/**
 * Log database errors
 */
function logDatabaseError($query, $error, $params = []) {
    $message = sprintf(
        "[%s] DATABASE ERROR\nQuery: %s\nError: %s\nParams: %s\nUser: %s\nRequest: %s\n%s\n",
        date('Y-m-d H:i:s'),
        $query,
        $error,
        json_encode($params),
        $_SESSION['user_id'] ?? 'Not logged in',
        $_SERVER['REQUEST_URI'] ?? 'CLI',
        str_repeat('-', 80)
    );
    
    error_log($message, 3, ERROR_LOG_FILE);
}

/**
 * Log security events
 */
function logSecurityEvent($event, $details = '') {
    $message = sprintf(
        "[%s] SECURITY: %s\nDetails: %s\nIP: %s\nUser: %s\nRequest: %s\n%s\n",
        date('Y-m-d H:i:s'),
        $event,
        $details,
        $_SERVER['REMOTE_ADDR'] ?? 'Unknown',
        $_SESSION['user_id'] ?? 'Not logged in',
        $_SERVER['REQUEST_URI'] ?? 'CLI',
        str_repeat('-', 80)
    );
    
    error_log($message, 3, SECURITY_LOG_FILE);
}

/**
 * Log debug information
 */
function logDebug($message, $context = []) {
    $logMessage = sprintf(
        "[%s] DEBUG: %s\nContext: %s\n%s\n",
        date('Y-m-d H:i:s'),
        $message,
        json_encode($context),
        str_repeat('-', 80)
    );
    
    error_log($logMessage, 3, DEBUG_LOG_FILE);
}

/**
 * Display user-friendly error page
 */
function displayErrorPage($title, $message) {
    // Only display if not already sent headers
    if (!headers_sent()) {
        http_response_code(500);
        ?>
        <!DOCTYPE html>
        <html lang="en">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title><?php echo htmlspecialchars($title); ?></title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body {
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                    min-height: 100vh;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    padding: 20px;
                }
                .error-container {
                    background: white;
                    padding: 40px;
                    border-radius: 12px;
                    box-shadow: 0 20px 60px rgba(0,0,0,0.3);
                    max-width: 500px;
                    width: 100%;
                    text-align: center;
                }
                .error-icon {
                    font-size: 64px;
                    margin-bottom: 20px;
                }
                h1 {
                    color: #333;
                    margin-bottom: 10px;
                    font-size: 24px;
                }
                p {
                    color: #666;
                    margin-bottom: 30px;
                    line-height: 1.6;
                }
                .btn {
                    display: inline-block;
                    padding: 12px 24px;
                    background: #667eea;
                    color: white;
                    text-decoration: none;
                    border-radius: 6px;
                    font-weight: 600;
                    transition: background 0.3s;
                }
                .btn:hover {
                    background: #5568d3;
                }
                .error-id {
                    margin-top: 20px;
                    padding-top: 20px;
                    border-top: 1px solid #eee;
                    color: #999;
                    font-size: 12px;
                }
            </style>
        </head>
        <body>
            <div class="error-container">
                <div class="error-icon">⚠️</div>
                <h1><?php echo htmlspecialchars($title); ?></h1>
                <p><?php echo htmlspecialchars($message); ?></p>
                <a href="dashboard.php" class="btn">Return to Dashboard</a>
                <div class="error-id">
                    Error ID: <?php echo date('YmdHis'); ?>
                </div>
            </div>
        </body>
        </html>
        <?php
        exit;
    }
}

/**
 * Safe database query wrapper
 */
function safeQuery($pdo, $query, $params = []) {
    try {
        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        return $stmt;
    } catch (PDOException $e) {
        logDatabaseError($query, $e->getMessage(), $params);
        throw $e;
    }
}

/**
 * Get error statistics
 */
function getErrorStats() {
    $stats = [
        'total_errors' => 0,
        'recent_errors' => 0,
        'security_events' => 0,
        'last_error' => null
    ];
    
    if (file_exists(ERROR_LOG_FILE)) {
        $lines = file(ERROR_LOG_FILE);
        $stats['total_errors'] = count($lines);
        
        // Count recent errors (last 24 hours)
        $yesterday = strtotime('-24 hours');
        foreach ($lines as $line) {
            if (preg_match('/\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\]/', $line, $matches)) {
                $timestamp = strtotime($matches[1]);
                if ($timestamp > $yesterday) {
                    $stats['recent_errors']++;
                }
            }
        }
        
        // Get last error
        $lastError = end($lines);
        if (preg_match('/\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\] (.+)/', $lastError, $matches)) {
            $stats['last_error'] = [
                'time' => $matches[1],
                'message' => $matches[2]
            ];
        }
    }
    
    if (file_exists(SECURITY_LOG_FILE)) {
        $stats['security_events'] = count(file(SECURITY_LOG_FILE));
    }
    
    return $stats;
}

/**
 * Clear old logs
 */
function clearOldLogs($days = 30) {
    $files = [ERROR_LOG_FILE, DEBUG_LOG_FILE, SECURITY_LOG_FILE];
    $cutoff = strtotime("-{$days} days");
    
    foreach ($files as $file) {
        if (!file_exists($file)) continue;
        
        $lines = file($file);
        $newLines = [];
        
        foreach ($lines as $line) {
            if (preg_match('/\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})\]/', $line, $matches)) {
                $timestamp = strtotime($matches[1]);
                if ($timestamp > $cutoff) {
                    $newLines[] = $line;
                }
            }
        }
        
        file_put_contents($file, implode('', $newLines));
    }
}

// Initialize error reporting
initErrorReporting();
