<?php
/**
 * Email Helper Library
 * Comprehensive email sending with SMTP support
 * Uses PHPMailer for reliable email delivery
 */

// Email log file
define('EMAIL_LOG_FILE', __DIR__ . '/../logs/email.log');

/**
 * Send email using SMTP
 * 
 * @param string|array $to Recipient email(s)
 * @param string $subject Email subject
 * @param string $body Email body (HTML supported)
 * @param array $options Additional options (cc, bcc, attachments, etc.)
 * @return bool Success status
 */
function sendEmail($to, $subject, $body, $options = []) {
    // Check if SMTP is enabled
    if (!defined('SMTP_ENABLED') || !SMTP_ENABLED) {
        logEmail('Email not sent - SMTP disabled', $to, $subject);
        return false;
    }
    
    // Validate configuration
    if (empty(SMTP_HOST) || empty(SMTP_USERNAME) || empty(SMTP_PASSWORD)) {
        logEmail('Email not sent - SMTP not configured', $to, $subject);
        error_log('SMTP configuration incomplete. Check config.php');
        return false;
    }
    
    // Try to use PHPMailer if available, otherwise use basic PHP mail
    if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
        return sendEmailPHPMailer($to, $subject, $body, $options);
    } else {
        return sendEmailBasic($to, $subject, $body, $options);
    }
}

/**
 * Send email using PHPMailer (recommended)
 */
function sendEmailPHPMailer($to, $subject, $body, $options = []) {
    require_once __DIR__ . '/phpmailer/PHPMailer.php';
    require_once __DIR__ . '/phpmailer/SMTP.php';
    require_once __DIR__ . '/phpmailer/Exception.php';
    
    $mail = new PHPMailer\PHPMailer\PHPMailer(true);
    
    try {
        // Server settings
        if (EMAIL_DEBUG) {
            $mail->SMTPDebug = 2; // Detailed debug output
        }
        
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        $mail->Timeout = SMTP_TIMEOUT;
        $mail->CharSet = 'UTF-8';
        
        // Recipients
        $mail->setFrom(SMTP_FROM_EMAIL, SMTP_FROM_NAME);
        
        // Add recipients
        if (is_array($to)) {
            foreach ($to as $email) {
                $mail->addAddress(trim($email));
            }
        } else {
            $mail->addAddress($to);
        }
        
        // Reply-to
        $replyTo = $options['reply_to'] ?? SMTP_REPLY_TO ?? SMTP_FROM_EMAIL;
        if ($replyTo) {
            $mail->addReplyTo($replyTo);
        }
        
        // CC
        if (!empty($options['cc'])) {
            if (is_array($options['cc'])) {
                foreach ($options['cc'] as $email) {
                    $mail->addCC(trim($email));
                }
            } else {
                $mail->addCC($options['cc']);
            }
        }
        
        // BCC
        if (!empty($options['bcc'])) {
            if (is_array($options['bcc'])) {
                foreach ($options['bcc'] as $email) {
                    $mail->addBCC(trim($email));
                }
            } else {
                $mail->addBCC($options['bcc']);
            }
        }
        
        // Attachments
        if (!empty($options['attachments'])) {
            foreach ($options['attachments'] as $attachment) {
                if (is_array($attachment)) {
                    // Array format: ['path' => '/path/to/file', 'name' => 'display_name']
                    $mail->addAttachment($attachment['path'], $attachment['name'] ?? '');
                } else {
                    // Simple path
                    $mail->addAttachment($attachment);
                }
            }
        }
        
        // Content
        $mail->isHTML(!empty($options['html']) ? $options['html'] : true);
        $mail->Subject = $subject;
        $mail->Body = $body;
        
        // Alternative plain text
        if (!empty($options['alt_body'])) {
            $mail->AltBody = $options['alt_body'];
        } else {
            // Auto-generate plain text from HTML
            $mail->AltBody = strip_tags($body);
        }
        
        // Send
        $mail->send();
        
        logEmail('Email sent successfully', $to, $subject);
        return true;
        
    } catch (Exception $e) {
        logEmail('Email failed: ' . $mail->ErrorInfo, $to, $subject);
        error_log("Email error: {$mail->ErrorInfo}");
        return false;
    }
}

/**
 * Send email using basic PHP mail() function
 * Fallback when PHPMailer is not available
 */
function sendEmailBasic($to, $subject, $body, $options = []) {
    // This is a basic implementation using fsockopen
    // For production, PHPMailer is highly recommended
    
    try {
        $socket = fsockopen(SMTP_HOST, SMTP_PORT, $errno, $errstr, SMTP_TIMEOUT);
        
        if (!$socket) {
            throw new Exception("Cannot connect to SMTP server: $errstr ($errno)");
        }
        
        // Read server response
        $response = fgets($socket, 515);
        if (substr($response, 0, 3) != '220') {
            throw new Exception("Server not ready: $response");
        }
        
        // EHLO
        fputs($socket, "EHLO " . gethostname() . "\r\n");
        $response = fgets($socket, 515);
        
        // STARTTLS
        if (SMTP_ENCRYPTION === 'tls') {
            fputs($socket, "STARTTLS\r\n");
            $response = fgets($socket, 515);
            stream_socket_enable_crypto($socket, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
            
            // EHLO again after TLS
            fputs($socket, "EHLO " . gethostname() . "\r\n");
            $response = fgets($socket, 515);
        }
        
        // AUTH LOGIN
        fputs($socket, "AUTH LOGIN\r\n");
        $response = fgets($socket, 515);
        
        fputs($socket, base64_encode(SMTP_USERNAME) . "\r\n");
        $response = fgets($socket, 515);
        
        fputs($socket, base64_encode(SMTP_PASSWORD) . "\r\n");
        $response = fgets($socket, 515);
        
        if (substr($response, 0, 3) != '235') {
            throw new Exception("Authentication failed: $response");
        }
        
        // MAIL FROM
        fputs($socket, "MAIL FROM: <" . SMTP_FROM_EMAIL . ">\r\n");
        $response = fgets($socket, 515);
        
        // RCPT TO
        $recipients = is_array($to) ? $to : [$to];
        foreach ($recipients as $recipient) {
            fputs($socket, "RCPT TO: <" . trim($recipient) . ">\r\n");
            $response = fgets($socket, 515);
        }
        
        // DATA
        fputs($socket, "DATA\r\n");
        $response = fgets($socket, 515);
        
        // Headers
        $headers = "From: " . SMTP_FROM_NAME . " <" . SMTP_FROM_EMAIL . ">\r\n";
        $headers .= "Reply-To: " . (SMTP_REPLY_TO ?? SMTP_FROM_EMAIL) . "\r\n";
        $headers .= "MIME-Version: 1.0\r\n";
        $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        $headers .= "Subject: " . $subject . "\r\n";
        
        // Send message
        fputs($socket, $headers . "\r\n" . $body . "\r\n.\r\n");
        $response = fgets($socket, 515);
        
        // QUIT
        fputs($socket, "QUIT\r\n");
        fclose($socket);
        
        logEmail('Email sent successfully (basic method)', $to, $subject);
        return true;
        
    } catch (Exception $e) {
        logEmail('Email failed: ' . $e->getMessage(), $to, $subject);
        error_log("Email error: " . $e->getMessage());
        return false;
    }
}

/**
 * Send assessment report email
 */
function sendAssessmentReportEmail($location_id, $assessment_id, $recipients = null) {
    global $pdo;
    
    // Get assessment details
    $stmt = $pdo->prepare("
        SELECT a.*, l.location_name, l.city, u.full_name as assessor_name
        FROM assessments a
        LEFT JOIN locations l ON a.location_id = l.id
        LEFT JOIN users u ON a.user_id = u.id
        WHERE a.id = ?
    ");
    $stmt->execute([$assessment_id]);
    $assessment = $stmt->fetch();
    
    if (!$assessment) {
        return false;
    }
    
    // Get recipients
    if (!$recipients) {
        $stmt = $pdo->prepare("SELECT report_emails FROM locations WHERE id = ?");
        $stmt->execute([$location_id]);
        $location = $stmt->fetch();
        
        if (empty($location['report_emails'])) {
            return false;
        }
        
        $recipients = array_map('trim', explode(',', $location['report_emails']));
    }
    
    // Count risk items
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN risk_score >= 20 THEN 1 ELSE 0 END) as high_risk,
            SUM(CASE WHEN risk_score >= 15 AND risk_score < 20 THEN 1 ELSE 0 END) as medium_risk,
            SUM(CASE WHEN risk_score < 15 THEN 1 ELSE 0 END) as low_risk,
            SUM(CASE WHEN additional_actions IS NOT NULL AND additional_actions != '' THEN 1 ELSE 0 END) as action_items
        FROM assessment_details
        WHERE assessment_id = ?
    ");
    $stmt->execute([$assessment_id]);
    $stats = $stmt->fetch();
    
    // Build email
    $subject = "Risk Assessment Complete - {$assessment['location_name']}";
    
    $body = generateAssessmentEmailBody($assessment, $stats);
    
    // Send email
    return sendEmail($recipients, $subject, $body);
}

/**
 * Generate assessment email body
 */
function generateAssessmentEmailBody($assessment, $stats) {
    $riskColor = [
        'LOW RISK' => '#10b981',
        'MEDIUM RISK' => '#f59e0b',
        'HIGH RISK' => '#ef4444',
        'CRITICAL RISK' => '#dc2626'
    ];
    
    $color = $riskColor[$assessment['overall_risk_level']] ?? '#6b7280';
    
    $html = "
    <!DOCTYPE html>
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; }
            .info-box { background: white; padding: 15px; margin: 15px 0; border-radius: 6px; border-left: 4px solid {$color}; }
            .info-label { font-weight: bold; color: #6b7280; font-size: 0.85rem; text-transform: uppercase; margin-bottom: 5px; }
            .info-value { font-size: 1.1rem; color: #111827; }
            .stats { display: grid; grid-template-columns: repeat(2, 1fr); gap: 15px; margin: 20px 0; }
            .stat-card { background: white; padding: 15px; border-radius: 6px; text-align: center; }
            .stat-value { font-size: 2rem; font-weight: bold; color: #667eea; }
            .stat-label { color: #6b7280; font-size: 0.9rem; }
            .risk-badge { display: inline-block; padding: 8px 16px; border-radius: 20px; font-weight: bold; color: white; background: {$color}; }
            .button { display: inline-block; padding: 12px 24px; background: #667eea; color: white; text-decoration: none; border-radius: 6px; margin: 10px 5px; }
            .footer { text-align: center; padding: 20px; color: #6b7280; font-size: 0.9rem; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h1>🎯 Risk Assessment Complete</h1>
            </div>
            
            <div class='content'>
                <p>Dear Team,</p>
                
                <p>A risk assessment has been completed for <strong>{$assessment['location_name']}</strong>:</p>
                
                <div class='info-box'>
                    <div class='info-label'>Assessment Number</div>
                    <div class='info-value'>{$assessment['assessment_number']}</div>
                </div>
                
                <div class='info-box'>
                    <div class='info-label'>Date & Time</div>
                    <div class='info-value'>" . date('d M Y H:i', strtotime($assessment['assessment_date'])) . "</div>
                </div>
                
                <div class='info-box'>
                    <div class='info-label'>Assessor</div>
                    <div class='info-value'>{$assessment['assessor_name']}</div>
                </div>
                
                <div class='info-box'>
                    <div class='info-label'>Overall Risk Level</div>
                    <div class='info-value'>
                        <span class='risk-badge'>{$assessment['overall_risk_level']}</span>
                    </div>
                </div>
                
                <h3>Summary Statistics</h3>
                
                <div class='stats'>
                    <div class='stat-card'>
                        <div class='stat-value'>{$stats['total']}</div>
                        <div class='stat-label'>Total Risk Items</div>
                    </div>
                    
                    <div class='stat-card'>
                        <div class='stat-value' style='color: #ef4444;'>{$stats['high_risk']}</div>
                        <div class='stat-label'>High Risk Items</div>
                    </div>
                    
                    <div class='stat-card'>
                        <div class='stat-value' style='color: #f59e0b;'>{$stats['medium_risk']}</div>
                        <div class='stat-label'>Medium Risk Items</div>
                    </div>
                    
                    <div class='stat-card'>
                        <div class='stat-value' style='color: #10b981;'>{$stats['low_risk']}</div>
                        <div class='stat-label'>Low Risk Items</div>
                    </div>
                </div>
                
                <div class='info-box'>
                    <div class='info-label'>Action Items</div>
                    <div class='info-value'>{$stats['action_items']} pending actions</div>
                </div>
                
                <div style='text-align: center; margin-top: 30px;'>
                    <a href='" . SITE_URL . "/public/view-assessment.php?id={$assessment['id']}' class='button'>
                        📋 View Full Assessment
                    </a>
                    
                    <a href='" . SITE_URL . "/public/action-items-report.php?location_id={$assessment['location_id']}' class='button'>
                        ⚠️ View Action Items
                    </a>
                </div>
            </div>
            
            <div class='footer'>
                <p>This is an automated message from the Risk Assessment System.</p>
                <p>Please do not reply to this email.</p>
            </div>
        </div>
    </body>
    </html>
    ";
    
    return $html;
}

/**
 * Send action items reminder email
 */
function sendActionItemsReminderEmail($location_id) {
    global $pdo;
    
    // Get location and recipients
    $stmt = $pdo->prepare("SELECT * FROM locations WHERE id = ?");
    $stmt->execute([$location_id]);
    $location = $stmt->fetch();
    
    if (empty($location['report_emails'])) {
        return false;
    }
    
    $recipients = array_map('trim', explode(',', $location['report_emails']));
    
    // Get overdue and due soon items
    $stmt = $pdo->prepare("
        SELECT 
            ad.id, ad.additional_actions, ad.action_owner, ad.target_date, ad.risk_score,
            ri.item_name, rc.category_name, a.assessment_number,
            CASE 
                WHEN ad.target_date < CURDATE() THEN 'overdue'
                WHEN ad.target_date <= DATE_ADD(CURDATE(), INTERVAL 7 DAY) THEN 'due_soon'
                ELSE 'on_track'
            END as status
        FROM assessment_details ad
        JOIN assessments a ON ad.assessment_id = a.id
        JOIN risk_items ri ON ad.risk_item_id = ri.id
        JOIN risk_categories rc ON ri.category_id = rc.id
        WHERE a.location_id = ? 
          AND a.status = 'completed'
          AND ad.additional_actions IS NOT NULL 
          AND ad.additional_actions != ''
          AND ad.target_date IS NOT NULL
          AND ad.target_date <= DATE_ADD(CURDATE(), INTERVAL 7 DAY)
        ORDER BY ad.target_date ASC
    ");
    $stmt->execute([$location_id]);
    $items = $stmt->fetchAll();
    
    if (empty($items)) {
        return false;
    }
    
    $subject = "Action Items Reminder - {$location['location_name']}";
    $body = generateActionItemsEmailBody($location, $items);
    
    return sendEmail($recipients, $subject, $body);
}

/**
 * Generate action items email body
 */
function generateActionItemsEmailBody($location, $items) {
    $overdue = array_filter($items, fn($i) => $i['status'] === 'overdue');
    $dueSoon = array_filter($items, fn($i) => $i['status'] === 'due_soon');
    
    $html = "
    <!DOCTYPE html>
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #f59e0b 0%, #ef4444 100%); color: white; padding: 30px; text-align: center; border-radius: 8px 8px 0 0; }
            .content { background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; }
            .alert-box { background: #fee2e2; border-left: 4px solid #ef4444; padding: 15px; margin: 15px 0; border-radius: 6px; }
            .item { background: white; padding: 15px; margin: 10px 0; border-radius: 6px; border-left: 4px solid #f59e0b; }
            .item-overdue { border-left-color: #ef4444; }
            .footer { text-align: center; padding: 20px; color: #6b7280; font-size: 0.9rem; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h1>⚠️ Action Items Reminder</h1>
            </div>
            
            <div class='content'>
                <p>Dear Team,</p>
                
                <p>This is a reminder about pending action items for <strong>{$location['location_name']}</strong>:</p>
                
                " . (count($overdue) > 0 ? "
                <div class='alert-box'>
                    <strong>⚠️ " . count($overdue) . " Overdue Action" . (count($overdue) > 1 ? 's' : '') . "</strong>
                </div>
                " : "") . "
                
                <h3>Action Items Requiring Attention:</h3>";
    
    foreach ($items as $item) {
        $daysText = $item['status'] === 'overdue' 
            ? 'OVERDUE by ' . abs((strtotime($item['target_date']) - time()) / 86400) . ' days'
            : 'Due in ' . ceil((strtotime($item['target_date']) - time()) / 86400) . ' days';
        
        $itemClass = $item['status'] === 'overdue' ? 'item item-overdue' : 'item';
        
        $html .= "
                <div class='$itemClass'>
                    <strong>{$item['category_name']}: {$item['item_name']}</strong><br>
                    <em>{$item['additional_actions']}</em><br>
                    <small>
                        Owner: {$item['action_owner']}<br>
                        Due: " . date('d M Y', strtotime($item['target_date'])) . " ($daysText)<br>
                        Risk Score: {$item['risk_score']}<br>
                        Assessment: {$item['assessment_number']}
                    </small>
                </div>";
    }
    
    $html .= "
                <div style='text-align: center; margin-top: 30px;'>
                    <a href='" . SITE_URL . "/public/action-items-report.php?location_id={$location['id']}' 
                       style='display: inline-block; padding: 12px 24px; background: #667eea; color: white; text-decoration: none; border-radius: 6px;'>
                        📋 View All Action Items
                    </a>
                </div>
            </div>
            
            <div class='footer'>
                <p>This is an automated reminder from the Risk Assessment System.</p>
            </div>
        </div>
    </body>
    </html>
    ";
    
    return $html;
}

/**
 * Log email attempt
 */
function logEmail($message, $to, $subject) {
    if (!EMAIL_LOG_ENABLED) {
        return;
    }
    
    $toList = is_array($to) ? implode(', ', $to) : $to;
    
    $logMessage = sprintf(
        "[%s] %s | To: %s | Subject: %s\n",
        date('Y-m-d H:i:s'),
        $message,
        $toList,
        $subject
    );
    
    error_log($logMessage, 3, EMAIL_LOG_FILE);
}

/**
 * Test email configuration
 */
function testEmailConfiguration() {
    $testEmail = SMTP_USERNAME; // Send test to yourself
    $subject = "Test Email - Risk Assessment System";
    $body = "
    <html>
    <body style='font-family: Arial, sans-serif; padding: 20px;'>
        <h2>✅ Email Configuration Test</h2>
        <p>If you're reading this, your email configuration is working correctly!</p>
        <p><strong>SMTP Settings:</strong></p>
        <ul>
            <li>Host: " . SMTP_HOST . "</li>
            <li>Port: " . SMTP_PORT . "</li>
            <li>Encryption: " . SMTP_ENCRYPTION . "</li>
            <li>From: " . SMTP_FROM_EMAIL . "</li>
        </ul>
        <p>Test sent at: " . date('Y-m-d H:i:s') . "</p>
    </body>
    </html>
    ";
    
    return sendEmail($testEmail, $subject, $body);
}
