#!/bin/bash

# ============================================================================
# Risk Assessment System - Emergency Fix Script
# Fixes all known issues and validates system
# ============================================================================

echo "=============================================="
echo "  Risk Assessment System - Emergency Fix     "
echo "=============================================="
echo ""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

ERRORS=0
WARNINGS=0
FIXED=0

echo "Step 1: Checking Database Connection"
echo "======================================"

# Check if MySQL is accessible
if command -v mysql >/dev/null 2>&1; then
    echo -e "${GREEN}✓${NC} MySQL command found"
else
    echo -e "${RED}✗${NC} MySQL command not found"
    echo "Please install MySQL or add it to PATH"
    exit 1
fi

echo ""
echo "Step 2: Running Emergency Database Fix"
echo "========================================"

# Ask for database credentials
read -p "Enter MySQL username (default: root): " DB_USER
DB_USER=${DB_USER:-root}

read -sp "Enter MySQL password: " DB_PASS
echo ""

read -p "Enter database name (default: scubatricky_risk): " DB_NAME
DB_NAME=${DB_NAME:-scubatricky_risk}

# Run the emergency fix SQL
echo "Running emergency fix migration..."
mysql -u "$DB_USER" -p"$DB_PASS" "$DB_NAME" < includes/emergency_fix.sql 2>&1

if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓${NC} Database emergency fix applied successfully"
    FIXED=$((FIXED + 1))
else
    echo -e "${RED}✗${NC} Failed to apply database fix"
    ERRORS=$((ERRORS + 1))
fi

echo ""
echo "Step 3: Checking File Permissions"
echo "=================================="

DIRS=("uploads" "logs" "public/uploads/branding")

for dir in "${DIRS[@]}"; do
    if [ ! -d "$dir" ]; then
        echo -e "${YELLOW}→${NC} Creating directory: $dir"
        mkdir -p "$dir"
        FIXED=$((FIXED + 1))
    fi
    
    if [ -w "$dir" ]; then
        echo -e "${GREEN}✓${NC} $dir is writable"
    else
        echo -e "${YELLOW}⚠${NC} $dir is not writable, attempting to fix..."
        chmod 755 "$dir" 2>/dev/null
        WARNINGS=$((WARNINGS + 1))
    fi
done

echo ""
echo "Step 4: Checking PHP Files"
echo "============================"

# Check for syntax errors in key PHP files
KEY_FILES=(
    "public/admin-panel.php"
    "public/custom-fields.php"
    "public/system-customization.php"
    "public/permission-management.php"
    "public/version-control.php"
    "includes/config.php"
    "templates/header.php"
)

for file in "${KEY_FILES[@]}"; do
    if [ -f "$file" ]; then
        php -l "$file" >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            echo -e "${GREEN}✓${NC} $file syntax OK"
        else
            echo -e "${RED}✗${NC} $file has syntax errors"
            ERRORS=$((ERRORS + 1))
        fi
    else
        echo -e "${YELLOW}⚠${NC} $file not found"
        WARNINGS=$((WARNINGS + 1))
    fi
done

echo ""
echo "Step 5: Checking Required Tables"
echo "=================================="

# Check if required tables exist
TABLES=(
    "users"
    "locations"
    "assessments"
    "assessment_details"
    "system_settings"
    "system_files"
    "custom_fields"
    "permissions"
)

for table in "${TABLES[@]}"; do
    RESULT=$(mysql -u "$DB_USER" -p"$DB_PASS" "$DB_NAME" -e "SHOW TABLES LIKE '$table'" 2>/dev/null | grep -c "$table")
    if [ "$RESULT" -eq 1 ]; then
        echo -e "${GREEN}✓${NC} Table $table exists"
    else
        echo -e "${RED}✗${NC} Table $table missing"
        ERRORS=$((ERRORS + 1))
    fi
done

echo ""
echo "Step 6: Verifying Config File"
echo "==============================="

if [ -f "includes/config.php" ]; then
    echo -e "${GREEN}✓${NC} config.php exists"
    
    # Check for critical config settings
    if grep -q "DB_HOST" includes/config.php; then
        echo -e "${GREEN}✓${NC} Database host configured"
    else
        echo -e "${RED}✗${NC} Database host not configured"
        ERRORS=$((ERRORS + 1))
    fi
else
    echo -e "${RED}✗${NC} config.php not found"
    ERRORS=$((ERRORS + 1))
fi

echo ""
echo "Step 7: Checking for Function Conflicts"
echo "=========================================="

# Check if header files have function_exists checks
if grep -q "function_exists('getSystemName')" templates/header.php; then
    echo -e "${GREEN}✓${NC} header.php has function_exists checks"
else
    echo -e "${YELLOW}⚠${NC} header.php may cause function redeclaration"
    WARNINGS=$((WARNINGS + 1))
fi

if grep -q "function_exists('getSystemName')" templates/header-responsive.php 2>/dev/null; then
    echo -e "${GREEN}✓${NC} header-responsive.php has function_exists checks"
else
    echo -e "${YELLOW}⚠${NC} header-responsive.php may cause function redeclaration"
    WARNINGS=$((WARNINGS + 1))
fi

echo ""
echo "Step 8: Creating Missing Files"
echo "================================"

# Check for .gitkeep files
for dir in uploads logs; do
    if [ ! -f "$dir/.gitkeep" ]; then
        touch "$dir/.gitkeep"
        echo -e "${GREEN}✓${NC} Created $dir/.gitkeep"
        FIXED=$((FIXED + 1))
    fi
done

echo ""
echo "=============================================="
echo "  Fix Summary                                "
echo "=============================================="
echo ""
echo -e "Errors Found:    ${RED}$ERRORS${NC}"
echo -e "Warnings:        ${YELLOW}$WARNINGS${NC}"
echo -e "Issues Fixed:    ${GREEN}$FIXED${NC}"
echo ""

if [ $ERRORS -eq 0 ]; then
    echo -e "${GREEN}✓ All critical issues resolved!${NC}"
    echo ""
    echo "Next steps:"
    echo "1. Clear browser cache"
    echo "2. Login to the application"
    echo "3. Test admin panel functionality"
    echo "4. Check error logs if issues persist"
else
    echo -e "${RED}✗ Some errors remain${NC}"
    echo ""
    echo "Please:"
    echo "1. Check the error messages above"
    echo "2. Review logs/system-errors.log"
    echo "3. Verify database credentials in config.php"
    echo "4. Ensure MySQL is running"
fi

echo ""
echo "Log files to check if issues persist:"
echo "  - logs/system-errors.log"
echo "  - logs/debug.log"
echo "  - logs/security.log"
echo ""
