# SMTP Email Configuration Guide

## 🎯 Overview

The Risk Assessment System includes comprehensive SMTP email functionality for sending automated assessment reports, action item reminders, and system notifications.

---

## ⚙️ Configuration

### **Step 1: Edit config.php**

Open `includes/config.php` and configure the SMTP settings:

```php
// SMTP Email Configuration
define('SMTP_ENABLED', true);                    // Enable/disable email
define('SMTP_HOST', 'smtp.gmail.com');          // SMTP server
define('SMTP_PORT', 587);                        // SMTP port
define('SMTP_ENCRYPTION', 'tls');                // tls, ssl, or '' for none
define('SMTP_USERNAME', 'your-email@gmail.com'); // Email address
define('SMTP_PASSWORD', 'your-app-password');    // Email password
define('SMTP_FROM_EMAIL', 'your-email@gmail.com'); // From address
define('SMTP_FROM_NAME', 'Risk Assessment System'); // From name
```

### **Step 2: Test Configuration**

Access the SMTP settings page:
```
Admin Panel → SMTP Email Settings → Send Test Email
```

---

## 📧 Common SMTP Providers

### **Gmail**

```php
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-email@gmail.com');
define('SMTP_PASSWORD', 'your-app-password'); // NOT regular password!
```

**Important:** Gmail requires an App Password:
1. Go to Google Account → Security
2. Enable 2-Step Verification
3. Generate App Password
4. Use that password (not your regular Gmail password)

**Create App Password:** https://support.google.com/accounts/answer/185833

### **Microsoft 365 / Outlook**

```php
define('SMTP_HOST', 'smtp.office365.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-email@outlook.com');
define('SMTP_PASSWORD', 'your-password');
```

### **SendGrid**

```php
define('SMTP_HOST', 'smtp.sendgrid.net');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'apikey'); // Literally "apikey"
define('SMTP_PASSWORD', 'your-sendgrid-api-key');
```

### **Amazon SES**

```php
define('SMTP_HOST', 'email-smtp.us-east-1.amazonaws.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-ses-smtp-username');
define('SMTP_PASSWORD', 'your-ses-smtp-password');
```

### **Custom SMTP Server**

```php
define('SMTP_HOST', 'mail.yourdomain.com');
define('SMTP_PORT', 587); // or 465 for SSL, 25 for none
define('SMTP_ENCRYPTION', 'tls'); // or 'ssl' or ''
define('SMTP_USERNAME', 'username@yourdomain.com');
define('SMTP_PASSWORD', 'your-password');
```

---

## 🔧 Port & Encryption Guide

| Port | Encryption | Use Case |
|------|-----------|----------|
| 587 | TLS | **Recommended** - Modern standard |
| 465 | SSL | Older standard, still widely used |
| 25 | None | Not recommended (unencrypted) |
| 2525 | TLS/SSL | Alternative when 587 is blocked |

**Recommendation:** Use port 587 with TLS encryption for best security.

---

## 📋 Email Functions

### **Send Email**

```php
require_once '../includes/email_helper.php';

// Simple email
sendEmail(
    'recipient@example.com',
    'Subject Line',
    '<h1>Email Body</h1><p>HTML supported!</p>'
);

// Multiple recipients
sendEmail(
    ['user1@example.com', 'user2@example.com'],
    'Subject Line',
    '<p>Email body</p>'
);

// With options
sendEmail(
    'recipient@example.com',
    'Subject Line',
    '<p>Email body</p>',
    [
        'cc' => ['cc@example.com'],
        'bcc' => ['bcc@example.com'],
        'reply_to' => 'reply@example.com',
        'attachments' => [
            ['path' => '/path/to/file.pdf', 'name' => 'Report.pdf']
        ],
        'html' => true
    ]
);
```

### **Send Assessment Report**

```php
// Send assessment completion email
sendAssessmentReportEmail($location_id, $assessment_id);

// Or specify recipients
sendAssessmentReportEmail(
    $location_id, 
    $assessment_id,
    ['manager@store.com', 'safety@company.com']
);
```

### **Send Action Items Reminder**

```php
// Send reminder for overdue/due soon items
sendActionItemsReminderEmail($location_id);
```

### **Test Configuration**

```php
// Send test email to verify settings
if (testEmailConfiguration()) {
    echo "Email configuration is working!";
} else {
    echo "Email configuration has errors.";
}
```

---

## 📊 Email Templates

### **Assessment Complete Email**

Automatically includes:
- ✅ Assessment number and date
- ✅ Location name
- ✅ Assessor name
- ✅ Overall risk level
- ✅ Statistics (high/medium/low risk counts)
- ✅ Action items count
- ✅ Links to view assessment and action items
- ✅ Professional HTML design

### **Action Items Reminder Email**

Automatically includes:
- ✅ Location name
- ✅ Overdue items count (highlighted)
- ✅ Each action item with:
  - Risk category and hazard
  - Action description
  - Owner and due date
  - Days overdue/until due
  - Risk score
- ✅ Link to action items report

---

## 🔍 Troubleshooting

### **Authentication Failed**

**Problem:** Can't authenticate with SMTP server

**Solutions:**
1. **Gmail:** Use App Password, not regular password
2. Verify username (usually full email address)
3. Check password for typos
4. Ensure 2FA is configured for Gmail
5. Try different SMTP server

### **Connection Timeout**

**Problem:** Can't connect to SMTP server

**Solutions:**
1. Check firewall settings
2. Verify port is open (test: `telnet smtp.gmail.com 587`)
3. Try alternative port (2525)
4. Check if hosting blocks outbound SMTP
5. Contact hosting provider

### **SSL/TLS Errors**

**Problem:** Encryption handshake fails

**Solutions:**
1. Try `tls` instead of `ssl`
2. Try `ssl` instead of `tls`
3. Try port 465 with SSL
4. Try port 587 with TLS
5. Update PHP OpenSSL extension

### **Email Not Sending**

**Problem:** No error but email not received

**Solutions:**
1. Check spam/junk folder
2. Verify `SMTP_ENABLED` is `true`
3. Check email logs: `logs/email.log`
4. Verify recipient email is valid
5. Check email rate limits
6. Test with different recipient

### **Emails Go to Spam**

**Problem:** Emails delivered to spam folder

**Solutions:**
1. Set up SPF record for your domain
2. Set up DKIM signing
3. Set up DMARC policy
4. Use authenticated SMTP server
5. Avoid spam trigger words
6. Include unsubscribe link

---

## 📁 File Locations

**Configuration:**
```
includes/config.php          # SMTP settings
includes/email_helper.php    # Email functions
```

**Admin Interface:**
```
public/smtp-settings.php     # SMTP configuration page
```

**Logs:**
```
logs/email.log              # Email activity log
logs/system-errors.log      # SMTP errors
```

---

## 🔐 Security Best Practices

### **Protect Credentials**

```bash
# Set restrictive permissions
chmod 600 includes/config.php

# Never commit to version control
# .gitignore should include:
includes/config.php
```

### **Use App Passwords**

For Gmail and similar:
- Never use your main account password
- Generate app-specific passwords
- Revoke if compromised

### **Enable Encryption**

```php
// Always use encryption
define('SMTP_ENCRYPTION', 'tls'); // Good ✓
define('SMTP_ENCRYPTION', '');    // Bad ✗
```

### **Limit Recipients**

```php
// Limit recipients per email
define('EMAIL_MAX_RECIPIENTS', 10);

// System enforces this limit
```

---

## 📈 Email Logging

All email attempts are logged to `logs/email.log`:

```
[2026-01-08 10:30:45] Email sent successfully | To: manager@store.com | Subject: Assessment Complete
[2026-01-08 10:31:12] Email failed: Authentication failed | To: user@example.com | Subject: Test
```

**View Logs:**
```
Admin Panel → SMTP Email Settings → Recent Email Activity
```

---

## 🔄 Automated Email Schedule

### **Immediate (After Assessment)**

```php
// In assessment completion code:
if ($location['email_frequency'] === 'immediate' && $location['email_enabled']) {
    sendAssessmentReportEmail($location_id, $assessment_id);
}
```

### **Daily Digest (Cron Job)**

```bash
# Add to crontab (runs at 6 PM daily)
0 18 * * * php /path/to/risk-system/includes/send-daily-digest.php
```

### **Weekly Summary (Cron Job)**

```bash
# Add to crontab (runs Monday 9 AM)
0 9 * * 1 php /path/to/risk-system/includes/send-weekly-summary.php
```

### **Monthly Report (Cron Job)**

```bash
# Add to crontab (runs 1st of month 9 AM)
0 9 1 * * php /path/to/risk-system/includes/send-monthly-report.php
```

---

## ✅ Testing Checklist

Before going live with email:

- [ ] SMTP settings configured in config.php
- [ ] SMTP_ENABLED set to true
- [ ] Test email sent successfully
- [ ] Test email received in inbox (not spam)
- [ ] Multiple recipients work
- [ ] HTML formatting displays correctly
- [ ] Links in emails work
- [ ] Attachments work (if used)
- [ ] Email logs are created
- [ ] Error handling works
- [ ] Rate limits respected

---

## 💡 Advanced Configuration

### **Email Queue (Optional)**

```php
// Enable queue for high-volume sending
define('EMAIL_QUEUE_ENABLED', true);

// Emails will be queued and sent by cron job
// Prevents timeouts on bulk sending
```

### **Debug Mode**

```php
// Enable detailed SMTP debug output
define('EMAIL_DEBUG', true);

// Shows detailed SMTP conversation
// Only use in development!
```

### **Custom From Address**

```php
// Use different reply-to address
define('SMTP_REPLY_TO', 'noreply@yourdomain.com');

// Or per-email:
sendEmail($to, $subject, $body, [
    'reply_to' => 'support@yourdomain.com'
]);
```

---

## 📞 Quick Reference

**Enable Email:**
```php
define('SMTP_ENABLED', true);
```

**Test Email:**
```
Admin Panel → SMTP Settings → Send Test Email
```

**Check Logs:**
```
Admin Panel → SMTP Settings → Recent Activity
OR
logs/email.log
```

**Send Email:**
```php
require_once 'includes/email_helper.php';
sendEmail($to, $subject, $body);
```

---

## 🎯 Summary

**Setup Steps:**
1. ✅ Configure SMTP in config.php
2. ✅ Test with "Send Test Email"
3. ✅ Configure location email recipients
4. ✅ Enable automated reports
5. ✅ Monitor email logs

**Email sent for:**
- Assessment completion (immediate mode)
- Action item reminders (scheduled)
- System notifications
- Custom triggers

**Start sending automated reports today!** 📧✨
