# ERROR REPORTING & CODE CHECKING SYSTEM

## 🎯 Overview

The Risk Assessment System now includes a comprehensive error reporting and code checking system to help monitor, debug, and maintain the application.

---

## 📋 Components

### 1. **Automated Code Checker** (`check-system.sh`)
Validates system integrity and code quality

### 2. **Error Handler** (`includes/error_handler.php`)
Centralized error logging and reporting

### 3. **Error Log Viewer** (`public/error-logs.php`)
Admin interface for viewing and managing logs

---

## 🔍 Code Checker

### **How to Run**

```bash
cd /path/to/risk-system
./check-system.sh
```

### **What It Checks**

**File Structure (Step 1):**
- ✓ Core directories exist (includes, public, templates, uploads, logs)
- ✓ Required files present

**Core PHP Files (Step 2):**
- ✓ config.php, permissions.php, version_control.php
- ✓ All core public pages (login, dashboard, admin, etc.)

**PHP Syntax (Step 3):**
- ✓ Validates PHP syntax in all files
- ✓ Reports syntax errors with file names

**Database Schema (Step 4):**
- ✓ Checks migration files exist
- ✓ Validates CREATE TABLE statements

**Configuration (Step 5):**
- ⚠️ Warns about localhost database
- ⚠️ Warns about empty passwords

**Permissions (Step 6):**
- ✓ Uploads directory writable
- ✓ Logs directory writable

**Common Issues (Step 7):**
- ✓ No incorrect password_hash column references
- ⚠️ Warns about hardcoded credentials
- ⚠️ Checks for SQL injection vulnerabilities

**Templates (Step 8):**
- ✓ Header and footer templates exist

**Documentation (Step 9):**
- ⚠️ Checks for README and guides

**Security (Step 10):**
- ⚠️ Warns about debug files in production
- ⚠️ Warns about .git directory

### **Output Example**

```
==========================================
  Risk Assessment System Code Checker   
==========================================

Step 1: Checking File Structure
==================================
✓ includes/ directory exists
✓ public/ directory exists
✓ templates/ directory exists
✓ uploads/ directory exists
✓ logs/ directory exists

Step 2: Checking Core PHP Files
==================================
✓ config.php exists
✓ permissions.php exists
✓ login.php exists
✓ dashboard.php exists
...

==========================================
  Code Check Summary                     
==========================================

Total Checks: 45
Passed: 42
Failed: 0
Warnings: 3

✓ All critical checks passed!
```

---

## 📊 Error Reporting System

### **Features**

**1. Automatic Error Logging**
- All PHP errors logged automatically
- Database errors captured
- Security events tracked
- Debug information when needed

**2. Environment Detection**
- **Production:** Errors logged, not displayed
- **Development:** Errors logged AND displayed

**3. Multiple Log Files**
- `logs/system-errors.log` - PHP and application errors
- `logs/debug.log` - Debug information
- `logs/security.log` - Security events

**4. User-Friendly Error Pages**
- Custom error pages for fatal errors
- No technical details exposed to users
- Clear call-to-action buttons

---

## 🔧 Error Handler Functions

### **Initialization**

```php
// Already included in config.php
require_once __DIR__ . '/error_handler.php';
```

### **Available Functions**

**1. Log Database Errors**
```php
try {
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
} catch (PDOException $e) {
    logDatabaseError($query, $e->getMessage(), $params);
    throw $e;
}
```

**2. Log Security Events**
```php
// Failed login attempt
logSecurityEvent('Failed login attempt', "Username: $username");

// Unauthorized access
logSecurityEvent('Unauthorized access', "User tried to access admin panel");

// Permission denial
logSecurityEvent('Permission denied', "User $user_id tried to $action");
```

**3. Log Debug Information**
```php
logDebug('Processing assessment', [
    'assessment_id' => $id,
    'user_id' => $_SESSION['user_id'],
    'action' => 'update'
]);
```

**4. Safe Database Queries**
```php
// Automatically logs errors
$stmt = safeQuery($pdo, "SELECT * FROM users WHERE id = ?", [$user_id]);
```

**5. Get Error Statistics**
```php
$stats = getErrorStats();
// Returns:
// - total_errors
// - recent_errors (last 24 hours)
// - security_events
// - last_error (time and message)
```

**6. Clear Old Logs**
```php
// Delete logs older than 30 days
clearOldLogs(30);
```

---

## 👀 Error Log Viewer

### **Access**

```
Admin Panel → Error Logs
OR
http://yoursite.com/public/error-logs.php
```

### **Features**

**1. Three Log Types**
- 📝 System Errors
- 🐛 Debug Log
- 🔒 Security Events

**2. Statistics Dashboard**
- Total errors count
- Last 24 hours count
- Security events count
- Last error time

**3. Viewing Options**
- Show 50/100/200/500 lines
- Newest entries first
- Color-coded by severity
- Timestamps visible

**4. Management Actions**
- 🗑️ Clear Log - Remove all entries
- 🧹 Cleanup Old - Delete entries >30 days
- 🔄 Refresh - Reload current view

**5. Visual Indicators**
- 🔴 Red - Errors and fatal errors
- 🟡 Yellow - Warnings and notices
- 🔴 Dark red - Security events

### **Log Entry Format**

```
[2026-01-08 17:30:45] ERROR: Division by zero in calculations.php on line 42
Request: /public/assessments.php
User: 1
--------------------------------------------------------------------------------
```

---

## 🚨 Error Types

### **System Errors**

Logged to: `logs/system-errors.log`

**Includes:**
- PHP errors (E_ERROR, E_WARNING, etc.)
- Exceptions
- Fatal errors
- Database errors
- Application errors

**Example:**
```
[2026-01-08 10:30:00] ERROR: Undefined variable $user_id in login.php on line 25
Request: /public/login.php
User: Not logged in
--------------------------------------------------------------------------------
```

### **Security Events**

Logged to: `logs/security.log`

**Includes:**
- Failed login attempts
- Unauthorized access attempts
- Permission denials
- Suspicious activity

**Example:**
```
[2026-01-08 10:35:00] SECURITY: Failed login attempt
Details: Username: admin
IP: 192.168.1.100
User: Not logged in
Request: /public/login.php
--------------------------------------------------------------------------------
```

### **Debug Information**

Logged to: `logs/debug.log`

**Includes:**
- Development debugging
- Process tracking
- Variable dumps
- Performance metrics

**Example:**
```
[2026-01-08 10:40:00] DEBUG: Processing assessment
Context: {"assessment_id":123,"user_id":1,"action":"update"}
--------------------------------------------------------------------------------
```

---

## 💡 Best Practices

### **Development**

```php
// Use debug logging
logDebug('Testing new feature', ['step' => 1, 'data' => $data]);

// Display errors for development
ini_set('display_errors', 1);
error_reporting(E_ALL);
```

### **Production**

```php
// Hide errors from users
ini_set('display_errors', 0);
error_reporting(E_ALL);

// Log everything
ini_set('log_errors', 1);

// Monitor logs regularly
// Check error-logs.php daily
```

### **Error Handling**

```php
// Always wrap risky operations
try {
    // Risky code
    $result = doSomething();
} catch (Exception $e) {
    // Log the error
    error_log($e->getMessage());
    
    // Show user-friendly message
    $error = "An error occurred. Please try again.";
}
```

### **Database Queries**

```php
// Use safeQuery for automatic error logging
$stmt = safeQuery($pdo, $query, $params);

// Or wrap in try-catch
try {
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
} catch (PDOException $e) {
    logDatabaseError($query, $e->getMessage(), $params);
    // Handle error appropriately
}
```

### **Security Logging**

```php
// Log all security-relevant events
if ($login_failed) {
    logSecurityEvent('Failed login', "Username: $username");
}

if (!hasPermission($user_id, 'admin_access')) {
    logSecurityEvent('Unauthorized access', "User $user_id");
}
```

---

## 📈 Monitoring

### **Daily Checks**

1. **View Error Logs**
   - Go to Admin Panel → Error Logs
   - Check for new errors
   - Review security events

2. **Check Statistics**
   - Review error counts
   - Monitor trends
   - Identify patterns

3. **Address Issues**
   - Fix critical errors immediately
   - Plan fixes for warnings
   - Document recurring issues

### **Weekly Maintenance**

1. **Run Code Checker**
   ```bash
   ./check-system.sh
   ```

2. **Review Debug Logs**
   - Check for performance issues
   - Identify optimization opportunities

3. **Cleanup Old Logs**
   - Use "Cleanup Old" button
   - Or run: `clearOldLogs(30)`

---

## 🔧 Configuration

### **Error Log Locations**

Defined in `includes/error_handler.php`:

```php
define('ERROR_LOG_FILE', __DIR__ . '/../logs/system-errors.log');
define('DEBUG_LOG_FILE', __DIR__ . '/../logs/debug.log');
define('SECURITY_LOG_FILE', __DIR__ . '/../logs/security.log');
```

### **Log Retention**

Default: 30 days

Change in function call:
```php
clearOldLogs(60); // Keep 60 days
```

### **Error Display**

Automatically detected:
- **localhost/127.0.0.1** = Development (display ON)
- **Other domains** = Production (display OFF)

---

## ✅ Quick Start

### **1. Run Code Checker**

```bash
cd /path/to/risk-system
./check-system.sh
```

Fix any critical issues reported.

### **2. Check Error Logs**

```
1. Login as admin
2. Go to Admin Panel
3. Click "Error Logs"
4. Review any errors
```

### **3. Monitor Regularly**

- Check logs daily
- Run code checker weekly
- Clean old logs monthly

---

## 🎯 Summary

**Components:**
- ✅ Automated code checker script
- ✅ Centralized error handler
- ✅ Admin log viewer interface
- ✅ Three separate log types
- ✅ Security event tracking
- ✅ Debug logging capability

**Benefits:**
- 🔍 Easy error monitoring
- 🐛 Faster debugging
- 🔒 Security auditing
- 📊 System health tracking
- ⚡ Proactive issue detection

**Access:**
- Code Checker: `./check-system.sh`
- Error Logs: Admin Panel → Error Logs
- Log Files: `logs/` directory

**Ready to use - start monitoring your system now!** 📊✨
