#!/bin/bash

# ============================================================================
# Risk Assessment System - Code Cleanup & Organization
# Tidies up code, removes debug files, organizes structure
# ============================================================================

echo "=========================================="
echo "  Risk Assessment Code Cleanup           "
echo "=========================================="
echo ""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Counters
FILES_REMOVED=0
FILES_MOVED=0
FILES_CLEANED=0

# Check if we're in the right directory
if [ ! -f "includes/config.php" ]; then
    echo -e "${RED}Error: Please run this script from the risk-system root directory${NC}"
    exit 1
fi

echo "Step 1: Creating Archive Directory for Debug Files"
echo "===================================================="

mkdir -p .archive/debug-files 2>/dev/null
mkdir -p .archive/backup-files 2>/dev/null

echo -e "${GREEN}✓${NC} Archive directories created"
echo ""

echo "Step 2: Moving Debug & Test Files to Archive"
echo "=============================================="

DEBUG_FILES=(
    "public/test-password.php"
    "public/debug-login.php"
    "public/diagnose-admin-panel.php"
    "public/diagnose-login.php"
    "includes/test-password.php"
)

for file in "${DEBUG_FILES[@]}"; do
    if [ -f "$file" ]; then
        mv "$file" ".archive/debug-files/" 2>/dev/null
        echo -e "${YELLOW}→${NC} Moved: $file"
        FILES_MOVED=$((FILES_MOVED + 1))
    fi
done

echo ""
echo "Step 3: Moving Backup Files to Archive"
echo "========================================"

BACKUP_FILES=(
    "public/admin-panel-backup.php"
    "public/view-location-backup.php"
    "public/admin-panel-safe.php"
)

for file in "${BACKUP_FILES[@]}"; do
    if [ -f "$file" ]; then
        mv "$file" ".archive/backup-files/" 2>/dev/null
        echo -e "${YELLOW}→${NC} Moved: $file"
        FILES_MOVED=$((FILES_MOVED + 1))
    fi
done

echo ""
echo "Step 4: Organizing Documentation"
echo "=================================="

mkdir -p docs 2>/dev/null

DOC_FILES=(
    "README.md"
    "LOGIN_TROUBLESHOOTING.md"
    "PASSWORD_RESET_GUIDE.md"
    "ACTION_ITEMS_REPORT_GUIDE.md"
    "ACTION_ITEMS_VISUAL_GUIDE.md"
    "ERROR_REPORTING_GUIDE.md"
    "AUTOMATED_EMAIL_GUIDE.md"
)

for file in "${DOC_FILES[@]}"; do
    if [ -f "$file" ]; then
        if [ "$file" != "README.md" ]; then
            cp "$file" "docs/" 2>/dev/null
            echo -e "${BLUE}→${NC} Copied to docs/: $file"
        fi
    fi
done

echo ""
echo "Step 5: Cleaning Up Empty Directories"
echo "======================================="

find . -type d -empty -delete 2>/dev/null
echo -e "${GREEN}✓${NC} Empty directories removed"

echo ""
echo "Step 6: Setting Proper Permissions"
echo "====================================="

# Make scripts executable
chmod +x check-system.sh 2>/dev/null
chmod +x diagnose-login.sh 2>/dev/null

# Set directory permissions
chmod 755 public includes templates 2>/dev/null
chmod 755 uploads logs 2>/dev/null || mkdir -p uploads logs && chmod 755 uploads logs

# Set file permissions
chmod 644 public/*.php includes/*.php templates/*.php 2>/dev/null
chmod 600 includes/config.php 2>/dev/null

echo -e "${GREEN}✓${NC} Permissions set"

echo ""
echo "Step 7: Creating .gitignore"
echo "============================="

cat > .gitignore << 'EOF'
# Configuration
includes/config.php

# Uploads & User Data
uploads/*
!uploads/.gitkeep

# Logs
logs/*
!logs/.gitkeep

# Debug & Test Files
*-test.php
*-debug.php
*-backup.php
test-*.php
debug-*.php

# Archive
.archive/

# System Files
.DS_Store
Thumbs.db
*.swp
*.swo
*~

# IDE
.vscode/
.idea/
*.sublime-*

# Node (if used)
node_modules/
package-lock.json

# Composer (if used)
vendor/
composer.lock

# Temporary
*.tmp
*.bak
*.old
EOF

echo -e "${GREEN}✓${NC} .gitignore created"

echo ""
echo "Step 8: Creating Project Structure Documentation"
echo "=================================================="

cat > PROJECT_STRUCTURE.md << 'EOF'
# Risk Assessment System - Project Structure

## Directory Organization

```
risk-system/
├── includes/              # Backend PHP includes
│   ├── config.php        # Database & app configuration
│   ├── permissions.php   # Permission system
│   ├── error_handler.php # Error reporting & logging
│   ├── version_control.php # Version control functions
│   ├── complete_schema.sql # Complete database schema
│   └── migration_*.sql   # Database migrations
│
├── public/               # Web-accessible files
│   ├── login.php         # Login page
│   ├── dashboard.php     # Main dashboard
│   ├── assessments.php   # Assessment management
│   ├── locations.php     # Location management
│   ├── users.php         # User management
│   ├── admin-panel.php   # Admin control panel
│   ├── reports.php       # Reports dashboard
│   ├── action-items-report.php # Action items report
│   ├── error-logs.php    # Error log viewer
│   ├── version-control.php # Version control UI
│   ├── css/              # Stylesheets
│   ├── js/               # JavaScript files
│   └── api/              # API endpoints
│
├── templates/            # PHP templates
│   ├── header.php        # Common header
│   └── footer.php        # Common footer
│
├── uploads/              # User uploads (photos, documents)
├── logs/                 # System logs
│   ├── system-errors.log # Error log
│   ├── debug.log         # Debug log
│   └── security.log      # Security events
│
├── docs/                 # Documentation
│   ├── LOGIN_TROUBLESHOOTING.md
│   ├── ACTION_ITEMS_REPORT_GUIDE.md
│   ├── ERROR_REPORTING_GUIDE.md
│   └── AUTOMATED_EMAIL_GUIDE.md
│
├── .archive/             # Archived debug/backup files
│   ├── debug-files/
│   └── backup-files/
│
├── check-system.sh       # System validation script
├── diagnose-login.sh     # Login diagnostic tool
├── cleanup-code.sh       # This cleanup script
└── README.md             # Main documentation
```

## File Naming Conventions

### PHP Files
- **Pages**: `page-name.php` (e.g., `login.php`, `dashboard.php`)
- **Includes**: `functionality.php` (e.g., `config.php`, `permissions.php`)
- **API**: `api/endpoint-name.php` (e.g., `api/save-assessment.php`)

### SQL Files
- **Schema**: `complete_schema.sql`
- **Migrations**: `migration_feature_name.sql`

### JavaScript Files
- **Components**: `component-name.js` (e.g., `vue-components.js`)
- **Utilities**: `utility-name.js` (e.g., `app.js`)

### CSS Files
- **Main**: `style.css`
- **Component**: `component.css`

## Code Organization

### Backend (PHP)
```
includes/
  ├── config.php           # Database connection, constants
  ├── permissions.php      # Authorization functions
  ├── error_handler.php    # Error logging & display
  └── version_control.php  # File versioning
```

### Frontend (Public)
```
public/
  ├── User Management      # users.php, login.php
  ├── Assessments          # assessments.php, new-assessment.php
  ├── Locations            # locations.php, edit-location.php
  ├── Reports              # reports.php, action-items-report.php
  └── Admin                # admin-panel.php, error-logs.php
```

### Templates
```
templates/
  ├── header.php           # <head>, navigation, opening <body>
  └── footer.php           # Closing tags, scripts
```

## Key Features by File

### Core System
- `config.php` - Database, sessions, constants
- `permissions.php` - Role-based access control
- `error_handler.php` - Error logging & handling

### User Interface
- `dashboard.php` - Main landing page
- `admin-panel.php` - System administration
- `reports.php` - Reporting dashboard

### Assessment System
- `assessments.php` - List/manage assessments
- `new-assessment.php` - Create assessment
- `action-items-report.php` - Action tracking

### Location Management
- `locations.php` - List locations
- `edit-location.php` - Edit/create location
- `view-location.php` - Location details

### Administration
- `users.php` - User management
- `error-logs.php` - Error monitoring
- `version-control.php` - File versioning

## Development Workflow

1. **Adding Features**: Create in `public/`, use includes from `includes/`
2. **Database Changes**: Create migration in `includes/migration_*.sql`
3. **Styling**: Update `public/css/style.css`
4. **Scripts**: Add to `public/js/`
5. **Documentation**: Update relevant guide in `docs/`

## Maintenance

- **Logs**: Check `logs/` directory regularly
- **Uploads**: Backup `uploads/` directory
- **Database**: Regular backups, run migrations
- **Code**: Run `check-system.sh` before deployment

## Security

- `includes/config.php` - Not in version control
- `uploads/` - User files, proper permissions
- `logs/` - Writable by web server
- `.htaccess` - Protect sensitive directories
EOF

echo -e "${GREEN}✓${NC} PROJECT_STRUCTURE.md created"

echo ""
echo "Step 9: Creating Deployment Checklist"
echo "======================================="

cat > DEPLOYMENT_CHECKLIST.md << 'EOF'
# Deployment Checklist

## Pre-Deployment

### Code Quality
- [ ] Run `./check-system.sh` - All checks pass
- [ ] Remove debug files (`./cleanup-code.sh`)
- [ ] Review error logs
- [ ] Test all critical functions
- [ ] Check PHP syntax errors

### Database
- [ ] Backup current database
- [ ] Test migrations on staging
- [ ] Verify schema matches complete_schema.sql
- [ ] Check indexes are in place
- [ ] Confirm foreign keys are correct

### Configuration
- [ ] Update `includes/config.php` with production values
- [ ] Set database credentials
- [ ] Configure SITE_URL
- [ ] Set session security (cookie_secure = 1 for HTTPS)
- [ ] Review upload path settings

### Security
- [ ] Change default admin password
- [ ] Remove test accounts
- [ ] Set proper file permissions (755 directories, 644 files)
- [ ] Restrict config.php (600 permissions)
- [ ] Enable HTTPS
- [ ] Configure firewall rules

### Files & Directories
- [ ] Create `uploads/` directory (writable)
- [ ] Create `logs/` directory (writable)
- [ ] Remove `.archive/` directory
- [ ] Verify `.gitignore` is correct
- [ ] Check no sensitive data in repository

## Deployment Steps

1. **Backup Production**
   ```bash
   mysqldump -u user -p database > backup_$(date +%Y%m%d).sql
   tar -czf files_backup_$(date +%Y%m%d).tar.gz public/ includes/
   ```

2. **Upload Files**
   ```bash
   rsync -avz --exclude='.git' --exclude='logs/*' --exclude='uploads/*' \
         ./ user@server:/path/to/app/
   ```

3. **Set Permissions**
   ```bash
   chmod 755 public/ includes/ templates/
   chmod 644 public/*.php includes/*.php
   chmod 600 includes/config.php
   chmod 755 uploads/ logs/
   ```

4. **Run Migrations**
   ```bash
   mysql -u user -p database < includes/complete_schema.sql
   mysql -u user -p database < includes/migration_*.sql
   ```

5. **Test Application**
   - [ ] Login works
   - [ ] Dashboard loads
   - [ ] Create test assessment
   - [ ] Upload photo
   - [ ] Generate report
   - [ ] Check error logs

## Post-Deployment

### Verification
- [ ] All pages load without errors
- [ ] Database connections work
- [ ] File uploads function
- [ ] Email settings configured
- [ ] Permissions work correctly
- [ ] Reports generate properly

### Monitoring
- [ ] Set up error log monitoring
- [ ] Configure backup schedule
- [ ] Monitor disk space
- [ ] Check performance
- [ ] Review security logs

### Documentation
- [ ] Update deployment date in README
- [ ] Document any custom configurations
- [ ] Share credentials securely
- [ ] Update maintenance schedule

## Rollback Plan

If deployment fails:

1. **Restore Database**
   ```bash
   mysql -u user -p database < backup_YYYYMMDD.sql
   ```

2. **Restore Files**
   ```bash
   tar -xzf files_backup_YYYYMMDD.tar.gz -C /path/to/app/
   ```

3. **Clear Cache**
   ```bash
   rm -rf logs/*.log
   ```

4. **Test System**
   - Verify login
   - Check database connection
   - Test core functions

## Production Maintenance

### Daily
- Check error logs
- Monitor disk space
- Review failed logins

### Weekly
- Backup database
- Review user activity
- Check system performance
- Update action items

### Monthly
- Full system backup
- Review permissions
- Update documentation
- Security audit
- Performance optimization

## Support Contacts

- **Technical Support**: support@company.com
- **Database Admin**: dba@company.com
- **Security Team**: security@company.com
EOF

echo -e "${GREEN}✓${NC} DEPLOYMENT_CHECKLIST.md created"

echo ""
echo "Step 10: Creating Development Guidelines"
echo "=========================================="

cat > DEVELOPMENT_GUIDELINES.md << 'EOF'
# Development Guidelines

## Code Standards

### PHP

**File Headers**
```php
<?php
/**
 * Feature Name
 * Brief description of what this file does
 */

require_once '../includes/config.php';
requireLogin(); // If authentication required
```

**Naming Conventions**
- Variables: `$snake_case`
- Functions: `camelCase()`
- Classes: `PascalCase`
- Constants: `UPPER_CASE`
- Files: `kebab-case.php`

**Best Practices**
```php
// Good
$user_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);

// Bad
$userID = $_SESSION['user_id'];
$result = mysql_query("SELECT * FROM users WHERE id = $userID");
```

### JavaScript

**Naming Conventions**
- Variables: `camelCase`
- Constants: `UPPER_CASE`
- Functions: `camelCase()`
- Classes: `PascalCase`

**Best Practices**
```javascript
// Good
const userId = getUserId();
async function fetchAssessments() {
    const response = await fetch('/api/assessments.php');
    return await response.json();
}

// Bad
var user_id = getUserId();
function fetch_assessments() {
    $.ajax({ url: '/api/assessments.php' });
}
```

### CSS

**Naming Conventions**
- Classes: `kebab-case`
- IDs: `camelCase` (use sparingly)
- Variables: `--kebab-case`

**Best Practices**
```css
/* Good */
.card-header {
    background: var(--primary);
}

/* Bad */
#CardHeader {
    background: #667eea;
}
```

### SQL

**Naming Conventions**
- Tables: `snake_case`
- Columns: `snake_case`
- Indexes: `idx_table_column`
- Foreign Keys: `fk_table_reference`

**Best Practices**
```sql
-- Good
CREATE TABLE user_assessments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    assessment_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_user_id (user_id),
    FOREIGN KEY (user_id) REFERENCES users(id)
);

-- Bad
CREATE TABLE UserAssessments (
    ID int,
    UserId int,
    Date timestamp
);
```

## Security Guidelines

### Input Validation
```php
// Always validate and sanitize
$user_input = filter_input(INPUT_POST, 'field', FILTER_SANITIZE_STRING);

// Use prepared statements
$stmt = $pdo->prepare("INSERT INTO table (column) VALUES (?)");
$stmt->execute([$user_input]);

// Never trust user input
if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $error = "Invalid email";
}
```

### Output Escaping
```php
// Always escape output
echo sanitizeOutput($user_data);
echo htmlspecialchars($user_data, ENT_QUOTES, 'UTF-8');

// In HTML attributes
<input value="<?php echo htmlspecialchars($value); ?>">
```

### Authentication
```php
// Always check authentication
requireLogin();

// Always check permissions
requirePermission('admin_access');

// Never trust client-side validation
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}
```

## Error Handling

### PHP
```php
// Use try-catch for risky operations
try {
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
} catch (PDOException $e) {
    logDatabaseError($query, $e->getMessage(), $params);
    $error = "An error occurred. Please try again.";
}

// Log errors, don't expose details
error_log("Error in file.php: " . $e->getMessage());

// Show user-friendly messages
$error = "We couldn't process your request. Please try again.";
```

### JavaScript
```javascript
// Handle fetch errors
try {
    const response = await fetch(url);
    if (!response.ok) throw new Error('Network error');
    const data = await response.json();
} catch (error) {
    console.error('Error:', error);
    showError('Failed to load data');
}
```

## Database Guidelines

### Migrations
```sql
-- Always check before adding
ALTER TABLE locations 
ADD COLUMN IF NOT EXISTS new_field VARCHAR(100);

-- Always comment
-- Migration: Add email fields to locations
-- Date: 2026-01-08
ALTER TABLE locations ADD COLUMN report_emails TEXT 
COMMENT 'Comma-separated email addresses';
```

### Queries
```php
// Use indexes
CREATE INDEX idx_user_id ON assessments(user_id);

// Use JOINs efficiently
SELECT a.*, u.full_name 
FROM assessments a
LEFT JOIN users u ON a.user_id = u.id
WHERE a.status = 'active';

// Avoid N+1 queries
// Good: One query with JOIN
// Bad: Loop with query per iteration
```

## Testing Guidelines

### Before Commit
1. Run syntax check: `php -l file.php`
2. Test functionality manually
3. Check error logs
4. Verify database queries
5. Test permissions

### Before Deploy
1. Run `./check-system.sh`
2. Test on staging environment
3. Review database migrations
4. Check file permissions
5. Test backup/restore

## Git Workflow

### Commit Messages
```
Good:
- "Add email fields to locations table"
- "Fix login error handling"
- "Update action items report with filters"

Bad:
- "changes"
- "fix bug"
- "update"
```

### Branch Names
```
feature/email-automation
bugfix/login-error
hotfix/security-patch
```

## Documentation

### Code Comments
```php
/**
 * Calculate risk score based on likelihood and severity
 * 
 * @param int $likelihood Value from 1-5
 * @param int $severity Value from 1-5
 * @return int Risk score (1-25)
 */
function calculateRiskScore($likelihood, $severity) {
    return $likelihood * $severity;
}
```

### File Headers
```php
<?php
/**
 * Action Items Report
 * 
 * Displays pending action items from assessments with filtering
 * by location, status, and priority. Supports export to Excel.
 * 
 * @requires permissions.php (view_reports permission)
 * @uses action_items_report view
 */
```

## Performance

### Database
- Use indexes on frequently queried columns
- Limit result sets with LIMIT
- Use prepared statements (cached)
- Avoid SELECT * (specify columns)

### PHP
- Use output buffering for large pages
- Cache expensive calculations
- Minimize database queries
- Use isset() before accessing arrays

### Frontend
- Minimize HTTP requests
- Compress CSS/JS
- Use CDN for libraries
- Lazy load images

## Maintenance

### Regular Tasks
- Review error logs weekly
- Update dependencies monthly
- Backup database daily
- Test backups monthly
- Security audit quarterly

### Code Reviews
- Check for security issues
- Verify error handling
- Confirm proper validation
- Review SQL queries
- Test edge cases
EOF

echo -e "${GREEN}✓${NC} DEVELOPMENT_GUIDELINES.md created"

echo ""
echo "=========================================="
echo "  Cleanup Summary                        "
echo "=========================================="
echo ""
echo -e "Files moved to archive: ${YELLOW}$FILES_MOVED${NC}"
echo -e "Documentation organized: ${BLUE}3 files${NC}"
echo -e "Permissions set: ${GREEN}✓${NC}"
echo -e "Project structure: ${GREEN}✓${NC}"
echo ""
echo -e "${GREEN}✓ Code cleanup complete!${NC}"
echo ""
echo "Next steps:"
echo "1. Review archived files in .archive/"
echo "2. Delete .archive/ if not needed"
echo "3. Review docs/ for all guides"
echo "4. Run ./check-system.sh to verify"
echo "5. Review DEPLOYMENT_CHECKLIST.md before deploying"
echo ""
