#!/bin/bash

# ============================================================================
# Risk Assessment System - Code Checker
# Comprehensive validation and error checking
# ============================================================================

echo "=========================================="
echo "  Risk Assessment System Code Checker   "
echo "=========================================="
echo ""

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Counters
TOTAL_CHECKS=0
PASSED_CHECKS=0
FAILED_CHECKS=0
WARNING_CHECKS=0

# Check function
check() {
    TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
    if [ $1 -eq 0 ]; then
        echo -e "${GREEN}✓${NC} $2"
        PASSED_CHECKS=$((PASSED_CHECKS + 1))
        return 0
    else
        echo -e "${RED}✗${NC} $2"
        FAILED_CHECKS=$((FAILED_CHECKS + 1))
        return 1
    fi
}

warn() {
    TOTAL_CHECKS=$((TOTAL_CHECKS + 1))
    echo -e "${YELLOW}⚠${NC} $1"
    WARNING_CHECKS=$((WARNING_CHECKS + 1))
}

# Check if we're in the right directory
if [ ! -f "includes/config.php" ]; then
    echo -e "${RED}Error: Please run this script from the risk-system root directory${NC}"
    exit 1
fi

echo "Step 1: Checking File Structure"
echo "=================================="

# Check core directories
[ -d "includes" ] && check 0 "includes/ directory exists" || check 1 "includes/ directory missing"
[ -d "public" ] && check 0 "public/ directory exists" || check 1 "public/ directory missing"
[ -d "templates" ] && check 0 "templates/ directory exists" || check 1 "templates/ directory missing"
[ -d "uploads" ] && check 0 "uploads/ directory exists" || check 1 "uploads/ directory missing"
[ -d "logs" ] && check 0 "logs/ directory exists" || check 1 "logs/ directory missing"

echo ""
echo "Step 2: Checking Core PHP Files"
echo "=================================="

# Core includes
[ -f "includes/config.php" ] && check 0 "config.php exists" || check 1 "config.php missing"
[ -f "includes/permissions.php" ] && check 0 "permissions.php exists" || check 1 "permissions.php missing"
[ -f "includes/version_control.php" ] && check 0 "version_control.php exists" || check 1 "version_control.php missing"

# Core public files
CORE_FILES=(
    "public/login.php"
    "public/dashboard.php"
    "public/assessments.php"
    "public/locations.php"
    "public/users.php"
    "public/admin-panel.php"
    "public/reports.php"
    "public/action-items-report.php"
)

for file in "${CORE_FILES[@]}"; do
    if [ -f "$file" ]; then
        check 0 "$(basename $file) exists"
    else
        check 1 "$(basename $file) missing"
    fi
done

echo ""
echo "Step 3: Checking PHP Syntax"
echo "=================================="

SYNTAX_ERRORS=0
for file in $(find includes public -name "*.php" 2>/dev/null); do
    if command -v php &> /dev/null; then
        php -l "$file" > /dev/null 2>&1
        if [ $? -eq 0 ]; then
            check 0 "$(basename $file) - Syntax OK"
        else
            check 1 "$(basename $file) - Syntax Error"
            SYNTAX_ERRORS=$((SYNTAX_ERRORS + 1))
        fi
    else
        warn "PHP not available in PATH - cannot check syntax for $(basename $file)"
    fi
done

echo ""
echo "Step 4: Checking Database Schema Files"
echo "=================================="

SCHEMA_FILES=(
    "includes/complete_schema.sql"
    "includes/migration_permissions.sql"
    "includes/migration_version_control.sql"
    "includes/migration_audit_log.sql"
)

for file in "${SCHEMA_FILES[@]}"; do
    if [ -f "$file" ]; then
        check 0 "$(basename $file) exists"
        # Check for basic SQL syntax issues
        if grep -q "CREATE TABLE" "$file"; then
            check 0 "$(basename $file) contains CREATE TABLE statements"
        else
            warn "$(basename $file) may be incomplete"
        fi
    else
        warn "$(basename $file) missing (may be optional)"
    fi
done

echo ""
echo "Step 5: Checking Configuration"
echo "=================================="

# Check config.php for common issues
if [ -f "includes/config.php" ]; then
    if grep -q "localhost" "includes/config.php"; then
        warn "Database host is 'localhost' - verify this is correct"
    fi
    
    if grep -q "password.*=.*['\"].*['\"]" "includes/config.php"; then
        check 0 "Database password is set"
    else
        warn "Database password may be empty"
    fi
fi

echo ""
echo "Step 6: Checking Permissions"
echo "=================================="

# Check directory permissions
if [ -w "uploads" ]; then
    check 0 "uploads/ is writable"
else
    check 1 "uploads/ is not writable"
fi

if [ -w "logs" ]; then
    check 0 "logs/ is writable"
else
    check 1 "logs/ is not writable"
fi

echo ""
echo "Step 7: Checking for Common Issues"
echo "=================================="

# Check for password_hash column references (should be 'password')
HASH_REFS=$(grep -r "password_hash" --include="*.php" includes/ public/ 2>/dev/null | grep -E "(SET|INSERT|SELECT|UPDATE)" | grep -v "password_hash(" | wc -l)
if [ "$HASH_REFS" -eq 0 ]; then
    check 0 "No incorrect 'password_hash' column references"
else
    check 1 "Found $HASH_REFS incorrect 'password_hash' column references"
fi

# Check for hardcoded credentials
CRED_COUNT=$(grep -r -i "password.*=.*['\"]Admin123" --include="*.php" public/ includes/ 2>/dev/null | grep -v "test-password\|reset-admin\|debug-login" | wc -l)
if [ "$CRED_COUNT" -eq 0 ]; then
    check 0 "No hardcoded credentials in production files"
else
    warn "Found $CRED_COUNT possible hardcoded credentials"
fi

# Check for SQL injection vulnerabilities (basic check)
UNSAFE_SQL=$(grep -r "\$_GET\|\$_POST\|\$_REQUEST" --include="*.php" includes/ public/ 2>/dev/null | grep "query\|execute" | grep -v "prepare" | wc -l)
if [ "$UNSAFE_SQL" -eq 0 ]; then
    check 0 "No obvious SQL injection vulnerabilities"
else
    warn "Found $UNSAFE_SQL potential SQL injection points (review needed)"
fi

echo ""
echo "Step 8: Checking Template Files"
echo "=================================="

[ -f "templates/header.php" ] && check 0 "header.php exists" || check 1 "header.php missing"
[ -f "templates/footer.php" ] && check 0 "footer.php exists" || check 1 "footer.php missing"

echo ""
echo "Step 9: Checking Documentation"
echo "=================================="

DOC_FILES=(
    "README.md"
    "LOGIN_TROUBLESHOOTING.md"
    "ACTION_ITEMS_REPORT_GUIDE.md"
)

for file in "${DOC_FILES[@]}"; do
    [ -f "$file" ] && check 0 "$file exists" || warn "$file missing"
done

echo ""
echo "Step 10: Security Checks"
echo "=================================="

# Check for debug files that should be deleted
DEBUG_FILES=(
    "public/test-password.php"
    "public/reset-admin.php"
    "public/debug-login.php"
    "public/diagnose-admin-panel.php"
)

for file in "${DEBUG_FILES[@]}"; do
    if [ -f "$file" ]; then
        warn "Debug file exists: $file (should be deleted in production)"
    else
        check 0 "$(basename $file) not present (good for production)"
    fi
done

# Check for exposed .git directory
if [ -d ".git" ]; then
    warn ".git directory exists (should not be deployed to production)"
else
    check 0 "No .git directory (production ready)"
fi

echo ""
echo "=========================================="
echo "  Code Check Summary                     "
echo "=========================================="
echo ""
echo -e "Total Checks: ${BLUE}$TOTAL_CHECKS${NC}"
echo -e "Passed: ${GREEN}$PASSED_CHECKS${NC}"
echo -e "Failed: ${RED}$FAILED_CHECKS${NC}"
echo -e "Warnings: ${YELLOW}$WARNING_CHECKS${NC}"
echo ""

if [ $FAILED_CHECKS -eq 0 ]; then
    echo -e "${GREEN}✓ All critical checks passed!${NC}"
    exit 0
else
    echo -e "${RED}✗ $FAILED_CHECKS critical issues found${NC}"
    echo "Please review and fix the issues above."
    exit 1
fi
