#!/bin/bash

# ============================================================================
# Header and Footer Reference Checker
# Verifies all PHP files have correct includes
# ============================================================================

echo "========================================"
echo "  Header/Footer Reference Checker      "
echo "========================================"
echo ""

cd "$(dirname "$0")"

TOTAL_FILES=0
GOOD_FILES=0
ISSUES_FOUND=0

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo "Checking PHP files in public/ directory..."
echo ""

# Find all PHP files in public (excluding api directory)
while IFS= read -r file; do
    TOTAL_FILES=$((TOTAL_FILES + 1))
    filename=$(basename "$file")
    
    # Skip API files and special files
    if [[ "$file" == *"/api/"* ]] || [[ "$filename" == "login.php" ]] || [[ "$filename" == "logout.php" ]]; then
        continue
    fi
    
    HAS_HEADER=0
    HAS_FOOTER=0
    HEADER_TYPE=""
    
    # Check for header include
    if grep -q "include.*header\.php" "$file"; then
        HAS_HEADER=1
        HEADER_TYPE="header.php"
    elif grep -q "include.*header-responsive\.php" "$file"; then
        HAS_HEADER=1
        HEADER_TYPE="header-responsive.php"
    fi
    
    # Check for footer include
    if grep -q "include.*footer\.php" "$file" || grep -q "include.*footer-responsive\.php" "$file"; then
        HAS_FOOTER=1
    fi
    
    # Report status
    if [ $HAS_HEADER -eq 1 ] && [ $HAS_FOOTER -eq 1 ]; then
        echo -e "${GREEN}✓${NC} $filename - ${HEADER_TYPE}"
        GOOD_FILES=$((GOOD_FILES + 1))
    else
        echo -e "${RED}✗${NC} $filename"
        ISSUES_FOUND=$((ISSUES_FOUND + 1))
        
        if [ $HAS_HEADER -eq 0 ]; then
            echo -e "  ${YELLOW}→${NC} Missing header include"
        fi
        if [ $HAS_FOOTER -eq 0 ]; then
            echo -e "  ${YELLOW}→${NC} Missing footer include"
        fi
    fi
    
done < <(find public -name "*.php" -type f | grep -v "/api/" | sort)

echo ""
echo "========================================"
echo "  Summary                               "
echo "========================================"
echo ""
echo "Total PHP files checked: $TOTAL_FILES"
echo -e "Files with correct includes: ${GREEN}$GOOD_FILES${NC}"
echo -e "Files with issues: ${RED}$ISSUES_FOUND${NC}"
echo ""

# Check template files exist
echo "Checking template files..."
echo ""

if [ -f "templates/header.php" ]; then
    # Check if header has DOCTYPE
    if grep -q "<!DOCTYPE html>" "templates/header.php"; then
        echo -e "${GREEN}✓${NC} templates/header.php exists and has DOCTYPE"
    else
        echo -e "${YELLOW}⚠${NC} templates/header.php exists but missing DOCTYPE"
    fi
else
    echo -e "${RED}✗${NC} templates/header.php NOT FOUND"
fi

if [ -f "templates/header-responsive.php" ]; then
    if grep -q "<!DOCTYPE html>" "templates/header-responsive.php"; then
        echo -e "${GREEN}✓${NC} templates/header-responsive.php exists and has DOCTYPE"
    else
        echo -e "${YELLOW}⚠${NC} templates/header-responsive.php exists but missing DOCTYPE"
    fi
else
    echo -e "${YELLOW}⚠${NC} templates/header-responsive.php not found"
fi

if [ -f "templates/footer.php" ]; then
    echo -e "${GREEN}✓${NC} templates/footer.php exists"
else
    echo -e "${RED}✗${NC} templates/footer.php NOT FOUND"
fi

if [ -f "templates/footer-responsive.php" ]; then
    echo -e "${GREEN}✓${NC} templates/footer-responsive.php exists"
else
    echo -e "${YELLOW}⚠${NC} templates/footer-responsive.php not found"
fi

echo ""

# Check CSS files
echo "Checking CSS files..."
echo ""

if [ -f "public/css/responsive-framework.css" ]; then
    SIZE=$(du -h "public/css/responsive-framework.css" | cut -f1)
    echo -e "${GREEN}✓${NC} public/css/responsive-framework.css exists ($SIZE)"
else
    echo -e "${RED}✗${NC} public/css/responsive-framework.css NOT FOUND"
fi

if [ -f "public/css/style.css" ]; then
    SIZE=$(du -h "public/css/style.css" | cut -f1)
    echo -e "${GREEN}✓${NC} public/css/style.css exists ($SIZE)"
else
    echo -e "${YELLOW}⚠${NC} public/css/style.css not found (optional)"
fi

echo ""
echo "========================================"

if [ $ISSUES_FOUND -eq 0 ]; then
    echo -e "${GREEN}✓ All files have correct header/footer includes!${NC}"
else
    echo -e "${YELLOW}⚠ Found $ISSUES_FOUND file(s) with issues${NC}"
    echo ""
    echo "To fix manually:"
    echo "1. Add header: include '../templates/header.php';"
    echo "2. Add footer: include '../templates/footer.php';"
fi

echo ""
