# TROUBLESHOOTING GUIDE - Database Errors

## 🚨 Common Database Errors & Fixes

---

## Error 1: "Column not found: 1054 Unknown column 'entity_type'"

### **Problem:**
The `custom_fields` table doesn't exist or is missing columns.

### **Quick Fix:**

```bash
# Run the quick fix patch
mysql -u root -p scubatricky_risk < quick-fix-missing-tables.sql
```

### **Alternative Fix:**

1. Open the Custom Fields page - it will auto-create the table
2. Or run complete schema: `mysql -u root -p scubatricky_risk < schema.sql`

### **Manual Fix (phpMyAdmin):**

1. Open phpMyAdmin
2. Select `scubatricky_risk` database
3. Go to SQL tab
4. Copy and paste from `quick-fix-missing-tables.sql`
5. Click "Go"

---

## Error 2: "Table 'scubatricky_risk.system_files' doesn't exist"

### **Problem:**
Version control tables are missing.

### **Quick Fix:**

```bash
mysql -u root -p scubatricky_risk < quick-fix-missing-tables.sql
```

### **What it creates:**
- `system_files` - File tracking
- `file_versions` - Version history

---

## Error 3: "Table 'scubatricky_risk.permissions' doesn't exist"

### **Problem:**
Permission management table is missing.

### **Quick Fix:**

```bash
mysql -u root -p scubatricky_risk < quick-fix-missing-tables.sql
```

### **What it creates:**
- `permissions` table with default role permissions

---

## Error 4: "Cannot redeclare getSystemName()"

### **Problem:**
Function declared multiple times.

### **Status:**
✅ **ALREADY FIXED** in latest version

Both header files now have `function_exists()` checks:
```php
if (!function_exists('getSystemName')) {
    function getSystemName($pdo) {
        // ...
    }
}
```

### **If still happening:**
1. Clear browser cache (Ctrl + Shift + Delete)
2. Re-upload latest header.php files
3. Check no custom modifications

---

## Error 5: "Column 'report_emails' doesn't exist in table 'locations'"

### **Problem:**
Locations table is missing new email columns.

### **Quick Fix:**

```bash
mysql -u root -p scubatricky_risk < quick-fix-missing-tables.sql
```

### **Manual Fix:**

```sql
ALTER TABLE locations 
ADD COLUMN report_emails TEXT,
ADD COLUMN email_frequency ENUM('immediate', 'daily', 'weekly', 'monthly', 'none') DEFAULT 'none',
ADD COLUMN last_report_sent TIMESTAMP NULL,
ADD COLUMN email_enabled BOOLEAN DEFAULT FALSE;
```

---

## Error 6: General "Table doesn't exist" errors

### **Root Cause:**
Database schema incomplete or outdated.

### **Best Fix - Use Complete Schema:**

```bash
# Backup existing data first!
mysqldump -u root -p scubatricky_risk > backup.sql

# Import complete schema
mysql -u root -p scubatricky_risk < schema.sql
```

**Note:** This is safe - it uses `CREATE TABLE IF NOT EXISTS`

### **Alternative - Quick Patch Only:**

```bash
mysql -u root -p scubatricky_risk < quick-fix-missing-tables.sql
```

---

## 🔍 **Diagnostic Commands**

### **Check which tables exist:**

```sql
SHOW TABLES;
```

**Expected tables (16):**
- users
- locations
- categories
- risk_items
- assessments
- assessment_details
- action_items
- permissions
- system_settings
- custom_fields
- custom_field_values
- system_files
- file_versions
- audit_log
- email_queue
- sessions

### **Check specific table structure:**

```sql
DESCRIBE custom_fields;
DESCRIBE system_files;
DESCRIBE permissions;
```

### **Check for missing columns:**

```sql
SHOW COLUMNS FROM locations LIKE 'report_emails';
SHOW COLUMNS FROM custom_fields LIKE 'entity_type';
```

---

## 📋 **Complete Schema Check**

Run this to see what's missing:

```sql
-- Check all required tables
SELECT 
    'users' as required_table,
    CASE WHEN COUNT(*) > 0 THEN '✓' ELSE '✗' END as exists
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'scubatricky_risk' AND TABLE_NAME = 'users'
UNION ALL
SELECT 'custom_fields', CASE WHEN COUNT(*) > 0 THEN '✓' ELSE '✗' END
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'scubatricky_risk' AND TABLE_NAME = 'custom_fields'
UNION ALL
SELECT 'system_files', CASE WHEN COUNT(*) > 0 THEN '✓' ELSE '✗' END
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'scubatricky_risk' AND TABLE_NAME = 'system_files'
UNION ALL
SELECT 'permissions', CASE WHEN COUNT(*) > 0 THEN '✓' ELSE '✗' END
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'scubatricky_risk' AND TABLE_NAME = 'permissions';
```

---

## 🎯 **Quick Decision Tree**

### **Missing ONE table?**
→ Run: `quick-fix-missing-tables.sql`

### **Missing MULTIPLE tables?**
→ Run: `schema.sql` (complete)

### **All tables exist but errors persist?**
→ Check column names with `DESCRIBE table_name`

### **Fresh installation?**
→ Run: `schema.sql` (complete schema)

---

## 🛠️ **Fix Application Flow**

### **Method 1: Quick Patch (5 seconds)**

```bash
cd risk-system
mysql -u root -p scubatricky_risk < quick-fix-missing-tables.sql
```

**Fixes:**
- ✅ custom_fields tables
- ✅ system_files tables
- ✅ permissions table
- ✅ system_settings table
- ✅ Missing location columns

### **Method 2: Complete Schema (30 seconds)**

```bash
cd risk-system
mysql -u root -p scubatricky_risk < schema.sql
```

**Creates:**
- ✅ All 16 tables
- ✅ All views
- ✅ All triggers
- ✅ All default data
- ✅ All indexes

**Safe:** Uses `IF NOT EXISTS` - won't delete data

### **Method 3: Manual (phpMyAdmin)**

1. Login to phpMyAdmin
2. Select database
3. Go to "SQL" tab
4. Choose method:
   - **Quick:** Import `quick-fix-missing-tables.sql`
   - **Complete:** Import `schema.sql`
5. Click "Go"

---

## ✅ **Verification After Fix**

### **Test 1: Check Tables**

```sql
SELECT COUNT(*) as table_count 
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'scubatricky_risk';
```

**Should return:** 16

### **Test 2: Access Custom Fields**

1. Login to application
2. Go to Admin Panel
3. Click "Custom Fields"
4. Should load without errors

### **Test 3: Check Version Control**

1. Admin Panel → Version Control
2. Should display file list (or empty state)
3. No table errors

### **Test 4: Check Permissions**

1. Admin Panel → Permission Management
2. Should show role permissions
3. No errors

---

## 🚨 **Still Having Issues?**

### **Problem: Fixes don't apply**

**Check database user permissions:**

```sql
SHOW GRANTS FOR 'your_username'@'localhost';
```

**Should have:**
- SELECT, INSERT, UPDATE, DELETE
- CREATE, ALTER, DROP
- INDEX, REFERENCES

### **Problem: Foreign key errors**

**Temporarily disable:**

```sql
SET FOREIGN_KEY_CHECKS = 0;
SOURCE quick-fix-missing-tables.sql;
SET FOREIGN_KEY_CHECKS = 1;
```

### **Problem: Permission denied**

**Check database credentials in config.php:**

```php
define('DB_USER', 'correct_username');
define('DB_PASS', 'correct_password');
define('DB_NAME', 'scubatricky_risk');
```

---

## 📞 **Get More Help**

### **Check error logs:**

```bash
tail -50 logs/system-errors.log
tail -50 logs/debug.log
```

### **Check PHP errors (XAMPP):**

```bash
tail -50 C:\xampp\php\logs\php_error_log
```

### **Enable detailed errors:**

Add to top of problematic PHP file:

```php
ini_set('display_errors', 1);
error_reporting(E_ALL);
```

---

## 📚 **Summary of Fixes**

| Error | Fix File | Time |
|-------|----------|------|
| Missing tables | quick-fix-missing-tables.sql | 5 sec |
| Incomplete schema | schema.sql | 30 sec |
| Missing columns | quick-fix-missing-tables.sql | 5 sec |
| Function redeclare | ✅ Already fixed | 0 sec |

---

## ✨ **Prevention**

### **For fresh installations:**

```bash
# Always use complete schema
mysql -u root -p scubatricky_risk < schema.sql
```

### **For updates:**

```bash
# Run quick fix after updates
mysql -u root -p scubatricky_risk < quick-fix-missing-tables.sql
```

### **Regular checks:**

```bash
# Verify table count
mysql -u root -p scubatricky_risk -e "SELECT COUNT(*) FROM information_schema.TABLES WHERE TABLE_SCHEMA = 'scubatricky_risk'"
```

Should return: 16

---

**Most errors fixed in under 30 seconds with the quick fix patch!** ✅
