# Risk Item Photos Feature

## Overview

The Risk Item Photos feature allows users to attach photos to individual risk items within an assessment, providing visual evidence and documentation for each specific hazard identified.

## Key Features

### 📸 **Photo Attachment to Individual Risks**
- Upload photos specifically for each risk item
- Multiple photos per risk item
- Separate from general assessment photos
- Visual evidence for each hazard

### 🎯 **Organized Photo Management**
- Photos grouped by risk item
- Easy upload interface
- View, caption, and delete functionality
- Automatic organization

### 🔒 **Access Control**
- Only assessors and admins can upload
- Photo ownership tracking
- Delete permissions (owner or admin)
- Audit trail

### 💾 **Smart Storage**
- Organized file structure
- Automatic directory creation
- File size limits (5MB)
- Multiple format support

## Installation

### Step 1: Run Migration

```bash
# For existing installations
mysql -u scubatricky_risk -p scubatricky_risk < includes/migration_risk_item_photos.sql

# For new installations, table is already in complete_schema.sql
```

### Step 2: Create Upload Directory

```bash
# Create risk items photos directory
mkdir -p public/uploads/risk-items
chmod 755 public/uploads/risk-items
chown www-data:www-data public/uploads/risk-items
```

### Step 3: Test Feature

```
Navigate to any assessment
→ View Assessment (Enhanced)
→ Find any risk item
→ Click "Add Photo"
→ Upload photo
```

## Usage

### Uploading Photos to Risk Items

**Method 1: During Assessment Creation**
1. Create new assessment
2. Add risk items with evaluations
3. Save assessment
4. View assessment
5. For each risk item, click "Add Photo"
6. Select photo and add caption
7. Click "Upload Photo"

**Method 2: After Assessment Created**
1. Open existing assessment
2. Navigate to specific risk item
3. Click "📷 Add Photo" button
4. Upload photo with optional caption
5. Photo appears immediately

### Viewing Risk Item Photos

1. Open assessment
2. Scroll to risk items section
3. Photos display under each risk item
4. Click photo to view full size
5. See caption and upload info

### Managing Photos

**Edit Caption:**
- Click on photo caption
- Edit text
- Save changes

**Delete Photo:**
- Click 🗑️ delete button
- Confirm deletion
- Photo removed immediately

**Reorder Photos:**
- Drag and drop (coming soon)
- Or use display order in database

## File Organization

### Directory Structure

```
public/uploads/risk-items/
├── {assessment_id}/
│   ├── {risk_item_id}/
│   │   ├── risk_12345_1234567890.jpg
│   │   ├── risk_12346_1234567891.png
│   │   └── risk_12347_1234567892.jpg
│   ├── {risk_item_id}/
│   │   └── risk_12348_1234567893.jpg
├── {assessment_id}/
│   └── {risk_item_id}/
│       └── risk_12349_1234567894.jpg
```

**Benefits:**
- ✅ Easy to find photos
- ✅ Logical organization
- ✅ Assessment isolation
- ✅ Risk item grouping
- ✅ Automatic cleanup on delete

### Filename Format

```
risk_{unique_id}_{timestamp}.{extension}
```

**Example:**
```
risk_6789abc_1704643200.jpg
```

## Database Schema

### Table: `risk_item_photos`

```sql
CREATE TABLE risk_item_photos (
    id INT AUTO_INCREMENT PRIMARY KEY,
    assessment_detail_id INT NOT NULL,       -- Link to specific assessment detail
    assessment_id INT NOT NULL,              -- Parent assessment
    risk_item_id INT NOT NULL,               -- Which risk item
    file_path VARCHAR(255) NOT NULL,         -- uploads/risk-items/...
    file_name VARCHAR(255) NOT NULL,         -- Original filename
    file_size INT,                           -- File size in bytes
    mime_type VARCHAR(50),                   -- image/jpeg, etc.
    caption TEXT,                            -- Photo description
    uploaded_by INT NOT NULL,                -- User who uploaded
    display_order INT DEFAULT 0,             -- Sort order
    uploaded_at TIMESTAMP,                   -- Upload timestamp
    
    FOREIGN KEY (assessment_detail_id) REFERENCES assessment_details(id),
    FOREIGN KEY (assessment_id) REFERENCES assessments(id),
    FOREIGN KEY (risk_item_id) REFERENCES risk_items(id),
    FOREIGN KEY (uploaded_by) REFERENCES users(id)
);
```

### Relationships

```
assessments (1) ─── (many) assessment_details
     │                           │
     │                           │
     │                    (many) risk_item_photos
     │                           │
     └──────────────────────────┘
```

## API Endpoints

### Upload Photo

**Endpoint:** `POST /api/risk-item-photos.php`

**Parameters:**
```
photo: File (required)
assessment_id: Integer (required)
risk_item_id: Integer (required)
assessment_detail_id: Integer (optional)
caption: String (optional)
```

**Response:**
```json
{
    "success": true,
    "photo_id": 123,
    "filename": "risk_abc_1234567890.jpg",
    "path": "uploads/risk-items/45/12/risk_abc_1234567890.jpg",
    "message": "Photo uploaded successfully"
}
```

### Get Photos for Risk Item

**Endpoint:** `GET /api/risk-item-photos.php`

**Parameters:**
```
assessment_id: Integer (required)
risk_item_id: Integer (required)
```

**Response:**
```json
{
    "success": true,
    "photos": [
        {
            "id": 123,
            "file_path": "uploads/risk-items/45/12/...",
            "caption": "Wet floor hazard",
            "uploaded_at": "2026-01-07 10:30:00",
            "uploader_name": "John Smith"
        }
    ]
}
```

### Delete Photo

**Endpoint:** `DELETE /api/risk-item-photos.php?id={photo_id}`

**Response:**
```json
{
    "success": true,
    "message": "Photo deleted successfully"
}
```

### Update Caption

**Endpoint:** `PUT /api/risk-item-photos.php`

**Parameters:**
```
id: Integer (required)
caption: String (required)
```

**Response:**
```json
{
    "success": true,
    "message": "Caption updated successfully"
}
```

## Vue Component

### Component: `RiskItemPhotos`

**Props:**
```javascript
{
    assessmentId: Number,      // Required
    riskItemId: Number,        // Required
    assessmentDetailId: Number,// Optional
    editable: Boolean          // Default: true
}
```

**Usage:**
```html
<risk-item-photos
    :assessment-id="45"
    :risk-item-id="12"
    :assessment-detail-id="89"
    :editable="true"
></risk-item-photos>
```

**Events:**
```javascript
@photo-uploaded="handleUpload"
@photo-deleted="handleDelete"
```

**Features:**
- ✅ Photo upload interface
- ✅ Grid display of photos
- ✅ Lightbox view (click to enlarge)
- ✅ Caption editing
- ✅ Delete functionality
- ✅ Loading states
- ✅ Empty states

## Use Cases

### 1. Vehicle Damage Documentation
**Scenario:** Assessing delivery vehicle risks

**Photos to Attach:**
- Narrow entrance/exit → Photo of tight access
- Steps/stairs → Photo of steps
- Poor lighting → Photo of dark area
- Obstructions → Photo of blocked path

**Benefits:**
- Visual evidence of hazards
- Support for risk ratings
- Clear communication to drivers
- Justification for controls

### 2. Loading Bay Hazards
**Scenario:** Warehouse delivery assessment

**Photos to Attach:**
- Uneven ground → Photo of potholes
- Limited visibility → Photo of blind spots
- Equipment damage → Photo of broken equipment
- Weather conditions → Photo of ice/water

**Benefits:**
- Document current conditions
- Track hazard changes over time
- Support risk assessments
- Evidence for improvements

### 3. Manual Handling Risks
**Scenario:** Heavy items delivery

**Photos to Attach:**
- Heavy items → Photo of large packages
- Awkward shapes → Photo of difficult items
- Long carry distance → Photo of delivery route
- Poor access → Photo of narrow corridors

**Benefits:**
- Visual proof of risk factors
- Training materials
- Equipment justification
- Route planning

### 4. Environmental Hazards
**Scenario:** Outdoor delivery location

**Photos to Attach:**
- Wet surfaces → Photo of puddles
- Icy conditions → Photo of ice
- Extreme weather → Photo of conditions
- Poor visibility → Photo of fog/darkness

**Benefits:**
- Weather documentation
- Seasonal risk tracking
- Pattern identification
- Control effectiveness

## Best Practices

### Photo Quality:
- ✅ Clear, focused images
- ✅ Good lighting
- ✅ Show full context
- ✅ Include reference objects for scale
- ✅ Multiple angles if needed

### Captions:
- ✅ Descriptive and specific
- ✅ Mention location within site
- ✅ Note date/time if relevant
- ✅ Highlight specific hazard
- ✅ Keep concise but informative

### Organization:
- ✅ One photo per specific hazard
- ✅ Attach to correct risk item
- ✅ Delete duplicate/poor photos
- ✅ Update captions as needed
- ✅ Review photos before finalizing

### Security:
- ✅ No people's faces (GDPR)
- ✅ No sensitive information
- ✅ No vehicle license plates
- ✅ No company confidential data
- ✅ Follow privacy policies

## Comparison: General vs Risk Item Photos

### General Assessment Photos
**Use for:**
- Overall site overview
- General conditions
- Location context
- Site layout
- Multiple hazards in one image

**Location:** Main photos section

### Risk Item Photos
**Use for:**
- Specific hazard evidence
- Individual risk documentation
- Detailed hazard views
- Risk-specific context
- Supporting individual ratings

**Location:** Under each risk item

### When to Use Which:

| Scenario | General Photos | Risk Item Photos |
|----------|---------------|------------------|
| Site entrance | ✅ | ❌ |
| Loading bay overview | ✅ | ❌ |
| Specific pothole | ❌ | ✅ |
| Narrow doorway | ❌ | ✅ |
| Weather conditions | ✅ | Maybe ✅ |
| Equipment damage | ❌ | ✅ |
| Site map/signage | ✅ | ❌ |
| Specific hazard detail | ❌ | ✅ |

## Troubleshooting

### Issue: Photo Upload Fails

**Causes:**
- File too large (>5MB)
- Wrong file type
- Directory permissions
- Disk space

**Fix:**
```bash
# Check directory permissions
ls -la public/uploads/risk-items/
chmod 755 public/uploads/risk-items/

# Check disk space
df -h

# Check PHP upload limits
php -i | grep upload_max_filesize
php -i | grep post_max_size

# Increase limits in php.ini:
upload_max_filesize = 10M
post_max_size = 10M
```

### Issue: Photos Not Displaying

**Causes:**
- Wrong file path
- File deleted
- Permissions issue
- Database mismatch

**Fix:**
```bash
# Check files exist
ls public/uploads/risk-items/{assessment_id}/{risk_item_id}/

# Fix permissions
chmod 644 public/uploads/risk-items/*/*/*

# Check database
mysql -u scubatricky_risk -p scubatricky_risk -e \
  "SELECT * FROM risk_item_photos WHERE assessment_id = X;"
```

### Issue: Cannot Delete Photos

**Causes:**
- Permission denied (not owner/admin)
- File already deleted
- Database lock

**Fix:**
```sql
-- Check ownership
SELECT * FROM risk_item_photos WHERE id = X;

-- Admin can delete via SQL if needed
DELETE FROM risk_item_photos WHERE id = X;

-- Then delete file
rm public/uploads/risk-items/.../filename.jpg
```

### Issue: Duplicate Photos

**Causes:**
- Multiple uploads
- Browser back button
- Slow connection

**Prevention:**
- Disable button while uploading
- Show upload progress
- Clear form after success

**Fix:**
```sql
-- Find duplicates
SELECT file_path, COUNT(*) 
FROM risk_item_photos 
GROUP BY file_path 
HAVING COUNT(*) > 1;

-- Delete duplicates (keep one)
DELETE FROM risk_item_photos 
WHERE id NOT IN (
    SELECT MIN(id) FROM risk_item_photos GROUP BY file_path
);
```

## Performance

### File Size Limits
- Maximum: 5MB per photo
- Recommended: 1-2MB
- Optimize before upload
- Use JPEG for photos

### Storage Considerations
- 100 assessments × 10 photos/assessment × 2MB = ~2GB
- Monitor disk space
- Implement cleanup for old assessments
- Consider cloud storage for large deployments

### Loading Optimization
- Lazy load photos (load when visible)
- Thumbnail generation (future enhancement)
- Progressive image loading
- Optimize image format

## Future Enhancements

### Planned Features:
- [ ] Thumbnail generation
- [ ] Image optimization on upload
- [ ] Drag-and-drop photo ordering
- [ ] Bulk photo upload
- [ ] Photo annotations/markup
- [ ] Photo comparison (before/after)
- [ ] Export photos with PDF report
- [ ] Photo library/search
- [ ] Cloud storage integration
- [ ] Mobile camera integration

### Possible Improvements:
- Image editing tools
- Automatic hazard detection (AI)
- Photo geolocation
- Photo timestamps
- Photo watermarking
- Photo analytics

## Summary

**Features:**
- ✅ Upload photos to individual risk items
- ✅ Multiple photos per risk
- ✅ Caption and metadata
- ✅ Easy management
- ✅ Organized storage
- ✅ Access control
- ✅ Vue component
- ✅ REST API

**Benefits:**
- Better risk documentation
- Visual evidence
- Improved assessments
- Training materials
- Audit trail
- Professional reports

**Requirements:**
- risk_item_photos table
- uploads/risk-items directory
- Vue.js (for component)
- Write permissions

**Files:**
- migration_risk_item_photos.sql (database)
- api/risk-item-photos.php (API)
- js/risk-item-photos.js (Vue component)
- view-assessment-enhanced.php (UI)

---

**Document your risks with photos for better assessments!** 📸✨
