# Role-Based Access Control (RBAC) Guide

## Overview

The RBAC system provides granular control over who can access which features in the Risk Assessment System. Administrators can manage permissions at both the role level and individual user level.

## Key Concepts

### 1. Roles
Three default roles with predefined permission sets:
- **Admin** - Full system access
- **Assessor** - Standard assessment workflow
- **Viewer** - Read-only access

### 2. Permissions
Fine-grained access controls for specific features:
- **assessment.view** - View all assessments
- **assessment.create** - Create new assessments
- **location.edit** - Edit location details
- **user.manage_permissions** - Grant/revoke permissions
- And 30+ more...

### 3. Permission Inheritance
```
Role Permissions (Default for all users with that role)
         ↓
User-Specific Overrides (Grant/Revoke for individual users)
         ↓
Final Permission Check
```

### 4. Feature Toggles
System-wide on/off switches for entire features:
- **assessments_enabled** - Enable/disable assessments
- **pdf_export_enabled** - Enable/disable PDF generation
- **photo_uploads_enabled** - Enable/disable photo uploads
- And more...

## Installation

### Step 1: Run Migration

```bash
# For existing installations
mysql -u scubatricky_risk -p scubatricky_risk < includes/migration_rbac.sql

# For new installations, RBAC is already in complete_schema.sql
```

### Step 2: Verify Permissions

```bash
# Check tables created
mysql -u scubatricky_risk -p scubatricky_risk -e "SHOW TABLES LIKE '%permission%';"

# Should show:
# - permissions
# - role_permissions
# - user_permissions
# - feature_toggles
```

### Step 3: Test Access

```
Login as Admin → Admin Panel → Permissions Management
```

## Permission Categories

### Assessments (9 permissions)
```
assessment.view              - View all assessments
assessment.view_own          - View only own assessments
assessment.create            - Create new assessments
assessment.edit              - Edit any assessment
assessment.edit_own          - Edit own assessments only
assessment.delete            - Delete assessments
assessment.export            - Export to PDF/Excel
assessment.photos            - Manage general photos
assessment.risk_photos       - Manage risk item photos
```

### Locations (5 permissions)
```
location.view                - View locations
location.create              - Add new locations
location.edit                - Edit locations
location.delete              - Remove locations
location.documents           - Manage planning docs
location.custom_fields       - Edit custom fields
```

### Users (5 permissions)
```
user.view                    - View user list
user.create                  - Create user accounts
user.edit                    - Edit user accounts
user.delete                  - Delete user accounts
user.change_role             - Change user roles
user.manage_permissions      - Grant/revoke permissions
```

### Reports (3 permissions)
```
report.view                  - View reports
report.export                - Export report data
report.advanced              - Advanced analytics
```

### Admin (7 permissions)
```
admin.panel                  - Access admin panel
admin.risk_items             - Manage risk items
admin.vehicle_types          - Manage vehicle types
admin.custom_fields          - Configure custom fields
admin.pdf_templates          - Manage PDF templates
admin.settings               - System settings
admin.customization          - Colors & logo
```

### System (4 permissions)
```
system.version_control       - Create revisions
system.archive               - Archive assessments
system.audit_log             - View audit logs
system.backup                - Database backup
```

## Default Role Permissions

### Admin Role
✅ **All permissions** (cannot be modified)

### Assessor Role
✅ assessment.view
✅ assessment.view_own
✅ assessment.create
✅ assessment.edit_own
✅ assessment.export
✅ assessment.photos
✅ assessment.risk_photos
✅ location.view
✅ location.custom_fields
✅ location.documents
✅ report.view

### Viewer Role
✅ assessment.view
✅ assessment.view_own
✅ location.view
✅ report.view

## Using Permissions in Code

### Check Permission

```php
// Check if current user has permission
if (hasPermission($pdo, 'assessment.create')) {
    // Show create button
}

// Check for specific user
if (hasPermission($pdo, 'user.edit', $user_id)) {
    // Allow editing
}
```

### Require Permission

```php
// At top of page - redirect if no permission
requirePermission($pdo, 'admin.panel');

// Custom redirect
requirePermission($pdo, 'report.advanced', 'reports.php');
```

### Check Multiple Permissions

```php
// User needs ANY of these permissions
if (hasAnyPermission($pdo, ['assessment.edit', 'assessment.edit_own'])) {
    // Show edit button
}

// User needs ALL of these permissions
if (hasAllPermissions($pdo, ['user.edit', 'user.change_role'])) {
    // Allow role change
}
```

### Check Feature Enabled

```php
// Check if feature is enabled system-wide
if (isFeatureEnabled($pdo, 'pdf_export_enabled')) {
    // Show PDF export button
}

// Require feature or redirect
requireFeature($pdo, 'version_control_enabled');
```

### Combined Check

```php
// Check both permission AND feature
if (hasPermission($pdo, 'assessment.export') && 
    isFeatureEnabled($pdo, 'pdf_export_enabled')) {
    // Show PDF button
}
```

## Managing Permissions

### Via Admin Interface

**Access:** Admin Panel → Permissions Management

#### Role Permissions Tab
1. Select role (Admin/Assessor/Viewer)
2. Check/uncheck permissions
3. Click "Save Permissions"
4. Changes apply to all users with that role

#### User Permissions Tab
1. Find user in list
2. Click "Manage Permissions"
3. Grant or revoke specific permissions
4. Optional: Set expiry date
5. Changes override role permissions

#### Feature Toggles Tab
1. Toggle switches to enable/disable features
2. Changes apply system-wide immediately
3. Disabled features hidden from all users

### Via Code

#### Grant Permission to User

```php
// Grant permission
grantPermission($pdo, $user_id, 'report.advanced', $granted_by_user_id);

// Grant with expiry
$expiry = date('Y-m-d H:i:s', strtotime('+30 days'));
grantPermission($pdo, $user_id, 'assessment.delete', $admin_id, $expiry);
```

#### Revoke Permission

```php
// Revoke permission from user
revokePermission($pdo, $user_id, 'assessment.delete', $admin_id);
```

#### Get User Permissions

```php
// Get all permissions for user
$permissions = getUserPermissions($pdo, $user_id);
// Returns: ['assessment.view', 'assessment.create', ...]

// Check what user can do
foreach ($permissions as $perm) {
    echo $perm . "\n";
}
```

## Use Cases

### Case 1: Temporary Admin Access

**Scenario:** Give user admin access for 7 days

```php
// Grant admin panel access
grantPermission($pdo, $user_id, 'admin.panel', $_SESSION['user_id']);

// Grant settings access with expiry
$expiry = date('Y-m-d H:i:s', strtotime('+7 days'));
grantPermission($pdo, $user_id, 'admin.settings', $_SESSION['user_id'], $expiry);
```

### Case 2: Read-Only Assessor

**Scenario:** Assessor who can only view, not edit

```php
// Keep role as assessor but revoke edit permissions
revokePermission($pdo, $user_id, 'assessment.create');
revokePermission($pdo, $user_id, 'assessment.edit_own');
```

### Case 3: Disable Feature During Maintenance

**Scenario:** Temporarily disable PDF export

```sql
UPDATE feature_toggles 
SET is_enabled = 0 
WHERE feature_key = 'pdf_export_enabled';
```

Or via admin interface: Feature Toggles → Toggle PDF Export

### Case 4: Custom Role Behavior

**Scenario:** Create "Auditor" role with special permissions

```sql
-- Auditors can view everything but not edit
INSERT INTO role_permissions (role, permission_id)
SELECT 'auditor', id FROM permissions 
WHERE permission_key LIKE '%.view%' OR permission_key LIKE 'report.%';
```

## Permission Checking Flow

```
1. User attempts action
         ↓
2. Check if admin → Grant (always)
         ↓
3. Check user_permissions table → Use override if exists
         ↓
4. Check role_permissions table → Use role default
         ↓
5. Grant or Deny access
```

## Database Schema

### permissions Table
```sql
CREATE TABLE permissions (
    id INT PRIMARY KEY,
    permission_key VARCHAR(100) UNIQUE,  -- 'assessment.create'
    permission_name VARCHAR(200),         -- 'Create Assessments'
    permission_category ENUM(...),        -- 'assessments'
    description TEXT,                     -- What it allows
    is_active BOOLEAN
);
```

### role_permissions Table
```sql
CREATE TABLE role_permissions (
    id INT PRIMARY KEY,
    role VARCHAR(50),                     -- 'admin', 'assessor', 'viewer'
    permission_id INT,                    -- Links to permissions.id
    granted_by INT,                       -- Admin who granted it
    UNIQUE(role, permission_id)
);
```

### user_permissions Table
```sql
CREATE TABLE user_permissions (
    id INT PRIMARY KEY,
    user_id INT,                          -- Specific user
    permission_id INT,                    -- Permission being granted/revoked
    is_granted BOOLEAN,                   -- TRUE = grant, FALSE = revoke
    granted_by INT,                       -- Admin who made change
    expires_at TIMESTAMP,                 -- Optional expiry
    UNIQUE(user_id, permission_id)
);
```

### feature_toggles Table
```sql
CREATE TABLE feature_toggles (
    id INT PRIMARY KEY,
    feature_key VARCHAR(100) UNIQUE,      -- 'pdf_export_enabled'
    feature_name VARCHAR(200),            -- 'PDF Export'
    is_enabled BOOLEAN,                   -- ON/OFF
    description TEXT
);
```

## Security Best Practices

### 1. Principle of Least Privilege
- Grant minimum permissions needed
- Use viewer role by default for new users
- Elevate only when necessary

### 2. Regular Audits
```sql
-- Find users with unusual permissions
SELECT u.full_name, p.permission_name, up.expires_at
FROM user_permissions up
JOIN users u ON up.user_id = u.id
JOIN permissions p ON up.permission_id = p.id
WHERE up.is_granted = 1
ORDER BY u.full_name;
```

### 3. Time-Limited Access
- Use `expires_at` for temporary elevated access
- Review expired permissions monthly

### 4. Separation of Duties
- Don't give one user all admin permissions
- Separate user management from system settings
- Multiple admins for critical operations

### 5. Audit Logging
```php
// Log all permission changes
logPermissionChange($pdo, 'grant', $user_id, 'admin.panel', $_SESSION['user_id']);
```

## Troubleshooting

### Issue: User Can't Access Feature

**Check 1:** Is user logged in?
```php
var_dump($_SESSION['user_id']); // Should have value
```

**Check 2:** Does user have permission?
```php
var_dump(hasPermission($pdo, 'feature.permission'));
```

**Check 3:** Is feature enabled?
```php
var_dump(isFeatureEnabled($pdo, 'feature_key'));
```

**Check 4:** Check database
```sql
-- Get user role
SELECT role FROM users WHERE id = X;

-- Get role permissions
SELECT p.permission_key 
FROM role_permissions rp
JOIN permissions p ON rp.permission_id = p.id
WHERE rp.role = 'assessor';

-- Check user overrides
SELECT p.permission_key, up.is_granted
FROM user_permissions up
JOIN permissions p ON up.permission_id = p.id
WHERE up.user_id = X;
```

### Issue: Permission Denied Despite Being Admin

**Cause:** Admin check might not be working

**Fix:**
```php
// Verify admin role
$stmt = $pdo->prepare("SELECT role FROM users WHERE id = ?");
$stmt->execute([$_SESSION['user_id']]);
$user = $stmt->fetch();
echo $user['role']; // Should be 'admin'
```

### Issue: Changes Not Taking Effect

**Cause:** Permission checks might be cached

**Fix:**
- Logout and login again
- Clear PHP session
- Restart PHP-FPM if using

### Issue: Feature Toggle Not Working

**Check table exists:**
```sql
SHOW TABLES LIKE 'feature_toggles';
```

**Check feature status:**
```sql
SELECT * FROM feature_toggles WHERE feature_key = 'your_feature';
```

## API Endpoints (Optional)

### Get Permissions

```php
// GET /api/permissions.php?user_id=5
{
  "success": true,
  "permissions": ["assessment.view", "assessment.create", ...]
}
```

### Grant Permission

```php
// POST /api/permissions.php
{
  "action": "grant",
  "user_id": 5,
  "permission_key": "report.advanced",
  "expires_at": "2026-02-01 00:00:00"
}
```

### Toggle Feature

```php
// POST /api/features.php
{
  "feature_key": "pdf_export_enabled",
  "is_enabled": true
}
```

## Migration from Simple Roles

**Before:** Only 3 roles with fixed permissions

**After:** Flexible role-based + user-specific permissions

**Migration Steps:**
1. Run migration SQL
2. Default permissions auto-assigned
3. Existing users keep current access
4. Can now customize per-user

## Summary

**Features:**
- ✅ 35+ granular permissions
- ✅ 3 default roles (customizable)
- ✅ User-specific overrides
- ✅ Permission expiry dates
- ✅ Feature toggles
- ✅ Admin management UI
- ✅ Helper functions
- ✅ Audit logging

**Benefits:**
- Fine-grained access control
- Flexible permission system
- Temporary access grants
- System-wide feature control
- Security best practices
- Easy to use API

**Requirements:**
- 4 new database tables
- includes/permissions.php
- permissions-management.php (admin UI)
- Updated complete_schema.sql

---

**Control access to your system with precision!** 🔐✨
