# Photo Upload Feature Guide

## Overview

The Risk Assessment System now includes comprehensive photo upload functionality. Add photos to assessments with detailed captions to provide visual documentation of delivery points.

## ✨ Features

- **Multiple Photo Upload**: Upload multiple photos at once
- **Photo Captions**: Add detailed descriptions to each photo
- **Image Preview**: See photos before submitting
- **Image Optimization**: Automatic resizing and compression
- **Secure Storage**: Photos stored securely with access control
- **Photo Gallery**: View all photos in assessment with captions
- **Full-Size View**: Click photos to view in modal/lightbox
- **File Validation**: Size and type checking (JPG, PNG, GIF, WebP)

## 📸 Using Photo Upload

### Adding Photos to New Assessment

1. **Create Assessment**:
   - Go to "New Assessment"
   - Fill in basic information and risk items

2. **Upload Photos**:
   - Scroll to "📷 Assessment Photos" section
   - Click "Choose File" or "Browse"
   - Select one or multiple photos (Ctrl+Click or Shift+Click)
   - Photos appear as previews below

3. **Add Captions**:
   - Each photo has a caption box
   - Describe what the photo shows
   - Be specific and detailed

4. **Example Captions**:
   ```
   ✅ Good: "Loading bay entrance showing 4.2m height restriction sign and narrow turning circle"
   ✅ Good: "Uneven concrete surface at delivery point with visible cracks and puddles"
   ✅ Good: "Limited parking area with three marked bays, often full during peak hours"
   
   ❌ Poor: "Bay entrance"
   ❌ Poor: "Photo 1"
   ❌ Poor: "Here"
   ```

5. **Remove Photos**:
   - Click "🗑️ Remove" button on any photo
   - Photo removed immediately

6. **Submit Assessment**:
   - Click "💾 Save Assessment"
   - Photos upload automatically
   - System shows "📤 Uploading photos..." progress

### Viewing Photos in Assessment

1. **View Assessment**:
   - Open any completed assessment
   - Scroll to "📷 Assessment Photos" section

2. **Photo Gallery**:
   - Photos displayed in responsive grid
   - Shows caption below each photo
   - Displays uploader name and timestamp

3. **Full-Size View**:
   - Click any photo
   - Opens in modal/lightbox
   - Caption displayed below
   - Press ESC or click X to close

## 🔧 Technical Details

### File Requirements

**Accepted Formats**:
- JPEG / JPG
- PNG (with transparency support)
- GIF
- WebP

**File Size Limits**:
- Maximum: 10MB per photo
- Recommended: 2-5MB for optimal loading

**Image Processing**:
- Auto-resize to max 1920x1920px
- JPEG quality: 85%
- Maintains aspect ratio
- Preserves PNG transparency

### Storage Structure

```
risk-system/
└── uploads/
    └── photos/
        ├── .htaccess (security)
        ├── serve-photo.php (secure serving)
        └── photo_1_abc123.jpg (uploaded files)
```

### Database Schema

```sql
CREATE TABLE photos (
    id INT PRIMARY KEY,
    assessment_id INT,  -- Links to assessment
    filename VARCHAR(255),  -- Stored filename
    original_filename VARCHAR(255),  -- Original name
    file_path VARCHAR(500),  -- Relative path
    file_size INT,  -- Bytes
    mime_type VARCHAR(100),  -- image/jpeg, etc
    caption TEXT,  -- Photo description
    display_order INT,  -- Sort order
    uploaded_at TIMESTAMP,
    uploaded_by INT  -- User who uploaded
);
```

### Security Features

1. **Access Control**:
   - Only logged-in users can view photos
   - Photos served through PHP script (not direct access)
   - File path validation prevents directory traversal

2. **File Validation**:
   - MIME type checking
   - File extension verification
   - Size limit enforcement
   - Malicious file detection

3. **Storage Protection**:
   - .htaccess denies direct access
   - Unique filenames prevent conflicts
   - No user-supplied filenames in storage

## 💡 Best Practices

### Taking Good Photos

**Do**:
- ✅ Take photos during actual delivery conditions
- ✅ Capture from multiple angles
- ✅ Show context (surrounding area)
- ✅ Include measuring references when possible
- ✅ Take photos in good lighting
- ✅ Focus on hazards and key features

**Don't**:
- ❌ Include people's faces (privacy)
- ❌ Include vehicle registration numbers
- ❌ Take photos in poor lighting
- ❌ Upload blurry or unclear photos
- ❌ Include sensitive business information

### Writing Good Captions

**Structure**:
1. **What**: Describe what's in the photo
2. **Where**: Specific location within site
3. **Why**: Why it's relevant to assessment
4. **Details**: Measurements, conditions, concerns

**Example**:
```
"Loading bay surface showing significant cracking and 
pothole (approx 15cm deep) near dock door. Surface 
deterioration creates trip hazard during loading/unloading. 
Bay typically wet due to poor drainage."
```

**Helpful Details**:
- Measurements (height, width, depth)
- Time-relevant info (morning/evening, weather)
- Frequency (always/sometimes/rarely)
- Severity (minor/moderate/severe)
- Action needed (repair/replace/monitor)

### Photo Organization

**Recommended Number**:
- Minimum: 3-5 photos per assessment
- Optimal: 8-12 photos covering all areas
- Maximum: 20 photos (avoid overwhelming)

**Suggested Coverage**:
1. **Approach** (1-2 photos): Road access, turning space
2. **Loading Area** (2-3 photos): Bay, surface, equipment
3. **Hazards** (2-4 photos): Each identified risk
4. **Signage** (1-2 photos): Height restrictions, warnings
5. **Context** (1-2 photos): Overall site layout

### File Management

**Before Upload**:
- Review photos on device
- Delete duplicates or poor quality
- Rename files descriptively (optional)
- Compress if very large (>5MB)

**Naming Convention** (if pre-organizing):
```
[Site]_[Area]_[Feature]_[Date].jpg

Examples:
Tesco_Manchester_LoadingBay_Entrance_2024.jpg
Sainsbury_Leeds_Surface_Hazard_2024.jpg
Asda_Bristol_Height_Restriction_2024.jpg
```

## 🚀 Advanced Features

### Bulk Upload

Upload multiple photos efficiently:

1. **Select Multiple**:
   - Windows: Ctrl+Click each file
   - Mac: Cmd+Click each file
   - Range: Shift+Click first and last

2. **Process**:
   - All photos preview together
   - Add captions to each individually
   - Upload together with assessment

### Photo Reordering

Photos display in upload order by default. To reorder in database:

```sql
-- Update display_order for specific photos
UPDATE photos SET display_order = 1 WHERE id = 123;
UPDATE photos SET display_order = 2 WHERE id = 124;
UPDATE photos SET display_order = 3 WHERE id = 125;
```

### Adding Photos to Existing Assessment

Currently, photos must be added during assessment creation. To add later:

**Option 1: Edit Assessment** (if edit function implemented)
- Edit assessment
- Add photos section will be available

**Option 2: Direct Upload** (requires custom code)
```php
// Upload to existing assessment
POST /api/upload-photo.php
{
    assessment_id: 123,
    caption: "New photo caption",
    photo: [file data]
}
```

## 🔍 Troubleshooting

### "File too large" Error

**Problem**: Photo exceeds 10MB limit

**Solutions**:
1. Compress photo before upload:
   - Windows: Paint → Resize → Save
   - Mac: Preview → Tools → Adjust Size
   - Online: TinyPNG, CompressJPEG
2. Reduce resolution if very high (>4000px)
3. Convert from PNG to JPG (if appropriate)

### "Invalid file type" Error

**Problem**: File format not supported

**Solutions**:
1. Convert to JPG/PNG:
   - Windows: Paint → Save As → JPEG
   - Mac: Preview → Export → JPEG
   - Online: CloudConvert, Convertio
2. Check file actually is an image (not renamed)
3. Avoid formats like TIFF, BMP, RAW

### Photos Not Displaying

**Problem**: Photos not showing on assessment view

**Possible Causes & Fixes**:

1. **File Permissions**:
   ```bash
   chmod 755 uploads/photos
   chmod 644 uploads/photos/*
   ```

2. **Missing .htaccess**:
   ```bash
   # Verify it exists
   ls -la uploads/photos/.htaccess
   ```

3. **PHP GD Library** (for optimization):
   ```bash
   # Check if installed
   php -m | grep gd
   
   # Install if missing
   sudo apt-get install php-gd
   sudo systemctl restart apache2
   ```

4. **Path Issues**:
   ```php
   // Check in serve-photo.php
   echo __DIR__;  // Should show correct path
   ```

### Upload Fails Silently

**Check**:
1. Browser console (F12) for errors
2. PHP error log: `tail -f /var/log/apache2/error.log`
3. Upload directory writable: `ls -la uploads/`
4. PHP upload limits in `php.ini`:
   ```ini
   upload_max_filesize = 10M
   post_max_size = 15M
   ```

### Photos Blurry After Upload

**Cause**: Aggressive compression

**Fix**: Adjust quality in `upload-photo.php`:
```php
// Change from 85 to 90 or 95
$quality = 90;
imagejpeg($destination, $filepath, $quality);
```

## 📊 Database Queries

### View All Photos for Assessment

```sql
SELECT 
    p.*,
    u.full_name as uploader,
    a.assessment_number
FROM photos p
JOIN assessments a ON p.assessment_id = a.id
JOIN users u ON p.uploaded_by = u.id
WHERE a.id = 123
ORDER BY p.display_order, p.uploaded_at;
```

### Photos Without Captions

```sql
SELECT 
    p.id,
    p.filename,
    a.assessment_number,
    a.assessment_date
FROM photos p
JOIN assessments a ON p.assessment_id = a.id
WHERE p.caption IS NULL OR p.caption = ''
ORDER BY a.assessment_date DESC;
```

### Storage Usage

```sql
-- Total storage by assessment
SELECT 
    a.assessment_number,
    a.location_id,
    COUNT(p.id) as photo_count,
    SUM(p.file_size) / 1024 / 1024 as total_mb
FROM assessments a
LEFT JOIN photos p ON a.id = p.assessment_id
GROUP BY a.id
ORDER BY total_mb DESC;

-- Overall storage
SELECT 
    COUNT(*) as total_photos,
    SUM(file_size) / 1024 / 1024 as total_mb
FROM photos;
```

### Delete Old Photos

```bash
# Delete photos from assessments older than 2 years
DELETE FROM photos 
WHERE assessment_id IN (
    SELECT id FROM assessments 
    WHERE assessment_date < DATE_SUB(NOW(), INTERVAL 2 YEAR)
);

# Remember to also delete physical files!
```

## 🎓 Training Users

### Quick Guide for Assessors

**Photo Upload in 3 Steps**:
1. Click "Choose File" in assessment form
2. Select photos from your device
3. Add description in caption box
4. Save assessment

**Tips**:
- Take 5-10 photos covering all areas
- Write detailed captions
- Include measurements when possible
- Show hazards clearly

### Training Checklist

- [ ] How to take good assessment photos
- [ ] Understanding file size limits
- [ ] Writing descriptive captions
- [ ] Reviewing photos before upload
- [ ] Viewing photos in assessments
- [ ] Privacy considerations (no faces)
- [ ] Security awareness (no sensitive data)

## 🔐 Security & Privacy

### Personal Data

**Avoid Capturing**:
- People's faces
- Vehicle registration plates
- Personal identification
- Confidential documents
- Security codes/keys visible

**If Accidentally Captured**:
- Don't upload the photo
- Retake without sensitive information
- Blur/crop before upload (external tool)
- Delete from device

### Data Retention

**Policy Considerations**:
- How long to keep photos
- When to archive/delete
- Backup procedures
- Access logs

**Recommended**:
- Keep with assessment (same retention)
- Delete if assessment archived
- Regular backups
- Audit photo access

---

**The photo upload feature provides comprehensive visual documentation to support risk assessments and improve delivery point safety!** 📸
