# ============================================================================
# Risk Assessment System - Windows 11 Installation Script
# Automated installer with GUI and dependency management
# ============================================================================

#Requires -Version 5.1
#Requires -RunAsAdministrator

param(
    [switch]$Unattended,
    [string]$InstallPath = "C:\inetpub\wwwroot\risk-assessment",
    [string]$MySQLRootPassword = "",
    [switch]$SkipDependencies
)

# Set strict mode
Set-StrictMode -Version Latest
$ErrorActionPreference = "Stop"

# ============================================================================
# CONFIGURATION
# ============================================================================

$script:Config = @{
    AppName = "Risk Assessment System"
    Version = "1.0.0"
    PHPVersion = "8.2"
    MySQLVersion = "8.0"
    ApacheVersion = "2.4"
    
    # Download URLs
    PHPUrl = "https://windows.php.net/downloads/releases/php-8.2.14-Win32-vs16-x64.zip"
    MySQLUrl = "https://dev.mysql.com/get/Downloads/MySQLInstaller/mysql-installer-community-8.0.35.0.msi"
    VCRedistUrl = "https://aka.ms/vs/17/release/vc_redist.x64.exe"
    
    # Paths
    TempPath = "$env:TEMP\RiskAssessmentInstaller"
    LogPath = "$env:TEMP\RiskAssessmentInstaller\install.log"
    
    # Database
    DBName = "risk_assessment"
    DBUser = "risk_user"
    DBPassword = ""  # Will be generated
}

# ============================================================================
# LOGGING
# ============================================================================

function Write-Log {
    param(
        [string]$Message,
        [ValidateSet('Info', 'Success', 'Warning', 'Error')]
        [string]$Level = 'Info'
    )
    
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "[$timestamp] [$Level] $Message"
    
    # Console output with colors
    switch ($Level) {
        'Success' { Write-Host $logMessage -ForegroundColor Green }
        'Warning' { Write-Host $logMessage -ForegroundColor Yellow }
        'Error' { Write-Host $logMessage -ForegroundColor Red }
        default { Write-Host $logMessage -ForegroundColor White }
    }
    
    # File output
    Add-Content -Path $script:Config.LogPath -Value $logMessage
}

# ============================================================================
# UTILITY FUNCTIONS
# ============================================================================

function Test-Administrator {
    $currentUser = [Security.Principal.WindowsIdentity]::GetCurrent()
    $principal = New-Object Security.Principal.WindowsPrincipal($currentUser)
    return $principal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

function New-RandomPassword {
    param([int]$Length = 16)
    
    $chars = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%"
    $password = -join ((1..$Length) | ForEach-Object { $chars[(Get-Random -Maximum $chars.Length)] })
    return $password
}

function Test-Port {
    param([int]$Port)
    
    try {
        $listener = [System.Net.Sockets.TcpListener]::new([System.Net.IPAddress]::Any, $Port)
        $listener.Start()
        $listener.Stop()
        return $true
    } catch {
        return $false
    }
}

function Show-Banner {
    Clear-Host
    Write-Host "============================================================================" -ForegroundColor Cyan
    Write-Host "  Risk Assessment System - Windows 11 Installer" -ForegroundColor Cyan
    Write-Host "  Version $($script:Config.Version)" -ForegroundColor Cyan
    Write-Host "============================================================================" -ForegroundColor Cyan
    Write-Host ""
}

function Show-Progress {
    param(
        [string]$Activity,
        [string]$Status,
        [int]$PercentComplete
    )
    
    Write-Progress -Activity $Activity -Status $Status -PercentComplete $PercentComplete
}

# ============================================================================
# PRE-INSTALLATION CHECKS
# ============================================================================

function Test-Prerequisites {
    Write-Log "Checking system prerequisites..." -Level Info
    
    # Check Windows version
    $os = Get-CimInstance Win32_OperatingSystem
    if ($os.Caption -notlike "*Windows 11*" -and $os.Caption -notlike "*Windows 10*" -and $os.Caption -notlike "*Server*") {
        Write-Log "Unsupported Windows version: $($os.Caption)" -Level Error
        return $false
    }
    Write-Log "Windows version OK: $($os.Caption)" -Level Success
    
    # Check if IIS is installed
    $iis = Get-WindowsOptionalFeature -Online -FeatureName IIS-WebServerRole -ErrorAction SilentlyContinue
    if ($iis.State -eq "Enabled") {
        Write-Log "IIS detected and will be used" -Level Info
        $script:UseIIS = $true
    } else {
        Write-Log "IIS not found, will use standalone PHP server" -Level Info
        $script:UseIIS = $false
    }
    
    # Check available disk space (need at least 2GB)
    $drive = (Get-Item $InstallPath -ErrorAction SilentlyContinue).PSDrive.Name
    if (-not $drive) {
        $drive = $InstallPath.Substring(0, 1)
    }
    $disk = Get-PSDrive $drive
    $freeSpaceGB = [math]::Round($disk.Free / 1GB, 2)
    
    if ($freeSpaceGB -lt 2) {
        Write-Log "Insufficient disk space: ${freeSpaceGB}GB available, need 2GB minimum" -Level Error
        return $false
    }
    Write-Log "Disk space OK: ${freeSpaceGB}GB available" -Level Success
    
    # Check internet connectivity
    try {
        $null = Invoke-WebRequest -Uri "https://www.google.com" -UseBasicParsing -TimeoutSec 5
        Write-Log "Internet connectivity OK" -Level Success
    } catch {
        Write-Log "No internet connection detected. Downloads may fail." -Level Warning
    }
    
    return $true
}

# ============================================================================
# DEPENDENCY INSTALLATION
# ============================================================================

function Install-VisualCRedist {
    Write-Log "Installing Visual C++ Redistributable..." -Level Info
    Show-Progress -Activity "Installing Dependencies" -Status "Visual C++ Redistributable" -PercentComplete 10
    
    $vcRedistPath = Join-Path $script:Config.TempPath "vc_redist.x64.exe"
    
    try {
        # Download
        Write-Log "Downloading VC++ Redistributable..."
        Invoke-WebRequest -Uri $script:Config.VCRedistUrl -OutFile $vcRedistPath -UseBasicParsing
        
        # Install silently
        Write-Log "Installing VC++ Redistributable..."
        Start-Process -FilePath $vcRedistPath -ArgumentList "/install", "/quiet", "/norestart" -Wait
        
        Write-Log "Visual C++ Redistributable installed successfully" -Level Success
        return $true
    } catch {
        Write-Log "Failed to install Visual C++ Redistributable: $_" -Level Error
        return $false
    }
}

function Install-PHP {
    Write-Log "Installing PHP..." -Level Info
    Show-Progress -Activity "Installing Dependencies" -Status "PHP" -PercentComplete 20
    
    $phpPath = "C:\PHP"
    $phpZip = Join-Path $script:Config.TempPath "php.zip"
    
    try {
        # Download PHP
        Write-Log "Downloading PHP $($script:Config.PHPVersion)..."
        Invoke-WebRequest -Uri $script:Config.PHPUrl -OutFile $phpZip -UseBasicParsing
        
        # Extract
        Write-Log "Extracting PHP..."
        if (Test-Path $phpPath) {
            Remove-Item $phpPath -Recurse -Force
        }
        Expand-Archive -Path $phpZip -DestinationPath $phpPath -Force
        
        # Configure PHP
        Write-Log "Configuring PHP..."
        $phpIni = Join-Path $phpPath "php.ini"
        if (-not (Test-Path $phpIni)) {
            Copy-Item (Join-Path $phpPath "php.ini-production") $phpIni
        }
        
        # Enable extensions
        $extensions = @(
            "extension=curl",
            "extension=fileinfo",
            "extension=gd",
            "extension=mbstring",
            "extension=mysqli",
            "extension=openssl",
            "extension=pdo_mysql"
        )
        
        $iniContent = Get-Content $phpIni
        foreach ($ext in $extensions) {
            if ($iniContent -notcontains $ext) {
                Add-Content $phpIni "`n$ext"
            }
        }
        
        # Set other PHP settings
        $settings = @(
            "upload_max_filesize = 10M",
            "post_max_size = 12M",
            "memory_limit = 256M",
            "max_execution_time = 60",
            "date.timezone = UTC"
        )
        
        foreach ($setting in $settings) {
            Add-Content $phpIni "`n$setting"
        }
        
        # Add PHP to PATH
        $currentPath = [Environment]::GetEnvironmentVariable("Path", "Machine")
        if ($currentPath -notlike "*$phpPath*") {
            [Environment]::SetEnvironmentVariable("Path", "$currentPath;$phpPath", "Machine")
            $env:Path = "$env:Path;$phpPath"
        }
        
        Write-Log "PHP installed successfully at $phpPath" -Level Success
        return $true
    } catch {
        Write-Log "Failed to install PHP: $_" -Level Error
        return $false
    }
}

function Install-MySQL {
    Write-Log "Installing MySQL..." -Level Info
    Show-Progress -Activity "Installing Dependencies" -Status "MySQL" -PercentComplete 40
    
    $mysqlInstaller = Join-Path $script:Config.TempPath "mysql-installer.msi"
    
    try {
        # Download MySQL Installer
        Write-Log "Downloading MySQL $($script:Config.MySQLVersion)..."
        Invoke-WebRequest -Uri $script:Config.MySQLUrl -OutFile $mysqlInstaller -UseBasicParsing
        
        # Install MySQL silently
        Write-Log "Installing MySQL (this may take several minutes)..."
        
        if ([string]::IsNullOrEmpty($MySQLRootPassword)) {
            $script:Config.MySQLRootPassword = New-RandomPassword
        } else {
            $script:Config.MySQLRootPassword = $MySQLRootPassword
        }
        
        $installArgs = @(
            "/i",
            "`"$mysqlInstaller`"",
            "/quiet",
            "/norestart"
        )
        
        Start-Process "msiexec.exe" -ArgumentList $installArgs -Wait -NoNewWindow
        
        # Wait for MySQL service to start
        Write-Log "Waiting for MySQL service..."
        Start-Sleep -Seconds 10
        
        $mysqlService = Get-Service -Name "MySQL*" -ErrorAction SilentlyContinue | Select-Object -First 1
        if ($mysqlService) {
            if ($mysqlService.Status -ne "Running") {
                Start-Service $mysqlService.Name
            }
            Write-Log "MySQL service started: $($mysqlService.Name)" -Level Success
        }
        
        # Add MySQL to PATH
        $mysqlPath = "C:\Program Files\MySQL\MySQL Server 8.0\bin"
        if (Test-Path $mysqlPath) {
            $currentPath = [Environment]::GetEnvironmentVariable("Path", "Machine")
            if ($currentPath -notlike "*$mysqlPath*") {
                [Environment]::SetEnvironmentVariable("Path", "$currentPath;$mysqlPath", "Machine")
                $env:Path = "$env:Path;$mysqlPath"
            }
        }
        
        Write-Log "MySQL installed successfully" -Level Success
        Write-Log "MySQL Root Password: $($script:Config.MySQLRootPassword)" -Level Info
        
        return $true
    } catch {
        Write-Log "Failed to install MySQL: $_" -Level Error
        return $false
    }
}

# ============================================================================
# APPLICATION INSTALLATION
# ============================================================================

function Install-Application {
    Write-Log "Installing Risk Assessment System..." -Level Info
    Show-Progress -Activity "Installing Application" -Status "Extracting files" -PercentComplete 60
    
    try {
        # Create installation directory
        if (-not (Test-Path $InstallPath)) {
            New-Item -Path $InstallPath -ItemType Directory -Force | Out-Null
        }
        
        # Find and extract the application archive
        $archivePath = Get-ChildItem -Path . -Filter "risk-system*.tar.gz" | Select-Object -First 1 -ExpandProperty FullName
        
        if (-not $archivePath) {
            Write-Log "Application archive not found in current directory" -Level Error
            return $false
        }
        
        Write-Log "Found archive: $archivePath"
        
        # Extract using tar (built into Windows 10/11)
        Write-Log "Extracting application files..."
        tar -xzf "$archivePath" -C "$InstallPath" --strip-components=1
        
        Write-Log "Application files extracted successfully" -Level Success
        return $true
    } catch {
        Write-Log "Failed to install application: $_" -Level Error
        return $false
    }
}

function Initialize-Database {
    Write-Log "Initializing database..." -Level Info
    Show-Progress -Activity "Installing Application" -Status "Setting up database" -PercentComplete 70
    
    try {
        # Generate database password
        $script:Config.DBPassword = New-RandomPassword
        
        # MySQL executable path
        $mysqlExe = "C:\Program Files\MySQL\MySQL Server 8.0\bin\mysql.exe"
        
        if (-not (Test-Path $mysqlExe)) {
            $mysqlExe = "mysql"  # Try PATH
        }
        
        # Create database and user
        $sqlCommands = @"
CREATE DATABASE IF NOT EXISTS $($script:Config.DBName);
CREATE USER IF NOT EXISTS '$($script:Config.DBUser)'@'localhost' IDENTIFIED BY '$($script:Config.DBPassword)';
GRANT ALL PRIVILEGES ON $($script:Config.DBName).* TO '$($script:Config.DBUser)'@'localhost';
FLUSH PRIVILEGES;
USE $($script:Config.DBName);
SOURCE $InstallPath/includes/complete_schema.sql;
"@
        
        $sqlFile = Join-Path $script:Config.TempPath "setup.sql"
        $sqlCommands | Out-File -FilePath $sqlFile -Encoding UTF8
        
        Write-Log "Creating database and importing schema..."
        
        # Execute SQL
        $processArgs = @(
            "-u", "root",
            "-p$($script:Config.MySQLRootPassword)",
            "-e", "source $sqlFile"
        )
        
        Start-Process -FilePath $mysqlExe -ArgumentList $processArgs -Wait -NoNewWindow
        
        Write-Log "Database initialized successfully" -Level Success
        Write-Log "Database Name: $($script:Config.DBName)" -Level Info
        Write-Log "Database User: $($script:Config.DBUser)" -Level Info
        Write-Log "Database Password: $($script:Config.DBPassword)" -Level Info
        
        return $true
    } catch {
        Write-Log "Failed to initialize database: $_" -Level Error
        return $false
    }
}

function Update-Configuration {
    Write-Log "Updating configuration..." -Level Info
    Show-Progress -Activity "Installing Application" -Status "Configuring application" -PercentComplete 80
    
    try {
        $configPath = Join-Path $InstallPath "includes\config.php"
        
        if (-not (Test-Path $configPath)) {
            Write-Log "Config file not found: $configPath" -Level Error
            return $false
        }
        
        # Read config
        $config = Get-Content $configPath -Raw
        
        # Update database settings
        $config = $config -replace "define\('DB_HOST',\s*'[^']*'\);", "define('DB_HOST', 'localhost');"
        $config = $config -replace "define\('DB_USER',\s*'[^']*'\);", "define('DB_USER', '$($script:Config.DBUser)');"
        $config = $config -replace "define\('DB_PASS',\s*'[^']*'\);", "define('DB_PASS', '$($script:Config.DBPassword)');"
        $config = $config -replace "define\('DB_NAME',\s*'[^']*'\);", "define('DB_NAME', '$($script:Config.DBName)');"
        
        # Update site URL
        $siteUrl = "http://localhost:8080"
        $config = $config -replace "define\('SITE_URL',\s*'[^']*'\);", "define('SITE_URL', '$siteUrl');"
        
        # Enable SMTP (optional)
        $config = $config -replace "define\('SMTP_ENABLED',\s*false\);", "define('SMTP_ENABLED', false);"
        
        # Save config
        $config | Out-File -FilePath $configPath -Encoding UTF8 -Force
        
        Write-Log "Configuration updated successfully" -Level Success
        return $true
    } catch {
        Write-Log "Failed to update configuration: $_" -Level Error
        return $false
    }
}

function Set-Permissions {
    Write-Log "Setting file permissions..." -Level Info
    Show-Progress -Activity "Installing Application" -Status "Setting permissions" -PercentComplete 85
    
    try {
        # Create directories if they don't exist
        $directories = @(
            (Join-Path $InstallPath "uploads"),
            (Join-Path $InstallPath "logs")
        )
        
        foreach ($dir in $directories) {
            if (-not (Test-Path $dir)) {
                New-Item -Path $dir -ItemType Directory -Force | Out-Null
            }
            
            # Set permissions for IIS_IUSRS if using IIS
            if ($script:UseIIS) {
                $acl = Get-Acl $dir
                $permission = "IIS_IUSRS", "FullControl", "ContainerInherit,ObjectInherit", "None", "Allow"
                $accessRule = New-Object System.Security.AccessControl.FileSystemAccessRule $permission
                $acl.SetAccessRule($accessRule)
                Set-Acl $dir $acl
            }
        }
        
        Write-Log "Permissions set successfully" -Level Success
        return $true
    } catch {
        Write-Log "Failed to set permissions: $_" -Level Error
        return $false
    }
}

# ============================================================================
# WEB SERVER CONFIGURATION
# ============================================================================

function Configure-IIS {
    Write-Log "Configuring IIS..." -Level Info
    Show-Progress -Activity "Configuring Server" -Status "IIS Setup" -PercentComplete 90
    
    try {
        Import-Module WebAdministration -ErrorAction Stop
        
        # Create application pool
        $appPoolName = "RiskAssessmentPool"
        if (-not (Test-Path "IIS:\AppPools\$appPoolName")) {
            New-WebAppPool -Name $appPoolName
            Set-ItemProperty "IIS:\AppPools\$appPoolName" -Name "managedRuntimeVersion" -Value ""
        }
        
        # Create website
        $siteName = "RiskAssessment"
        $port = 8080
        
        # Find available port
        while (-not (Test-Port -Port $port) -and $port -lt 9000) {
            $port++
        }
        
        if (Test-Path "IIS:\Sites\$siteName") {
            Remove-Website -Name $siteName
        }
        
        New-Website -Name $siteName -PhysicalPath "$InstallPath\public" -ApplicationPool $appPoolName -Port $port
        
        # Configure PHP for IIS
        $phpPath = "C:\PHP\php-cgi.exe"
        if (Test-Path $phpPath) {
            & "$env:SystemRoot\System32\inetsrv\appcmd.exe" set config -section:system.webServer/fastCgi `
                /+"[fullPath='$phpPath']" /commit:apphost
            
            & "$env:SystemRoot\System32\inetsrv\appcmd.exe" set config -section:system.webServer/handlers `
                /+"[name='PHP',path='*.php',verb='GET,HEAD,POST',modules='FastCgiModule',scriptProcessor='$phpPath',resourceType='Either']"
        }
        
        Write-Log "IIS configured successfully" -Level Success
        Write-Log "Application URL: http://localhost:$port" -Level Info
        
        $script:ApplicationUrl = "http://localhost:$port"
        
        return $true
    } catch {
        Write-Log "Failed to configure IIS: $_" -Level Warning
        Write-Log "Will use PHP built-in server instead" -Level Info
        return Configure-PHPServer
    }
}

function Configure-PHPServer {
    Write-Log "Configuring PHP built-in server..." -Level Info
    
    try {
        $port = 8080
        
        # Find available port
        while (-not (Test-Port -Port $port) -and $port -lt 9000) {
            $port++
        }
        
        # Create startup script
        $startupScript = @"
@echo off
echo Starting Risk Assessment System...
echo.
echo Server running at: http://localhost:$port
echo Press Ctrl+C to stop the server
echo.
cd /d "$InstallPath\public"
php -S localhost:$port
"@
        
        $scriptPath = Join-Path $InstallPath "start-server.bat"
        $startupScript | Out-File -FilePath $scriptPath -Encoding ASCII
        
        # Create Windows shortcut
        $shell = New-Object -ComObject WScript.Shell
        $shortcut = $shell.CreateShortcut("$env:USERPROFILE\Desktop\Risk Assessment Server.lnk")
        $shortcut.TargetPath = $scriptPath
        $shortcut.WorkingDirectory = "$InstallPath\public"
        $shortcut.IconLocation = "imageres.dll,1"
        $shortcut.Save()
        
        Write-Log "PHP server configured successfully" -Level Success
        Write-Log "Application URL: http://localhost:$port" -Level Info
        Write-Log "Start server using: $scriptPath" -Level Info
        
        $script:ApplicationUrl = "http://localhost:$port"
        $script:ServerScript = $scriptPath
        
        return $true
    } catch {
        Write-Log "Failed to configure PHP server: $_" -Level Error
        return $false
    }
}

# ============================================================================
# POST-INSTALLATION
# ============================================================================

function Show-Summary {
    Write-Log "" -Level Info
    Write-Log "============================================================================" -Level Success
    Write-Log "  Installation Complete!" -Level Success
    Write-Log "============================================================================" -Level Success
    Write-Log "" -Level Info
    
    Write-Host "`n📦 INSTALLATION SUMMARY" -ForegroundColor Cyan
    Write-Host "=" * 80 -ForegroundColor Cyan
    
    Write-Host "`n🌐 Application URL:" -ForegroundColor Yellow
    Write-Host "   $($script:ApplicationUrl)" -ForegroundColor White
    
    Write-Host "`n🔐 Default Login Credentials:" -ForegroundColor Yellow
    Write-Host "   Username: admin" -ForegroundColor White
    Write-Host "   Password: Admin123!" -ForegroundColor White
    Write-Host "   ⚠️  CHANGE THIS PASSWORD IMMEDIATELY AFTER LOGIN" -ForegroundColor Red
    
    Write-Host "`n💾 Database Configuration:" -ForegroundColor Yellow
    Write-Host "   Host:     localhost" -ForegroundColor White
    Write-Host "   Database: $($script:Config.DBName)" -ForegroundColor White
    Write-Host "   User:     $($script:Config.DBUser)" -ForegroundColor White
    Write-Host "   Password: $($script:Config.DBPassword)" -ForegroundColor White
    
    Write-Host "`n📁 Installation Directory:" -ForegroundColor Yellow
    Write-Host "   $InstallPath" -ForegroundColor White
    
    if ($script:ServerScript) {
        Write-Host "`n🚀 Starting the Server:" -ForegroundColor Yellow
        Write-Host "   Run: $($script:ServerScript)" -ForegroundColor White
        Write-Host "   Or use the desktop shortcut: 'Risk Assessment Server'" -ForegroundColor White
    }
    
    Write-Host "`n📝 Important Files:" -ForegroundColor Yellow
    Write-Host "   Config:  $InstallPath\includes\config.php" -ForegroundColor White
    Write-Host "   Logs:    $InstallPath\logs\" -ForegroundColor White
    Write-Host "   Uploads: $InstallPath\uploads\" -ForegroundColor White
    
    Write-Host "`n📚 Documentation:" -ForegroundColor Yellow
    Write-Host "   README:  $InstallPath\README.md" -ForegroundColor White
    Write-Host "   Guides:  $InstallPath\docs\" -ForegroundColor White
    
    Write-Host "`n📊 Installation Log:" -ForegroundColor Yellow
    Write-Host "   $($script:Config.LogPath)" -ForegroundColor White
    
    Write-Host "`n" -NoNewline
    Write-Host "=" * 80 -ForegroundColor Cyan
    Write-Host "`n✅ Installation completed successfully!`n" -ForegroundColor Green
    
    # Save summary to file
    $summaryPath = Join-Path $InstallPath "INSTALLATION_INFO.txt"
    @"
Risk Assessment System - Installation Summary
==============================================

Installation Date: $(Get-Date -Format "yyyy-MM-dd HH:mm:ss")

Application URL: $($script:ApplicationUrl)

Default Login:
  Username: admin
  Password: Admin123!
  ⚠️  CHANGE THIS PASSWORD IMMEDIATELY

Database Configuration:
  Host: localhost
  Database: $($script:Config.DBName)
  User: $($script:Config.DBUser)
  Password: $($script:Config.DBPassword)

Installation Directory: $InstallPath

MySQL Root Password: $($script:Config.MySQLRootPassword)

Server Start Command:
  $($script:ServerScript)

Important: Keep this file secure and delete it after noting the credentials.
"@ | Out-File -FilePath $summaryPath -Encoding UTF8
    
    Write-Host "💾 Installation summary saved to: $summaryPath`n" -ForegroundColor Cyan
}

function Start-Application {
    param([switch]$LaunchBrowser)
    
    if ($script:ServerScript) {
        Write-Host "`nWould you like to start the server now? (Y/N): " -NoNewline -ForegroundColor Yellow
        $response = Read-Host
        
        if ($response -eq 'Y' -or $response -eq 'y') {
            Write-Host "`nStarting PHP server..." -ForegroundColor Green
            Start-Process -FilePath $script:ServerScript
            Start-Sleep -Seconds 2
        }
    }
    
    if ($LaunchBrowser -or $response -eq 'Y' -or $response -eq 'y') {
        Write-Host "Opening browser..." -ForegroundColor Green
        Start-Sleep -Seconds 3
        Start-Process $script:ApplicationUrl
    }
}

# ============================================================================
# MAIN INSTALLATION FLOW
# ============================================================================

function Start-Installation {
    try {
        # Initialize
        Show-Banner
        
        # Create temp directory
        if (-not (Test-Path $script:Config.TempPath)) {
            New-Item -Path $script:Config.TempPath -ItemType Directory -Force | Out-Null
        }
        
        # Start logging
        "Installation started: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" | Out-File -FilePath $script:Config.LogPath
        
        Write-Log "Starting installation of $($script:Config.AppName) v$($script:Config.Version)" -Level Info
        Write-Log "Install path: $InstallPath" -Level Info
        
        # Pre-installation checks
        if (-not (Test-Administrator)) {
            Write-Log "This script must be run as Administrator" -Level Error
            Write-Host "`nPlease run PowerShell as Administrator and try again.`n" -ForegroundColor Red
            return
        }
        
        if (-not (Test-Prerequisites)) {
            Write-Log "Prerequisites check failed" -Level Error
            return
        }
        
        Write-Host "`n⚠️  This will install:" -ForegroundColor Yellow
        Write-Host "   • Visual C++ Redistributable" -ForegroundColor White
        Write-Host "   • PHP $($script:Config.PHPVersion)" -ForegroundColor White
        Write-Host "   • MySQL $($script:Config.MySQLVersion)" -ForegroundColor White
        Write-Host "   • Risk Assessment System`n" -ForegroundColor White
        
        if (-not $Unattended) {
            Write-Host "Press Enter to continue or Ctrl+C to cancel..." -ForegroundColor Yellow
            Read-Host
        }
        
        # Install dependencies
        if (-not $SkipDependencies) {
            if (-not (Install-VisualCRedist)) { return }
            if (-not (Install-PHP)) { return }
            if (-not (Install-MySQL)) { return }
        }
        
        # Install application
        if (-not (Install-Application)) { return }
        if (-not (Initialize-Database)) { return }
        if (-not (Update-Configuration)) { return }
        if (-not (Set-Permissions)) { return }
        
        # Configure web server
        if ($script:UseIIS) {
            Configure-IIS
        } else {
            Configure-PHPServer
        }
        
        # Complete
        Show-Progress -Activity "Installation" -Status "Complete" -PercentComplete 100
        Write-Progress -Activity "Installation" -Completed
        
        Show-Summary
        
        if (-not $Unattended) {
            Start-Application -LaunchBrowser
        }
        
    } catch {
        Write-Log "Installation failed: $_" -Level Error
        Write-Log $_.ScriptStackTrace -Level Error
        Write-Host "`n❌ Installation failed. Check log file: $($script:Config.LogPath)`n" -ForegroundColor Red
    } finally {
        # Cleanup temp files
        if (Test-Path $script:Config.TempPath) {
            Write-Log "Cleaning up temporary files..." -Level Info
            Remove-Item $script:Config.TempPath -Recurse -Force -ErrorAction SilentlyContinue
        }
    }
}

# ============================================================================
# SCRIPT ENTRY POINT
# ============================================================================

# Check if running as administrator
if (-not (Test-Administrator)) {
    Write-Host "ERROR: This script requires administrator privileges.`n" -ForegroundColor Red
    Write-Host "Please:" -ForegroundColor Yellow
    Write-Host "1. Right-click PowerShell" -ForegroundColor White
    Write-Host "2. Select 'Run as Administrator'" -ForegroundColor White
    Write-Host "3. Navigate to this directory and run the script again`n" -ForegroundColor White
    pause
    exit 1
}

# Run installation
Start-Installation
