# Directory Structure & Deployment Guide

## 📁 Correct Directory Structure

The application has a specific directory structure that must be maintained:

```
/var/www/html/risk-system/    (or your web root)
│
├── public/                             ← Document Root (Point Apache/Nginx here!)
│   ├── css/
│   │   └── style.css
│   ├── js/
│   │   ├── app.js
│   │   └── sw.js
│   ├── api/
│   │   ├── generate-pdf.php
│   │   └── generate_pdf.py
│   ├── images/                         (create this if needed)
│   ├── *.php                          (All PHP pages)
│   ├── manifest.json
│   ├── offline.html
│   └── .htaccess
│
├── includes/                          ← NOT web accessible
│   ├── config.php
│   ├── schema.sql
│   ├── migration_*.sql
│   └── reset-password.php
│
├── templates/                         ← NOT web accessible
│   └── header.php
│
├── reports/                           ← NOT web accessible (write permissions needed)
│   └── (generated PDFs)
│
└── *.md                               ← Documentation files
```

## ⚠️ CRITICAL: Web Server Configuration

### The Problem

If you're getting 404 errors on menu items, it's because your web server's **document root** is pointing to the wrong directory.

### The Solution

Your web server must point to the `public` directory, NOT the root `risk-system` directory!

## Apache Configuration

### Option 1: Virtual Host (Recommended)

Edit `/etc/apache2/sites-available/risk-assessment.conf`:

```apache
<VirtualHost *:80>
    ServerName your-domain.com
    ServerAlias www.your-domain.com
    
    # IMPORTANT: Point to the public directory!
    DocumentRoot /var/www/html/risk-system/public
    
    <Directory /var/www/html/risk-system/public>
        Options -Indexes +FollowSymLinks
        AllowOverride All
        Require all granted
        
        # Enable .htaccess
        <IfModule mod_rewrite.c>
            RewriteEngine On
        </IfModule>
    </Directory>
    
    # Protect sensitive directories
    <DirectoryMatch "/(includes|templates|reports)">
        Require all denied
    </DirectoryMatch>
    
    ErrorLog ${APACHE_LOG_DIR}/risk-assessment-error.log
    CustomLog ${APACHE_LOG_DIR}/risk-assessment-access.log combined
</VirtualHost>
```

Enable the site:
```bash
sudo a2ensite risk-assessment
sudo a2enmod rewrite
sudo systemctl reload apache2
```

### Option 2: .htaccess Redirect (If you can't change document root)

If your document root MUST be `/var/www/html/risk-system`, add this to the ROOT `.htaccess`:

```apache
<IfModule mod_rewrite.c>
    RewriteEngine On
    RewriteBase /
    
    # Redirect everything to public directory
    RewriteCond %{REQUEST_URI} !^/public/
    RewriteRule ^(.*)$ public/$1 [L]
</IfModule>
```

### Option 3: Symlink (Quick fix for development)

```bash
cd /var/www/html
ln -s risk-system/public risk-assessment
# Now access via: http://localhost/risk-assessment/
```

## Nginx Configuration

Edit `/etc/nginx/sites-available/risk-assessment`:

```nginx
server {
    listen 80;
    server_name your-domain.com www.your-domain.com;
    
    # IMPORTANT: Point to the public directory!
    root /var/www/html/risk-system/public;
    index index.php index.html;
    
    # Security - deny access to sensitive directories
    location ~ ^/(includes|templates|reports) {
        deny all;
        return 404;
    }
    
    # PHP handling
    location ~ \.php$ {
        include snippets/fastcgi-php.conf;
        fastcgi_pass unix:/var/run/php/php7.4-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }
    
    # Deny access to hidden files
    location ~ /\. {
        deny all;
        access_log off;
        log_not_found off;
    }
    
    error_log /var/log/nginx/risk-assessment-error.log;
    access_log /var/log/nginx/risk-assessment-access.log;
}
```

Enable the site:
```bash
sudo ln -s /etc/nginx/sites-available/risk-assessment /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx
```

## cPanel / Shared Hosting

### Method 1: Change Document Root (Recommended)

1. Login to cPanel
2. Go to "Domains" or "Addon Domains"
3. Find your domain
4. Click "Manage" or "Document Root"
5. Change from: `/public_html/risk-system`
6. Change to: `/public_html/risk-system/public`
7. Save

### Method 2: Move Files

If you can't change document root:

```bash
# Move public contents to root
cd /home/username/public_html/risk-system
mv public/* .
mv public/.htaccess .
rmdir public

# Update config.php paths
sed -i "s|__DIR__ . '/../|__DIR__ . '/|g" includes/config.php
```

## Verifying Correct Setup

### Test 1: Check Document Root

Visit: `http://your-domain.com/`

- ✅ Should show login page or dashboard
- ❌ Should NOT show directory listing
- ❌ Should NOT show 404 error

### Test 2: Check Navigation

Click on each menu item:
- ✅ Dashboard → Should load
- ✅ Assessments → Should load
- ✅ Locations → Should load
- ❌ Should NOT get 404 errors

### Test 3: Check Security

Try to access:
- `http://your-domain.com/includes/config.php` → Should be DENIED (403/404)
- `http://your-domain.com/templates/header.php` → Should be DENIED (403/404)
- `http://your-domain.com/reports/` → Should be DENIED (403/404)

## Common Issues & Fixes

### Issue 1: "404 Not Found" on menu items

**Problem**: Document root points to wrong directory

**Fix**: Point document root to `public/` directory (see above)

### Issue 2: "500 Internal Server Error"

**Problem**: .htaccess not allowed or mod_rewrite not enabled

**Fix**:
```bash
# Apache
sudo a2enmod rewrite
sudo systemctl restart apache2

# Check AllowOverride
# In apache2.conf or virtual host, ensure:
# AllowOverride All
```

### Issue 3: Pages load but CSS/JS missing

**Problem**: Relative paths broken

**Fix**: Check browser console (F12) for failed requests
- Verify paths in HTML source
- Check file permissions: `chmod 644 public/css/* public/js/*`

### Issue 4: "Access denied" errors

**Problem**: File permissions too restrictive

**Fix**:
```bash
# Set correct ownership
sudo chown -R www-data:www-data /var/www/html/risk-system

# Set correct permissions
find /var/www/html/risk-system -type d -exec chmod 755 {} \;
find /var/www/html/risk-system -type f -exec chmod 644 {} \;

# Make reports writable
chmod 775 /var/www/html/risk-system/reports
```

### Issue 5: "Database connection failed"

**Problem**: Wrong path in config.php or wrong credentials

**Fix**: Edit `includes/config.php` and verify:
```php
define('DB_HOST', 'localhost');  // Correct host
define('DB_USER', 'your_user');  // Correct user
define('DB_PASS', 'your_pass');  // Correct password
define('DB_NAME', 'scubatricky_risk');  // Correct database
```

## File Permissions Checklist

```bash
# Application root
chmod 755 risk-system/

# Directories
chmod 755 risk-system/public/
chmod 755 risk-system/includes/
chmod 755 risk-system/templates/
chmod 775 risk-system/reports/  # Needs write!

# PHP files
chmod 644 risk-system/public/*.php
chmod 644 risk-system/includes/*.php
chmod 644 risk-system/templates/*.php

# Python script
chmod 755 risk-system/public/api/generate_pdf.py

# Config (more restrictive)
chmod 600 risk-system/includes/config.php

# Ownership
chown -R www-data:www-data risk-system/
# OR on some systems:
chown -R apache:apache risk-system/
# OR on shared hosting:
chown -R username:username risk-system/
```

## Quick Deployment Checklist

- [ ] Extract files to server
- [ ] **Point document root to `public/` directory**
- [ ] Import database: `mysql < includes/schema.sql`
- [ ] Edit `includes/config.php` with database credentials
- [ ] Set file permissions (see above)
- [ ] Make `reports/` writable
- [ ] Enable mod_rewrite (Apache)
- [ ] Test: Visit site → Should see login page
- [ ] Test: Click menu items → Should work (no 404)
- [ ] Test: Login with admin / Admin123!
- [ ] Change admin password
- [ ] Delete `reset-admin.php`

## Production Checklist

Additional steps for production deployment:

- [ ] Enable HTTPS (Let's Encrypt recommended)
- [ ] Update `includes/config.php`:
  - [ ] Set session cookie_secure to 1
  - [ ] Update SITE_URL to https://
- [ ] Disable PHP error display
- [ ] Enable error logging
- [ ] Set up automated database backups
- [ ] Configure firewall (allow 80, 443 only)
- [ ] Remove any .git directories
- [ ] Remove reset-admin.php
- [ ] Test all functionality

## Getting Help

If you're still having 404 errors:

1. **Check Apache error log**:
   ```bash
   tail -f /var/log/apache2/error.log
   ```

2. **Check what Apache sees**:
   ```bash
   apache2ctl -S
   # Look for DocumentRoot line
   ```

3. **Verify file exists**:
   ```bash
   ls -la /var/www/html/risk-system/public/dashboard.php
   # Should exist and be readable
   ```

4. **Check .htaccess is loaded**:
   ```bash
   # In Apache config, ensure:
   AllowOverride All
   # NOT: AllowOverride None
   ```

---

**Key Point**: The `public` directory MUST be your web server's document root for the application to work correctly! All navigation links assume this structure.
